/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import shohaku.core.lang.Boxing;
import shohaku.core.lang.Eval;

/**
 * <code>KayValueIterator</code> インタフェースのユーティリティメソッドを提供します。
 */
public class KayValueIteratorUtils {

    /*
     * キーと値の反復子の抽象実装を提供します。
     */
    private static abstract class AbstractKayValueIterator implements KayValueIterator {

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /*
     * マップを基とするキーと値の反復子を提供します。
     */
    private static class MapKayValueIterator extends AbstractKayValueIterator {
        private final Iterator entrys;

        private final int size;

        private Map.Entry _entry = null;

        private MapKayValueIterator(Map m) {
            Set entrySet = m.entrySet();
            this.entrys = entrySet.iterator();
            this.size = entrySet.size();
        }

        public boolean hasNext() {
            return entrys.hasNext();
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return entry().getKey();
        }

        public Object getValue() {
            return entry().getValue();
        }

        public Object setValue(Object value) {
            return entry().setValue(value);
        }

        public int size() {
            return this.size;
        }

        private Map.Entry entry() {
            if (_entry == null) {
                throw new IllegalStateException();
            }
            return _entry;
        }

        private void _next() {
            _entry = (Map.Entry) entrys.next();
        }
    }

    /*
     * グループを基とするキーと値の反復子を提供します。
     */
    private static class GroupKayValueIterator extends AbstractKayValueIterator {
        private final Iterator entrys;

        private final int size;

        private Group.Entry _entry = null;

        private GroupKayValueIterator(Group g) {
            Set entrySet = g.entrySet();
            this.entrys = entrySet.iterator();
            this.size = entrySet.size();
        }

        public boolean hasNext() {
            return entrys.hasNext();
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return entry().getKey();
        }

        public Object getValue() {
            return entry().getValues();
        }

        public Object setValue(Object value) {
            throw new UnsupportedOperationException();
        }

        public int size() {
            return this.size;
        }

        private Group.Entry entry() {
            if (_entry == null) {
                throw new IllegalStateException();
            }
            return _entry;
        }

        private void _next() {
            _entry = (Group.Entry) entrys.next();
        }
    }

    /*
     * リストを基とするキーと値の反復子を提供します。
     */
    private static class ListKayValueIterator extends AbstractKayValueIterator {
        private final List list;

        private final int size;

        private int _index = -1;

        private ListKayValueIterator(List list) {
            this.list = list;
            this.size = list.size();
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return Boxing.box(index());
        }

        public Object getValue() {
            return list.get(index());
        }

        public Object setValue(Object value) {
            return list.set(index(), value);
        }

        public int size() {
            return this.size;
        }

        private int index() {
            if (_index == -1) {
                throw new IllegalStateException();
            }
            if (list.size() != size) {
                throw new ConcurrentModificationException();
            }
            return _index;
        }

        private void _next() {
            _index++;
        }
    }

    /*
     * 配列を基とするキーと値の反復子を提供します。
     */
    private static class ArrayKayValueIterator extends AbstractKayValueIterator {
        private final Object[] array;

        private final int size;

        private int _index = -1;

        private ArrayKayValueIterator(Object[] a) {
            this.array = a;
            this.size = a.length;
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return Boxing.box(index());
        }

        public Object getValue() {
            return array[index()];
        }

        public Object setValue(Object value) {
            return array[index()] = value;
        }

        public int size() {
            return this.size;
        }

        private int index() {
            if (_index == -1) {
                throw new IllegalStateException();
            }
            return _index;
        }

        private void _next() {
            _index++;
        }
    }

    /*
     * 二つのリストを基とするキーと値の反復子を提供します。
     */
    private static class IndexMappingListKayValueIterator extends AbstractKayValueIterator {
        private final List keys;

        private final List values;

        private final int size;

        private int _index = -1;

        private IndexMappingListKayValueIterator(List keys, List values) {
            if (keys == null || values == null) {
                throw new NullPointerException();
            }
            if (keys.size() == values.size()) {
                throw new IllegalArgumentException();
            }
            this.keys = keys;
            this.values = values;
            this.size = keys.size();
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return keys.get(index());
        }

        public Object getValue() {
            return values.get(index());
        }

        public Object setValue(Object value) {
            return values.set(index(), value);
        }

        public int size() {
            return this.size;
        }

        private int index() {
            if (_index == -1) {
                throw new IllegalStateException();
            }
            if (keys.size() != size || values.size() != size) {
                throw new ConcurrentModificationException();
            }
            return _index;
        }

        private void _next() {
            _index++;
        }
    }

    /*
     * 二つの配列を基とするキーと値の反復子を提供します。
     */
    private static class IndexMappingArrayKayValueIterator extends AbstractKayValueIterator {
        private final Object[] keys;

        private final Object[] values;

        private final int size;

        private int _index = -1;

        private IndexMappingArrayKayValueIterator(Object[] keys, Object[] values) {
            if (keys == null || values == null) {
                throw new NullPointerException();
            }
            if (keys.length == values.length) {
                throw new IllegalArgumentException();
            }
            this.keys = keys;
            this.values = values;
            this.size = keys.length;
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _next();
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return keys[index()];
        }

        public Object getValue() {
            return values[index()];
        }

        public Object setValue(Object value) {
            return values[index()] = value;
        }

        public int size() {
            return this.size;
        }

        private int index() {
            if (_index == -1) {
                throw new IllegalStateException();
            }
            return _index;
        }

        private void _next() {
            _index++;
        }
    }

    /*
     * リストを区画分割して認識するキーと値の反復子を提供します。
     */
    private static class SegmentListKayValueIterator extends AbstractKayValueIterator {
        private final int size;

        private final int segment;

        private final int koffset;

        private final int voffset;

        private final int start;

        private final List list;

        private int _index = -1;

        private SegmentListKayValueIterator(List list, int start, int segment, int koffset, int voffset) {
            if (list == null) {
                throw new NullPointerException();
            }
            int size = list.size();
            if (!Eval.isRange(segment, 0, size) || !Eval.isRange(koffset, 0, segment)
                    || !Eval.isRange(voffset, 0, segment) || !Eval.isRange(start, 0, (size - 1))) {
                throw new IllegalArgumentException("size:" + size + ", segment:" + segment + ", koffset:" + koffset
                        + ", voffset:" + voffset + ", start:" + start);
            }
            this.size = ((size - start) / segment);
            this.segment = segment;
            this.koffset = koffset;
            this.voffset = voffset;
            this.start = start;
            this.list = list;
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _index++;
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return list.get(keyIndex());
        }

        public Object getValue() {
            return list.get(valueIndex());
        }

        public Object setValue(Object value) {
            return list.set(valueIndex(), value);
        }

        public int size() {
            return this.size;
        }

        private int keyIndex() {
            check();
            return ((_index * segment + start) + koffset);
        }

        private int valueIndex() {
            check();
            return ((_index * segment + start) + voffset);
        }

        private void check() {
            if (0 > _index) {
                throw new IllegalStateException();
            }
            if (list.size() != size) {
                throw new ConcurrentModificationException();
            }
        }
    }

    /*
     * 配列を区画分割して認識するキーと値の反復子を提供します。
     */
    private static class SegmentArrayKayValueIterator extends AbstractKayValueIterator {
        private final int size;

        private final int segment;

        private final int koffset;

        private final int voffset;

        private final int start;

        private final Object[] array;

        private int _index = -1;

        private SegmentArrayKayValueIterator(Object[] a, int start, int segment, int koffset, int voffset) {
            if (a == null) {
                throw new NullPointerException();
            }
            int size = a.length;
            if (!Eval.isRange(segment, 0, size) || !Eval.isRange(koffset, 0, segment)
                    || !Eval.isRange(voffset, 0, segment) || !Eval.isRange(start, 0, (size - 1))) {
                throw new IllegalArgumentException("size:" + size + ", segment:" + segment + ", koffset:" + koffset
                        + ", voffset:" + voffset + ", start:" + start);
            }
            this.size = ((size - start) / segment);
            this.segment = segment;
            this.koffset = koffset;
            this.voffset = voffset;
            this.start = start;
            this.array = a;
        }

        public boolean hasNext() {
            return ((_index + 1) < size);
        }

        public Object next() {
            if (hasNext()) {
                _index++;
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            return array[keyIndex()];
        }

        public Object getValue() {
            return array[valueIndex()];
        }

        public Object setValue(Object value) {
            return array[valueIndex()] = value;
        }

        public int size() {
            return this.size;
        }

        private int keyIndex() {
            check();
            return ((_index * segment + start) + koffset);
        }

        private int valueIndex() {
            check();
            return ((_index * segment + start) + voffset);
        }

        private void check() {
            if (0 > _index) {
                throw new IllegalStateException();
            }
        }
    }

    /**
     * マップをキーと値の反復子に変換します。 <br>
     * キーの重複は発生しません。
     * 
     * @param map
     *            基となるマップ
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator asKayValueIterator(Map map) {
        return new MapKayValueIterator(map);
    }

    /**
     * グループをキーと値の反復子に変換します。 <br>
     * キーの重複は発生しません。
     * 
     * @param group
     *            基となるグループ
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator asKayValueIterator(Group group) {
        return new GroupKayValueIterator(group);
    }

    /**
     * リストをインデックスをキーとしてキーと値の反復子に変換します。 <br>
     * キーの重複は発生しません。
     * 
     * @param list
     *            基となるリスト
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator indexKayValueIterator(List list) {
        return new ListKayValueIterator(list);
    }

    /**
     * 配列をインデックスをキーとしてキーと値の反復子に変換します。 <br>
     * キーの重複は発生しません。
     * 
     * @param a
     *            基となる配列
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator indexKayValueIterator(Object[] a) {
        return new ArrayKayValueIterator(a);
    }

    /**
     * 二つのリストをキーと値として同一のインデックスで関連付けてキーと値の反復子に変換します。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param keys
     *            キーのリスト
     * @param values
     *            値のリスト
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator indexMappingKayValueIterator(List keys, List values) {
        return new IndexMappingListKayValueIterator(keys, values);
    }

    /**
     * 二つのリストをキーと値として同一のインデックスで関連付けてキーと値の反復子に変換します。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param keys
     *            キーのリスト
     * @param values
     *            値のリスト
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator indexMappingKayValueIterator(Object[] keys, Object[] values) {
        return new IndexMappingArrayKayValueIterator(keys, values);
    }

    /**
     * リストを２要素ずつの区画で分割してキーと値の反復子に変換します。 <br>
     * リスト順に <code>key, value, key, value...</code> で参照します。 <br>
     * <code>segmentKayValueIterator(list, 0, 0, 1, 2);</code> と同意です。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param list
     *            基となるリスト
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(List list) {
        return segmentKayValueIterator(list, 0, 2, 0, 1);
    }

    /**
     * 配列を２要素ずつの区画で分割してキーと値の反復子に変換します。 <br>
     * 配列順に <code>key, value, key, value...</code> で参照します。 <br>
     * <code>segmentKayValueIterator(a, 0, 0, 1, 2);</code> と同意です。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param a
     *            基となる配列
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(Object[] a) {
        return segmentKayValueIterator(a, 0, 2, 0, 1);
    }

    /**
     * リストを指定された区画で分割してキーと値の反復子に変換します。 <br>
     * <code>segmentKayValueIterator(list, 0, koffset, valueOffset, segment);</code> と同意です。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param list
     *            基となるリスト
     * @param segment
     *            区画数
     * @param koffset
     *            キーの相対Index
     * @param valueOffset
     *            値の相対Index
     * 
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(List list, int segment, int koffset, int valueOffset) {
        return segmentKayValueIterator(list, 0, segment, koffset, valueOffset);
    }

    /**
     * 配列を指定された区画で分割してキーと値の反復子に変換します。 <br>
     * <code>segmentKayValueIterator(a, 0, koffset, valueOffset, segment);</code> と同意です。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param a
     *            基となる配列
     * @param segment
     *            区画数
     * @param koffset
     *            キーの相対Index
     * @param valueOffset
     *            値の相対Index
     * 
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(Object[] a, int segment, int koffset, int valueOffset) {
        return segmentKayValueIterator(a, 0, segment, koffset, valueOffset);
    }

    /**
     * リストを指定された区画で分割してキーと値の反復子に変換します。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param list
     *            基となるリスト
     * @param start
     *            走査の開始位置
     * @param segment
     *            区画数
     * @param koffset
     *            キーの相対Index
     * @param voffset
     *            値の相対Index
     * 
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(List list, int start, int segment, int koffset, int voffset) {
        return new SegmentListKayValueIterator(list, start, segment, koffset, voffset);
    }

    /**
     * 配列を指定された区画で分割してキーと値の反復子に変換します。 <br>
     * キーの重複が発生する可能性があります。
     * 
     * @param a
     *            基となる配列
     * @param start
     *            走査の開始位置
     * @param segment
     *            区画数
     * @param koffset
     *            キーの相対Index
     * @param voffset
     *            値の相対Index
     * 
     * @return 変換された <code>KayValueIterator</code>
     */
    public static KayValueIterator segmentKayValueIterator(Object[] a, int start, int segment, int koffset, int voffset) {
        return new SegmentArrayKayValueIterator(a, start, segment, koffset, voffset);
    }

}
