/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections;

import java.util.Arrays;
import java.util.Collection;

import shohaku.core.collections.wrapper.SingletonGroup;
import shohaku.core.collections.wrapper.SynchronizedGroup;
import shohaku.core.collections.wrapper.UnmodifiableGroup;
import shohaku.core.collections.wrapper.WrappedGroup;

/**
 * <code>Group</code> インタフェースのユーティリティメソッドを提供します。
 */
public class GroupUtils {

    /**
     * <p>
     * 指定されたグループの変更不可能なビューを返します。
     * </p>
     * <p>
     * このメソッドを使うと、モジュールはユーザに内部グループへの「読み取り専用」アクセス権を提供できます。 返されたグループでのクエリーオペレーションは、指定されたグループを直接読み取ります。
     * 直接か、そのコレクションビューを使うかに関係なく、 返されたグループを変更しようとすると <code>UnsupportedOperationException</code> がスローされます。
     * </p>
     * <p>
     * 返されたグループは、指定されたグループが直列化可能の場合は直列化可能です。
     * </p>
     * 
     * @param g
     *            変更不可能なビューが返されるグループ
     * @return 指定されたグループの変更不可能なビュー
     */
    public static Group unmodifiableGroup(Group g) {
        return new UnmodifiableGroup(g);
    }

    /**
     * 指定されたコレクションに指定されたキーだけをマッピングする不変のグループを返します。返されるグループは直列化可能です。
     * 
     * @param key
     *            返されるグループに格納されている唯一のキー
     * @param values
     *            返されるグループによって <code>key</code> がマッピングされるコレクション
     * @return 指定したキーとコレクションのマッピングだけが格納される不変のグループ
     */
    public static Group singletonGroup(Object key, Object[] values) {
        return new SingletonGroup(key, Arrays.asList(values));
    }

    /**
     * 指定されたコレクションに指定されたキーだけをマッピングする不変のグループを返します。返されるグループは直列化可能です。
     * 
     * @param key
     *            返されるグループに格納されている唯一のキー
     * @param values
     *            返されるグループによって <code>key</code> がマッピングされるコレクション
     * @return 指定したキーとコレクションのマッピングだけが格納される不変のグループ
     */
    public static Group singletonGroup(Object key, Collection values) {
        return new SingletonGroup(key, values);
    }

    /**
     * 指定されたグループを基にする同期 (スレッドセーフな) グループを返します。 <br>
     * 確実に直列アクセスを実現するには、基になるグループへのアクセスはすべて、 <br>
     * 返されたグループを介して行う必要があります。 <br>
     * <br>
     * 返されたグループのコレクションビューでの繰り返し処理を行う場合、ユーザは、次に示すように手動で同期をとる必要があります。 <br>
     * 
     * <pre>
     *       Group g = Collections.synchronizedMap(new HashListGroup());
     *           ...
     *       Set s = g.keySet();  // Needn't be in synchronized block
     *           ...
     *       synchronized(g) {  // Synchronizing on m, not s!
     *           Iterator i = s.iterator(); // Must be in synchronized block
     *           while (i.hasNext())
     *              foo(i.next());
     *       }
     * </pre>
     * 
     * これを行わない場合、動作は保証されません。 <br>
     * <br>
     * 返されたグループは、指定されたグループが直列化可能の場合は直列化可能です。
     * 
     * @param group
     *            同期グループに「ラップ」されるグループ
     * @return 指定されたグループの同期ビュー
     */
    public static Group synchronizedGroup(Group group) {
        return new SynchronizedGroup(group);
    }

    /**
     * グループ型にラッピングしたグループを返却します。実装型に固有の機能へのアクセスを制限します。
     * 
     * @param group
     *            ラップするグループ
     * @return 実装をラッピングしたグループ
     */
    public static Group wrappedGroup(Group group) {
        return new WrappedGroup(group);
    }

}
