/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

/**
 * <p>
 * キーを識別子として複数の値をグループ化するオブジェクトです。 <br>
 * データ構造的には重複可能なマップ（マルチマップ）と類似です。
 * </p>
 * <p>
 * グループには、同一のキーを複数登録することはできません。 <br>
 * キーは必ず一意であり、キー対応する値を必ず０個以上保持しています。
 * </p>
 * <p>
 * グループ化された値の集まりに対して値の重複が許されるか、また要素に順序を持つかは各実装によって異なります。
 * </p>
 * <p>
 * このインターフェースは <code>Java Collections Framework </code> の独自拡張の位置付けです。
 * </p>
 */
public interface Group {

    /**
     * グループのエントリ (キーとコレクションのペア) です。 <br>
     * Group.entrySet メソッドは、このクラスに属する要素を持つグループのコレクションビューを返します。 <br>
     * Group.iterator メソッドは、このクラスに属する要素を持つグループの反復子を返します。 <br>
     * これらの Group.Entry オブジェクトは、繰り返し処理中にだけ有効です。 <br>
     * すなわち、グループエントリが反復子によって返されたあとに、基になっているグループが変更された場合、グループエントリの動作は定義されていません。
     */
    interface Entry {

        /**
         * このグループエントリのキーを返却します。
         * 
         * @return このエントリのキー
         */
        Object getKey();

        /**
         * <p>
         * このグループエントリされるコレクションを返却します。
         * </p>
         * <p>
         * 戻り値に <code>Collection</code> の継承インターフェースの型が戻す実装も有り得ます。 <br>
         * つまり左記の処理が有効である実装が有り得ます、 <code>List l = (List) entry.getValues();</code>。
         * </p>
         * 
         * @return このエントリのマッピングされるコレクション
         */
        Collection getValues();

        /**
         * 同値として認識出来るか検証します。 <br>
         * Group.Entry 型で同一のキーとコレクションを持つ場合に同一と認識されます。
         * <p>
         * グループエントリ e1 と e2 の同値性比較は、次のように定義されます。 <br>
         * <code>
         *      (e1.getKey().equals(e2) && e1.getValue().equals(e2))
         * </code>
         * </p>
         * 
         * @param o
         *            同値性を検証するオブジェクト
         * @return 同値として認識出来る場合True
         */
        boolean equals(Object o);

        /**
         * このグループエントリのハッシュコード値を返します。
         * <p>
         * グループエントリ e のハッシュコードは、次のように定義されます。 <br>
         * <code>
         *      (e.getKey().hashCode() + e.getValue().hashCode())
         * </code> これにより、Object.hashCode の一般規約によって要求される、 任意の 2
         * つのエントリ e1 と e2 で、e1.equals(e2) であれば e1.hashCode()==e2.hashCode() となることが保証されます。
         * </p>
         * 
         * @return グループエントリのハッシュコード値
         */
        int hashCode();

        /**
         * グループエントリの文字列表現を返却します。 <br>
         * <code>
         *      e.getKey() + e.getValue() の書式で出力します。
         * </code>
         * 
         * @return グループエントリの文字列表現
         */
        String toString();
    }

    /**
     * グループのマッピングをすべて削除します (任意のオペレーション)。
     */
    void clear();

    /**
     * グループがキーとコレクションのマッピングを保持しない場合に true を返します。
     * 
     * @return グループがキーとコレクションのマッピングを保持しない場合は true
     */
    boolean isEmpty();

    /**
     * 指定されたキーに対するコレクションを追加します (任意のオペレーション)。 <br>
     * グループに既にこのキーに対するマッピングがある場合何も行われません。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定されるコレクションが関連付けられるキー
     * @return 既にこのキーに対するマッピングがある場合 true
     */
    boolean add(Object key);

    /**
     * 指定された値を指定されたキーに対応するコレクションに追加します (任意のオペレーション)。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定される値が関連付けられるキー
     * @param value
     *            指定されるキーに関連付けられる値
     * @return 既にこのキーに対するマッピングがある場合 true
     */
    boolean add(Object key, Object value);

    /**
     * 指定されたコレクションの全要素をキーに対応するコレクションに追加します (任意のオペレーション)。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定されるコレクションの全要素が関連付けられるキー
     * @param c
     *            指定されるキーに関連付けられるコレクション
     * @return 既にこのキーに対するマッピングがある場合 true
     */
    boolean addAll(Object key, Collection c);

    /**
     * 指定されたグループのすべてのキーと対応する全値ををこのグループに追加します (任意のオペレーション)。
     * <p>
     * 指定されたグループのキー k から値 v までの各マッピングに関して、 この呼び出しの効果は、putAll(k, v) を呼び出した場合と同じです。
     * 指定されたグループがこのオペレーションの処理中に変更された場合、そのオペレーションの動作は指定外となります。
     * </p>
     * 
     * @param g
     *            グループに追加されるグループ
     */
    void addAll(Group g);

    /**
     * グループに含まれるグループエントリの反復子を返します。 <br>
     * 反復子の要素は Group.Entry 型に為ります。 エントリが存在しない場合空の反復子が返ります。
     * 
     * @return グループエントリの反復子
     */
    Iterator iterator();

    /**
     * 指定されたキーにマッピングされたコレクションの要素の反復子を返します。 <br>
     * 指定されたキーにマッピングされたコレクションが存在しない場合空の反復子が返ります。
     * 
     * @param key
     *            関連づけられている識別キー
     * @return キーにマッピングされたコレクションの反復子
     */
    Iterator iterator(Object key);

    /**
     * <p>
     * 指定されたキーにマッピングされたコレクションを返します。 <br>
     * グループがこのキーのマッピングを保持していない場合は <code>null</code> を返します。
     * </p>
     * <p>
     * 戻り値に <code>Collection</code> の継承インターフェースの型が戻す実装も有り得ます。 <br>
     * つまり左記の処理が有効である実装が有り得ます、 <code>List l = (List) group.get(key);</code>。
     * </p>
     * 
     * @param key
     *            関連づけられている識別キー
     * @return 指定のキーをマッピングするコレクション
     */
    Collection get(Object key);

    /**
     * このキーにマッピングがある場合に、そのマッピングをグループから削除します (任意のオペレーション)。
     * <p>
     * つまり、 <code>(key==null ? k==null : key.equals(k))</code> という条件で、キー k からコレクション c
     * までマッピングがグループに含まれる場合、このマッピングは削除されます。グループはこのようなマッピングを 1 つだけ含みます。
     * </p>
     * <p>
     * キーに以前マッピングされていたコレクションを返します。 <br>
     * このキーのマッピングがグループにない場合は、null を返します 1 度呼び出しが返れば、グループは指定されたキーのマッピングを含みません。
     * </p>
     * <p>
     * 戻り値に <code>Collection</code> の継承インターフェースの型が戻す実装も有り得ます。 <br>
     * つまり左記の処理が有効である実装が有り得ます、 <code>List l = (List) group.remove(key);</code>。
     * </p>
     * 
     * @param key
     *            関連づけられている識別キー
     * @return 指定されたキーとマッピングされていた以前のコレクション。キーのマッピングがなかった場合は null
     */
    Collection remove(Object key);

    /**
     * このキーにマッピングがある場合に、そのマッピングの要素から指定の値を削除します (任意のオペレーション)。
     * <p>
     * つまり、 (k.equals(key)) という条件で、キー key のマッピングがグループに含まれる場合、 コレクション c に対して要素 value が (c.remove(value)) の条件で、これを削除します。
     * </p>
     * <p>
     * 指定された要素がマッピングされたコレクション内に格納されていた場合には true を返します。 <br>
     * すなわち、この呼び出しの結果、グループが変更された場合に true を返します。
     * </p>
     * 
     * @param key
     *            関連づけられている識別キー
     * @param value
     * @return 指定されたキーと関連付けられていた以前のコレクション。キーのマッピングがなかった場合は null
     */
    boolean remove(Object key, Object value);

    /**
     * 指定されたキーのマッピングがグループに含まれている場合に true を返します。
     * <p>
     * つまり、 (k.equals(key)) という条件で、キー key のマッピングがグループに含まれる場合に true を返します。 <br>
     * グループはこのようなマッピングを 1 つだけ含むことができます。
     * </p>
     * 
     * @param key
     *            マッピングされているか判定されるキー
     * @return グループが指定のキーのマッピングを保持する場合は true
     */
    boolean containsKey(Object key);

    /**
     * グループが、キーのマッピングしているコレクションに指定された値を 1 つ以上含む場合に true を返します。
     * <p>
     * つまり、 (k.equals(key)) という条件で、キー key のマッピングがグループに含まれる場合、コレクション c に対して要素 value が (c.contains(value)) の条件で true の場合に
     * true を返します。
     * </p>
     * 
     * @param key
     *            マッピングされているか判定されるキー
     * @param value
     *            キーをマッピングしているコレクションの内に存在するか判定される値
     * @return グループが指定のキーにマッピングされるコレクションの内に値が存在する場合は true
     */
    boolean containsValue(Object key, Object value);

    /**
     * グループに含まれているキーのセットビューを返します。
     * <p>
     * セットはグループと連動しているので、グループに対する変更はセットに反映され、 また、セットに対する変更はグループに反映されます。 <br>
     * セットに対する反復の処理中にグループが変更された場合は、反復の結果は保証されません。 <br>
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って グループから対応するマッピングを削除する要素削除処理をサポートします。
     * <br>
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * </p>
     * 
     * @return グループに含まれているキーのセットビュー
     */
    Set keySet();

    /**
     * グループに含まれているグループエントリのセットビューを返します。
     * <p>
     * セットはグループと連動しているので、グループに対する変更はセットに反映され、 また、セットに対する変更はグループに反映されます。 <br>
     * セットに対する反復の処理中にグループが変更された場合は、反復の結果は保証されません。 <br>
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って グループから対応するマッピングを削除する要素削除処理をサポートします。
     * <br>
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * </p>
     * 
     * @return グループに含まれているグループエントリのセットビュー
     */
    Set entrySet();

    /**
     * グループ内のキーと値の集合の数を返します。 <br>
     * グループに Integer.MAX_VALUE より多くの要素がある場合は、Integer.MAX_VALUE を返します。
     * 
     * @return マッピングの数
     */
    int size();

    /**
     * グループ内のキーとマッピングする値のの要素数を返します。 <br>
     * この値に Integer.MAX_VALUE より多くの要素がある場合は、Integer.MAX_VALUE を返します。
     * 
     * @param key
     *            キー
     * @return キーとマッピングする値のの要素数
     */
    int size(Object key);

    /**
     * グループのハッシュコード値を返します。
     * <p>
     * グループのハッシュコードは、グループが保有する全てのグループエントリの hashCode の合計になるように定義されます。 <br>
     * これにより、Object.hashCode の一般規約によって要求されるように、任意の 2 つのグループ t1 と t2 で t1.equals(t2) であれば、t1.hashCode()==t2.hashCode()
     * となることが保証されます。
     * </p>
     * 
     * @return グループのハッシュコード値
     */
    int hashCode();

    /**
     * 定されたオブジェクトがこのグループと等しいかどうかを比較します。
     * <p>
     * 指定されたオブジェクトもグループであり、2 つの Group が同じグループエントリを表している場合は true を返します。 <br>
     * つまり、t1.entrySet().equals(t2.entrySet()) である場合、2 つのグループ t1 と t2 は同じグループエントリを表します。 <br>
     * これにより、Group インタフェースの実装が異なる場合でも、equals メソッドが正しく動作することが保証されます。
     * </p>
     * 
     * @param o
     *            グループと等しいかどうかを比較するオブジェクト
     * @return 指定されたオブジェクトがグループと等しい場合は true
     */
    boolean equals(Object o);

}