/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections;

/**
 * <code>Entry</code> インタフェースのユーティリティメソッドを提供します。
 */
public class EntryUtils {

    /**
     * 単一のオブジェクトを格納するエントリ。 <br>
     * このエントリの hashCode() は System.identityHashCode(Object o) と同値なる。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が参照同一性の比較 <code>==</code> で同値であることを条件とします。
     */
    static class IdentityEntry implements Entry {
        private final Object src;

        IdentityEntry(Object src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            return System.identityHashCode(src);
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            return (src == v);
        }
    }

    /**
     * 参照一致で同値性検証を行うオブジェクト配列を格納するエントリ。 <br>
     * このエントリの hashCode() は全配列要素の <code>System.identityHashCode(Object)</code> 総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値がオブジェクト配列であること又全要素の参照が一致することを条件とします。
     */
    static class IdentityObjectArrayEntry implements Entry {
        private final Object[] src;

        IdentityObjectArrayEntry(Object[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += System.identityHashCode(src[i]);
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (src[i] != a[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * equals(Object) で同値性検証を行うオブジェクト配列を格納するエントリ。 <br>
     * このエントリの hashCode() は全配列要素の hashCode() 総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値がオブジェクト配列であること又全要素の equals(Object) が true であることを条件とします。
     */
    static class ObjectArrayEntry implements Entry {
        private final Object[] src;

        ObjectArrayEntry(Object[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += (src[i] == null) ? 0 : src[i].hashCode();
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (src[i] == null || a[i] == null) {
                    if (src[i] != a[i]) {
                        return false;
                    }
                } else if (!src[i].equals(a[i])) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * byte 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Byte.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が byte 配列であること又全要素が同値であることを条件とします。
     */
    static class ByteArrayEntry implements Entry {
        private final byte[] src;

        ByteArrayEntry(byte[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += src[i];
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof byte[])) {
                return false;
            }
            byte[] a = (byte[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * short 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Short.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が short 配列であること又全要素が同値であることを条件とします。
     */
    static class ShortArrayEntry implements Entry {
        private final short[] src;

        ShortArrayEntry(short[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += src[i];
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof short[])) {
                return false;
            }
            short[] a = (short[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * int 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Integer.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が int 配列であること又全要素が同値であることを条件とします。
     */
    static class IntArrayEntry implements Entry {
        private final int[] src;

        IntArrayEntry(int[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += src[i];
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof int[])) {
                return false;
            }
            int[] a = (int[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * long 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Long.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が long 配列であること又全要素が同値であることを条件とします。
     */
    static class LongArrayEntry implements Entry {
        private final long[] src;

        LongArrayEntry(long[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += (int) (src[i] ^ (src[i] >>> 32));
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof long[])) {
                return false;
            }
            long[] a = (long[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * float 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Float.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が float 配列であること又全要素が同値であることを条件とします。
     */
    static class FloatArrayEntry implements Entry {
        private final float[] src;

        FloatArrayEntry(float[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += Float.floatToIntBits(src[i]);
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof float[])) {
                return false;
            }
            float[] a = (float[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * double 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Double.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が double 配列であること又全要素が同値であることを条件とします。
     */
    static class DoubleArrayEntry implements Entry {
        private final double[] src;

        DoubleArrayEntry(double[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                long bits = Double.doubleToLongBits(src[i]);
                hash += (int) (bits ^ (bits >>> 32));
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof double[])) {
                return false;
            }
            double[] a = (double[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * boolean 配列を格納するエントリ。 <br>
     * このエントリの hashCode() は Boolean.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が boolean 配列であること又全要素が同値であることを条件とします。
     */
    static class BooleanArrayEntry implements Entry {
        private final boolean[] src;

        BooleanArrayEntry(boolean[] src) {
            if (src == null) {
                throw new NullPointerException();
            }
            this.src = src;
        }

        /**
         * このエントリが保有するデータを返却します。
         * 
         * @return このエントリが保有するデータ
         */
        public Object getValue() {
            return src;
        }

        /**
         * ハッシュコード値を返します。
         * 
         * @return ハッシュコード値
         */
        public int hashCode() {
            int hash = 0;
            for (int i = 0; i < src.length; i++) {
                hash += (src[i] ? 1231 : 1237);
            }
            return hash;
        }

        /**
         * 同値性を比較します。
         * 
         * @param o
         *            比較するオブジェクト
         * @return 等しい場合は true
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            Object v = e.getValue();
            if (src == v) {
                return true;
            }
            if (v == null || !(v instanceof boolean[])) {
                return false;
            }
            boolean[] a = (boolean[]) v;
            if (src.length != a.length) {
                return false;
            }
            for (int i = 0; i < src.length; i++) {
                if (a[i] != src[i]) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Byte.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が byte 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(byte[] a) {
        return new ByteArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Short.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が short 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(short[] a) {
        return new ShortArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Integer.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が int 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(int[] a) {
        return new IntArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Long.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が long 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(long[] a) {
        return new LongArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Float.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が float 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(float[] a) {
        return new FloatArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Double.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が double 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(double[] a) {
        return new DoubleArrayEntry(a);
    }

    /**
     * 配列の全要素を比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は Boolean.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が boolean 配列であること又全要素が同値であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(boolean[] a) {
        return new BooleanArrayEntry(a);
    }

    /**
     * 配列の全要素を同値性で比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は全配列要素の hashCode() 総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値がオブジェクト配列であること又全要素の equals(Object) が true であることを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry arrayEntry(Object[] a) {
        return new ObjectArrayEntry(a);
    }

    /**
     * 配列の全要素を参照同一性で比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は全配列要素の <code>System.identityHashCode(Object)</code> 総和となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値がオブジェクト配列であること又全要素の参照が一致することを条件とします。
     * 
     * @param a
     *            登録する配列
     * @return 配列のエントリ
     */
    public static Entry identityArrayEntry(Object[] a) {
        return new IdentityObjectArrayEntry(a);
    }

    /**
     * 単一のオブジェクトを参照同一性で比較するエントリを生成して返します。 <br>
     * このエントリの hashCode() は System.identityHashCode(Object o) と同値となります。 <br>
     * このエントリの equals(Object) は引数が Entry の実装クラスであり、値が参照同一性の比較 <code>==</code> で同値であることを条件とします。
     * 
     * @param a
     *            登録するオブジェクト
     * @return 単一のオブジェクトのエントリ
     */
    public static Entry identityEntry(Object a) {
        return new IdentityEntry(a);
    }

}
