/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import shohaku.core.util.UnmodifiableStateException;

/**
 * メソッドのパラメータ情報のリストを格納する機能を提供します。
 */
public class ParametersInfo extends FeatureInfo {

    /* パラメータの型。 */
    private Class[] parameterTypes;

    /* パラメータの値。 */
    private Object[] parameterValues;

    /* パラメータ情報。 */
    private ParameterInfo[] parameteres;

    /**
     * デフォルトコンストラクタ。
     */
    public ParametersInfo() {
        //no op
    }

    /**
     * パラメータ情報の配列を指定して初期化します。
     * 
     * @param params
     *            パラメータの情報
     */
    public ParametersInfo(ParameterInfo[] params) {
        setParameteres(params);
    }

    /**
     * パラメータの型と値を指定して初期化します。
     * 
     * @param type
     *            パラメータの型
     * @param value
     *            パラメータの値
     */
    public ParametersInfo(Class type, Object value) {
        setParameteres(new Class[] { type }, new Object[] { value });
    }

    /**
     * パラメータの型と値を指定して初期化します。
     * 
     * @param types
     *            パラメータの型
     * @param values
     *            パラメータの値
     */
    public ParametersInfo(Class[] types, Object[] values) {
        setParameteres(types, values);
    }

    /*
     * Property
     */

    /**
     * パラメータの値を返却します。
     * 
     * @return パラメータの値
     */
    public Object[] getParameterValues() {
        return parameterValues;
    }

    /**
     * パラメータの型を返却します。
     * 
     * @return パラメータの型
     */
    public Class[] getParameterTypes() {
        return parameterTypes;
    }

    /**
     * パラメータ情報の配列を返却します。
     * 
     * @return パラメータ情報の配列
     */
    public ParameterInfo[] getParameteres() {
        return parameteres;
    }

    /**
     * パラメータ情報の配列を格納します。
     * 
     * @param params
     *            パラメータ情報の配列
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setParameteres(ParameterInfo[] params) {
        checkUnmodifiable();

        if (params == null) {
            throw new NullPointerException("argument is null.");
        }
        Class[] types = new Class[params.length];
        Object[] values = new Object[params.length];
        for (int i = 0; i < params.length; i++) {
            if (params[i].getType() == null) {
                throw new NullPointerException("argument params[i].types is null.");
            }
            types[i] = params[i].getType();
            values[i] = params[i].getValue();
        }
        this.parameterTypes = types;
        this.parameterValues = values;
        this.parameteres = params;
    }

    /**
     * パラメータの型と値を格納します。
     * 
     * @param types
     *            パラメータの型
     * @param values
     *            パラメータの値
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setParameteres(Class[] types, Object[] values) {
        checkUnmodifiable();

        if (types == null || values == null) {
            throw new NullPointerException("argument is null.");
        }
        if (types.length != values.length) {
            throw new IllegalArgumentException("types.length != values.length.");
        }

        ParameterInfo[] params = new ParameterInfo[types.length];
        for (int i = 0; i < params.length; i++) {
            if (types[i] == null) {
                throw new NullPointerException("argument params[i].types is null.");
            }
            params[i] = new ParameterInfo(types[i], values[i]);
        }
        this.parameterTypes = types;
        this.parameterValues = values;
        this.parameteres = params;
    }
}
