/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * オブジェクトに対する生成や制御処理の呼出に失敗した場合に発生します。 <br>
 * <br>
 * <code>SecurityException</code> や <code>NoSuchMethodException</code> 等のリフレクション処理時に発生する例外の一般化された例外として定義されている。 <br>
 * 発生時の状況に応じてリフレクション処理時に発生する例外を起因例外として保有します。
 */
public class InvocationBeansException extends Exception {

    /**
     * デフォルトコンストラクタ。
     */
    public InvocationBeansException() {
        super();
    }

    /**
     * エラーメッセージを設定して初期化します。
     * 
     * @param message
     *            エラーメッセージ
     */
    public InvocationBeansException(String message) {
        super(message);
    }

    /**
     * 起因例外を設定して初期化します。
     * 
     * @param cause
     *            起因例外
     */
    public InvocationBeansException(Throwable cause) {
        super(cause);
    }

    /**
     * エラーメッセージと起因例外を設定して初期化します。
     * 
     * @param message
     *            エラーメッセージ
     * @param cause
     *            起因例外
     */
    public InvocationBeansException(String message, Throwable cause) {
        super(message, cause);
    }

    /**
     * スタックトレースを出力します。 <br>
     * 起因例外のスタックトレースも出力します。
     */
    public void printStackTrace() {
        super.printStackTrace();
        Throwable cause = getCause();
        if (cause != null) {
            cause.printStackTrace();
        }
    }

    /**
     * PrintWriterにスタックトレースを出力します。 <br>
     * 起因例外のスタックトレースも出力します。
     * 
     * @param w
     *            java.io.PrintWriter
     */
    public void printStackTrace(PrintWriter w) {
        super.printStackTrace(w);
        Throwable cause = getCause();
        if (cause != null) {
            cause.printStackTrace(w);
        }
    }

    /**
     * PrintStreamにスタックトレースを出力します。 <br>
     * 起因例外のスタックトレースも出力します。
     * 
     * @param s
     *            java.io.PrintStream
     */
    public void printStackTrace(PrintStream s) {
        super.printStackTrace(s);
        Throwable cause = getCause();
        if (cause != null) {
            cause.printStackTrace(s);
        }
    }

}
