/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import java.lang.reflect.Field;

import shohaku.core.util.UnmodifiableStateException;

/**
 * オブジェクトのフィールド情報を格納する機能を提供します。
 */
public class FieldInfo extends ClassFeatureInfo {

    /* フィールドの名前。 */
    private String name;

    /**
     * デフォルトコンストラクタ。
     */
    public FieldInfo() {
        //no op
    }

    /**
     * 引数のプロパティを格納して初期化します。
     * 
     * @param name
     *            フィールドの名前
     */
    public FieldInfo(String name) {
        setName(name);
    }

    /**
     * 引数のプロパティを格納して初期化します。
     * 
     * @param name
     *            フィールドの名前
     * @param objectType
     *            オブジェクト型
     */
    public FieldInfo(String name, Class objectType) {
        setName(name);
        setObjectType(objectType);
    }

    /**
     * 登録されているクラスからフィールドの値を返します。
     * 
     * @return フィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object get() throws InvocationBeansException {
        return get(getObjectType(), null);
    }

    /**
     * 指定されたクラスからフィールドの値を返します。
     * 
     * @param c
     *            クラス
     * @return フィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object get(Class c) throws InvocationBeansException {
        return get(c, null);
    }

    /**
     * 指定されたインスタンスからフィールドの値を返します。
     * 
     * @param obj
     *            インスタンス
     * @return フィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object get(Object obj) throws InvocationBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return get(obj.getClass(), obj);
    }

    /**
     * 登録されているクラスのフィールドの値を変更し以前の値を返します。
     * 
     * @param newValue
     *            格納するフィールドの値
     * @return 格納前のフィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object set(Object newValue) throws InvocationBeansException {
        return set(getObjectType(), null, newValue);
    }

    /**
     * 指定されたクラスのフィールドの値を変更し以前の値を返します。
     * 
     * @param c
     *            クラス
     * @param newValue
     *            格納するフィールドの値
     * @return 格納前のフィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object set(Class c, Object newValue) throws InvocationBeansException {
        return set(c, null, newValue);
    }

    /**
     * 指定されたインスタンスのフィールドの値を変更し以前の値を返します。
     * 
     * @param obj
     *            インスタンス
     * @param newValue
     *            格納するフィールドの値
     * @return 格納前のフィールドの値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object set(Object obj, Object newValue) throws InvocationBeansException {
        if (obj == null) {
            throw new NullPointerException("argument obj is null.");
        }
        return set(obj.getClass(), obj, newValue);
    }

    /*
     * private
     */

    /* 指定されたクラスとそのインスタンスからフィールドの値を返します。 */
    private Object get(Class c, Object obj) throws InvocationBeansException {
        if (c == null) {
            throw new NullPointerException("argument class is null.");
        }
        Field field = getField(c);
        return BeanUtilities.getFieldValue(c, obj, field);
    }

    /* 指定されたクラスとそのインスタンスのフィールドの値を変更し以前の値を返します。 */
    private Object set(Class c, Object obj, Object newValue) throws InvocationBeansException {
        if (c == null) {
            throw new NullPointerException("argument class is null.");
        }
        Field field = getField(c);
        return BeanUtilities.setFieldValue(c, obj, field, newValue);

    }

    /* フィールドの取得。 */
    private Field getField(Class c) throws InvocationBeansException {
        Field field = BeanUtilities.getAccessibleField(c, getName());
        if (field == null) {
            throw new InvocationBeansException("no such field. class:" + c + ", field:" + getName());
        }
        return field;
    }

    /*
     * Property
     */

    /**
     * フィールドの名前を返却します。
     * 
     * @return フィールドの名前
     */
    public String getName() {
        return name;
    }

    /**
     * フィールドの名前を格納します。 <br>
     * 
     * @param name
     *            フィールドの名前
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setName(String name) {
        checkUnmodifiable();

        if (name == null) {
            throw new NullPointerException("argument is null.");
        }
        this.name = name;
    }

}
