/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import shohaku.core.util.UnmodifiableChangedObject;
import shohaku.core.util.UnmodifiableStateException;

/**
 * オブジェクトを動的に制御するための機能の一般化された定義を提供します。 <br>
 * <br>
 * このクラスは <code>UnmodifiableChangedObject</code> を実装するため変更不可の状態へ遷移した場合、プロパティ等への外部からの変更が出来なくなる必要がある。
 */
public class FeatureInfo implements UnmodifiableChangedObject {

    /*
     * UnmodifiableChangedObject
     */

    /* 参照基から変更不可能な状態。 */
    private boolean unmodifiable;

    /**
     * 参照基から変更不可能な状態を持つ場合 <code>true</code> を返却します．
     * 
     * @return 参照基から変更不可能な状態を持つ場合 <code>true</code>
     * @see shohaku.core.util.UnmodifiableChangedObject#isUnmodifiable()
     */
    public boolean isUnmodifiable() {
        return unmodifiable;
    }

    /**
     * <code>true</code> を指定すると参照基から変更不可能な状態に移行します． <br>
     * 変更不可の状態で <code>true</code> を指定しても何も発生しません。 <br>
     * 変更不可の状態で <code>false</code> が指定された場合 <code>UnmodifiableStateException</code> が発生します．
     * 
     * @param unmodifiable
     *            参照基から変更不可能な状態へ遷移する場合 <code>true</code>
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     * @see shohaku.core.util.UnmodifiableChangedObject#setUnmodifiable(boolean)
     */
    public void setUnmodifiable(boolean unmodifiable) {
        if (isUnmodifiable() == unmodifiable) {
            return;
        }
        checkUnmodifiable();

        this.unmodifiable = unmodifiable;
    }

    /**
     * 変更不可の状態で呼び出されたか検証します。 <br>
     * 変更不可の状態で呼び出された場合 <code>UnmodifiableStateException</code> が発生します．
     * 
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    protected void checkUnmodifiable() throws UnmodifiableStateException {
        if (isUnmodifiable()) {
            throw new UnmodifiableStateException("object is unmodifiable state.");
        }
    }

}
