/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import shohaku.core.util.UnmodifiableStateException;

/**
 * クラス情報を格納し生成する機能を提供します。
 */
public class ClassInfo extends ClassFeatureInfo {

    /* 生成メソッド情報。 */
    private FactoryMethodInfo factory;

    /* フィールド情報。 */
    private List fields = new ArrayList();

    /* 初期化メソッド情報。 */
    private List initMethods = new ArrayList();

    /* メソッド情報。 */
    private List methods = new ArrayList();

    /**
     * デフォルトコンストラクタ。
     */
    public ClassInfo() {
        //no op
    }

    /**
     * 登録されているクラスから、インスタンスを生成して返却します。
     * 
     * @return 生成されたインスタンス
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object newInstance() throws InvocationBeansException {
        return newInstance(getObjectType());
    }

    /**
     * 指定されたクラスから、インスタンスを生成して返却します。
     * 
     * @param c
     *            生成クラス
     * @return 生成されたインスタンス
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object newInstance(Class c) throws InvocationBeansException {
        if (c == null) {
            throw new NullPointerException("argument is null.");
        }

        //new instance
        Object o = null;
        if (null != getFactory()) {
            if (null == getFactory().getObjectType()) {
                o = getFactory().invoke(c);
            } else {
                o = getFactory().invoke();
            }
        } else {
            o = BeanUtilities.newInstance(c);
        }
        //invoke init methods
        Iterator i = initMethods.iterator();
        while (i.hasNext()) {
            MethodInfo m = (MethodInfo) i.next();
            m.invoke(o);
        }
        return o;
    }

    /**
     * 指定されたオブジェクトから、ゲットプロパティ型のメソッド情報を実行し、プロパティ名をキーとして値をマッピングして返却します。
     * 
     * @param o
     *            取得するオブジェクトインスタンス
     * @return プロパティを格納したマップ
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Map getPropertyMap(Object o) throws InvocationBeansException {
        if (o == null) {
            throw new NullPointerException("argument is null.");
        }
        Map map = new HashMap();
        Iterator i = methods.iterator();
        while (i.hasNext()) {
            MethodInfo m = (MethodInfo) i.next();
            if (m.getType() == MethodInfo.GET_PROPERTY) {
                map.put(m.getMethodName(), m.invoke(o));
            }
        }
        return map;
    }

    //TODO 追加：新規・既存インスタンスへのフィールドの設定、フィールドのマップ取得、既存インスタンスへのプロパティの設定

    /*
     * Property
     */

    /**
     * 生成メソッド情報を返却します。
     * 
     * @return 生成メソッド情報
     */
    public FactoryMethodInfo getFactory() {
        return factory;
    }

    /**
     * 生成メソッド情報を格納します。
     * 
     * @param factory
     *            生成メソッド情報
     */
    public void setFactory(FactoryMethodInfo factory) {
        checkUnmodifiable();

        this.factory = factory;
    }

    /**
     * フィールド情報を返却します。
     * 
     * @return フィールド情報
     */
    public FieldInfo[] getFields() {
        return (FieldInfo[]) fields.toArray(new FieldInfo[0]);
    }

    /**
     * フィールド情報を追加します。
     * 
     * @param f
     *            フィールド情報
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addField(FieldInfo f) {
        checkUnmodifiable();

        if (f == null) {
            throw new NullPointerException("argument is null.");
        }
        this.fields.add(f);
    }

    /**
     * フィールド情報を配列を全て追加します。
     * 
     * @param fs
     *            フィールド情報を配列
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addFieldAll(FieldInfo[] fs) {
        checkUnmodifiable();

        if (fs == null) {
            throw new NullPointerException("argument is null.");
        }
        this.fields.addAll(Arrays.asList(fs));
    }

    /**
     * 初期化メソッド情報を返却します。
     * 
     * @return 初期化メソッド情報
     */
    public MethodInfo[] getInitMethods() {
        return (MethodInfo[]) initMethods.toArray(new MethodInfo[0]);
    }

    /**
     * 初期化メソッド情報を追加します。
     * 
     * @param m
     *            初期化メソッド情報
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addInitMethod(MethodInfo m) {
        checkUnmodifiable();

        if (m == null) {
            throw new NullPointerException("argument is null.");
        }
        this.initMethods.add(m);
    }

    /**
     * 初期化メソッド情報を配列を全て追加します。
     * 
     * @param ms
     *            初期化メソッド情報を配列
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addInitMethodAll(MethodInfo[] ms) {
        checkUnmodifiable();

        if (ms == null) {
            throw new NullPointerException("argument is null.");
        }
        this.initMethods.addAll(Arrays.asList(ms));
    }

    /**
     * メソッド情報を返却します。
     * 
     * @return メソッド情報
     */
    public MethodInfo[] getMethods() {
        return (MethodInfo[]) methods.toArray(new MethodInfo[0]);
    }

    /**
     * メソッド情報を追加します。
     * 
     * @param m
     *            メソッド情報
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addMethod(MethodInfo m) {
        checkUnmodifiable();

        if (m == null) {
            throw new NullPointerException("argument is null.");
        }
        this.methods.add(m);
    }

    /**
     * メソッド情報を配列を全て追加します。
     * 
     * @param ms
     *            メソッド情報を配列
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void addMethodAll(MethodInfo[] ms) {
        checkUnmodifiable();

        if (ms == null) {
            throw new NullPointerException("argument is null.");
        }
        this.methods.addAll(Arrays.asList(ms));
    }

}
