/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import shohaku.core.util.UnmodifiableStateException;

/**
 * 更新規則と名前で拘束されたパラメータ情報のリストを格納する機能を提供します。
 */
public class BindParametersInfo extends FeatureInfo {

    /* パラメータの型。 */
    private Class[] parameterTypes;

    /* パラメータの値。 */
    private Object[] parameterValues;
    
    /* パラメータ拘束名。 */
    private String[] parameterNames;

    /* パラメータ拘束ルール。 */
    private int[] parameterRules;

    /* パラメータ情報。 */
    private BindParameterInfo[] parameteres;

    /**
     * デフォルトコンストラクタ。
     */
    public BindParametersInfo() {
        //no op
    }

    /**
     * パラメータ情報の配列を指定して初期化します。
     * 
     * @param params
     *            パラメータの情報
     */
    public BindParametersInfo(BindParameterInfo[] params) {
        setParameteres(params);
    }

    /*
     * Property
     */

    /**
     * パラメータの値を返却します。
     * 
     * @return パラメータの値
     */
    public Object[] getParameterValues() {
        return parameterValues;
    }

    /**
     * パラメータの型を返却します。
     * 
     * @return パラメータの型
     */
    public Class[] getParameterTypes() {
        return parameterTypes;
    }
    
    /**
     * パラメータ拘束名を返却します。
     * 
     * @return パラメータ拘束名
     */
    public String[] geParameterNames() {
        return parameterNames;
    }

    /**
     * パラメータの拘束ルールを返却します。
     * 
     * @return パラメータの拘束ルール
     */
    public int[] getParameterRules() {
        return parameterRules;
    }

    /**
     * パラメータ情報の配列を返却します。
     * 
     * @return パラメータ情報の配列
     */
    public BindParameterInfo[] getParameteres() {
        return parameteres;
    }

    /**
     * パラメータ情報の配列を格納します。
     * 
     * @param params
     *            パラメータ情報の配列
     * @throws UnmodifiableStateException
     *             変更不可の状態で呼び出された場合
     */
    public void setParameteres(BindParameterInfo[] params) {
        checkUnmodifiable();

        if (params == null) {
            throw new NullPointerException("argument is null.");
        }
        Class[] types = new Class[params.length];
        Object[] values = new Object[params.length];
        String[] names = new String[params.length];
        int[] rules = new int[params.length];
        for (int i = 0; i < params.length; i++) {
            types[i] = params[i].getType();
            values[i] = params[i].getValue();
            names[i] = params[i].getBindName();
            rules[i] = params[i].getBindRule();
        }
        this.parameterTypes = types;
        this.parameterValues = values;
        this.parameterNames = names;
        this.parameterRules = rules;
        this.parameteres = params;
    }
}
