/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import java.lang.reflect.Method;

import shohaku.core.collections.Parameters;
import shohaku.core.lang.Eval;
import shohaku.core.util.Format;

/**
 * <code>BindMethod</code> インターフェースのデフォルト実装を提供します。
 */
final class BindMethodImpl implements BindMethod {

    /* 実行基のオブジェクト型。 */
    private Class objectType;

    /* 実行基のインスタンス。 */
    private Object srcObject;

    /* メソッド。 */
    private Method method;

    /* メソッド名。 */
    private String methodName;

    /* パラメータ型。 */
    private Class[] parameterTypes;

    /* パラメータ値。 */
    private Object[] parameterValues;

    /* パラメータの拘束名。 */
    private String[] parameterNames;

    /* パラメータの拘束タイプ。 */
    private int[] parameterRules;

    /*
     * デフォルトコンストラクタ。
     */
    BindMethodImpl() {
        //no op
    }

    /**
     * メソッドを呼び出し結果を返します。
     * 
     * @param values
     *            名前で拘束されるメソッドの引数値
     * @return メソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public Object invoke(Parameters values) throws InvocationBeansException {
        return invokeMethod(getObjectType(), getSrcObject(), getMethod(), values);
    }

    /**
     * 実行基のオブジェクト型を返却します。
     * 
     * @return 実行基のオブジェクト型
     */
    public Class getObjectType() {
        return objectType;
    }

    /**
     * メソッド名を返却します。
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * 実行基のオブジェクトを返却します．
     * 
     * @return 実行基のオブジェクト
     */
    public Object getSrcObject() {
        return srcObject;
    }

    /**
     * メソッドを返却します．
     * 
     * @return メソッド
     */
    public Method getMethod() {
        return method;
    }

    /*
     * package
     */

    /* メソッドの拘束を実行する。 */
    void bind() throws InvocationBeansException {
        Class[] types = getParameterTypes();
        Object[] values = getParameterValues();
        String[] names = getParameterNames();
        int[] rules = getParameterRules();
        if (!Eval.isEqualsLength(new Object[] { types, values, names, rules })) {
            throw new InvocationBeansException("parameter is illegal.");
        }
        if (Eval.isOrBlank(names)) {
            throw new InvocationBeansException("parameter is illegal.");
        }
        if (!Eval.isRange(rules, BIND_RULE_FINAL, BIND_RULE_REQUIRED)) {
            throw new InvocationBeansException("parameter is illegal.");
        }
        try {
            setMethod(getMatchingAccessibleMethod());
        } catch (NoSuchMethodException e) {
            throw new InvocationBeansException("parameter is illegal.", e);
        }
    }

    /* メソッド名を格納します． */
    void setMethodName(String methodName) {
        this.methodName = methodName;
    }

    /* オブジェクト型を格納します． */
    void setObjectType(Class objectType) {
        this.objectType = objectType;
    }

    /* 実行基のオブジェクトを格納します． */
    void setSrcObject(Object srcObject) {
        this.srcObject = srcObject;
    }

    /* パラメータの拘束名を格納します。 */
    void setParameterNames(String[] parameterNames) {
        this.parameterNames = parameterNames;
    }

    /* パラメータの拘束タイプを格納します。 */
    void setParameterRules(int[] parameterRules) {
        this.parameterRules = parameterRules;
    }

    /* パラメータ型を格納します。 */
    void setParameterTypes(Class[] parameterTypes) {
        this.parameterTypes = parameterTypes;
    }

    /* パラメータ値を格納します。 */
    void setParameterValues(Object[] parameterValues) {
        this.parameterValues = parameterValues;
    }

    /*
     * private
     */

    /* メソッドを格納します． */
    private void setMethod(Method method) {
        this.method = method;
    }

    /* パラメータの値を返却します。 */
    private Object[] getParameterValues() {
        return (parameterValues == null) ? new Object[0] : parameterValues;
    }

    /* パラメータの型情報を返却します。 */
    private Class[] getParameterTypes() {
        return (parameterTypes == null) ? new Class[0] : parameterTypes;
    }

    /* パラメータの拘束名を返却します。 */
    private String[] getParameterNames() {
        return (parameterNames == null) ? new String[0] : parameterNames;
    }

    /* パラメータの拘束タイプを返却します。 */
    private int[] getParameterRules() {
        return (parameterRules == null) ? new int[0] : parameterRules;
    }

    /* 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。 */
    private Object invokeMethod(Class c, Object obj, Method method, Parameters params) throws InvocationBeansException {
        Object[] values = (Object[]) getParameterValues().clone();
        String[] names = getParameterNames();
        int[] rules = getParameterRules();
        for (int i = 0; i < values.length; i++) {
            int rule = rules[i];
            String name = names[i];
            if (rule == 1) {
                if (params.containsName(name)) {
                    values[i] = params.getValue(name);
                }
            } else if (rule == 2) {
                if (params.containsName(name)) {
                    values[i] = params.getValue(name);
                } else {
                    //err
                }
            }
        }
        return invokeMethod(c, obj, method, values);
    }

    /* 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。 */
    private Object invokeMethod(Class c, Object obj, Method method, Object[] parameterValues)
            throws InvocationBeansException {
        return BeanUtilities.invokeMethod(c, obj, method, parameterValues);
    }

    /* 指定されたクラスから処理の対象となるメソッドオブジェクトを返却します。 */
    private Method getMatchingAccessibleMethod() throws NoSuchMethodException {
        Method method = BeanUtilities
                .getMatchingAccessibleMethod(getObjectType(), getMethodName(), getParameterTypes());
        if (method == null) {
            throw new NoSuchMethodException(Format.format("class:{0}, method:{1}, args:{2,array}", getObjectType(),
                    getMethodName(), getParameterTypes()));
        }
        return method;
    }

}