/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import shohaku.core.lang.Eval;
import shohaku.core.lang.ObjectCreationException;

/**
 * <code>ClassInfo</code> から指定された生成基準で <code>JavaBeans</code> のインスタンスを生成する機能を提供します。
 */
public class BeansFactory {

    /** シングルトンインスタンスを生成し返します。 */
    public static final String SINGLETON = "singleton";

    /** リクエストごとにインスタンスを生成し返します。 */
    public static final String PROTOTYPE = "prototype";

    /* 単一のインスタンスを保存します。 */
    private Object cache;

    /* インスタンスの生成基準の種別。 */
    private final String type;

    /* クラス情報。 */
    private final ClassInfo classInfo;

    /* 生成基のオブジェクト型。 */
    private final Class createSourceType;

    /**
     * 指定されたクラス情報とデフォルトの管理種別（PROTOTYPE）で初期化します。
     * 
     * @param classInfo
     *            クラス情報
     * @param createSourceType
     *            生成基のオブジェクト型
     */
    public BeansFactory(ClassInfo classInfo, Class createSourceType) {
        this(classInfo, createSourceType, PROTOTYPE);
    }

    /**
     * 指定されたクラス情報と管理種別で初期化します。
     * 
     * @param classInfo
     *            クラス情報
     * @param createSourceType
     *            生成基のオブジェクト型
     * @param type
     *            管理種別
     */
    public BeansFactory(ClassInfo classInfo, Class createSourceType, String type) {

        if (Eval.isOrNull(classInfo, createSourceType, type)) {
            throw new NullPointerException("argument is null.");
        }

        if (!Eval.isOrEquals(type, getFactoryTypes())) {
            throw new IllegalArgumentException("type:" + type);
        }

        this.classInfo = classInfo;
        this.createSourceType = createSourceType;
        this.type = type;
    }

    /**
     * インスタンスの生成基準の種別を返却します。
     * 
     * @return インスタンスの生成基準の種別
     */
    public String getType() {
        return type;
    }

    /**
     * クラス情報を返却します。
     * 
     * @return クラス情報
     */
    public ClassInfo getClassInfo() {
        return classInfo;
    }

    /**
     * 生成基のオブジェクト型を返却します。
     * 
     * @return 生成基のオブジェクト型
     */
    public Class getCreateSourceType() {
        return createSourceType;
    }

    /*
     * create
     */

    /**
     * インスタンスを生成して返却します。
     * 
     * @return 生成されたインスタンス
     * @throws ObjectCreationException
     *             オブジェクトの生成に失敗した場合に発生します
     */
    public Object getInstance() throws ObjectCreationException {
        return createObject();
    }

    /**
     * インスタンスを生成して返却する、拡張のためのフックポイントです。
     * 
     * @return 生成されたインスタンス
     * @throws ObjectCreationException
     *             オブジェクトの生成に失敗した場合に発生します
     */
    protected Object createObject() throws ObjectCreationException {
        try {
            if (SINGLETON.equals(getType())) {

                synchronized (this) {
                    if (getCache() == null) {
                        setCache(getClassInfo().newInstance(getCreateSourceType()));
                    }
                    return getCache();
                }

            } else if (PROTOTYPE.equals(getType())) {

                return getClassInfo().newInstance(getCreateSourceType());

            } else {
                throw new ObjectCreationException("instance type is illegal. type:" + getType());
            }
        } catch (InvocationBeansException e) {
            throw new ObjectCreationException("ClassInfo:" + getClassInfo() + ", type:" + getType(), e);
        }

    }

    /**
     * 全ての有効なインスタンスの生成基準の種別を返します。 <br>
     * インスタンスの生成基準を追加する場合は必ずオーバライドしてください。
     * 
     * @return 有効なインスタンスの生成基準の種別
     */
    protected Object[] getFactoryTypes() {
        return new Object[] { PROTOTYPE, SINGLETON };
    }

    /**
     * 保存されているインスタンスを返却します。
     * 
     * @return 保存されているインスタンス
     */
    protected synchronized Object getCache() {
        return cache;
    }

    /**
     * インスタンスを保存します。
     * 
     * @param instance
     *            保存するインスタンス
     */
    protected synchronized void setCache(Object instance) {
        this.cache = instance;
    }
}
