/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import shohaku.core.lang.Boxing;
import shohaku.core.lang.Eval;
import shohaku.core.util.Format;
import shohaku.core.util.cel.CELBinder;

/**
 * <code>JavaBean</code> を制御するユーティリティを提供します。 <br>
 * <br>
 * ライブラリの依存性を最小化する方針から <code>Jakarta</code> の <code>BeanUtils</code> 等の高機能ライブラリは使用しません。 <br>
 * コアライブラリには利用頻度の高い機能のみを定義し、必要に応じて他のライブラリを使用します。
 */
public class BeanUtilities {

    /*
     * Property
     */

    /**
     * ビーンのプロパティをマップに格納して返却します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @return 全てのプロパティを格納するマップ
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static Map getProperties(Object bean) throws InvocationBeansException {

        if (bean == null) {
            throw new NullPointerException("argument is null.");
        }

        try {

            Map props = new HashMap();
            PropertyDescriptor[] ds = getPropertyDescriptors(bean);
            for (int i = 0; i < ds.length; i++) {
                Method method = ds[i].getReadMethod();
                if (method != null) {
                    props.put(ds[i].getName(), method.invoke(bean, null));
                }
            }
            return props;

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException(e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException(e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException(e);
        }

    }

    /**
     * ビーンのプロパティを文字列に変換して返却します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @return ビーンのプロパティの文字列表現
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static String toBeanString(Object bean) throws InvocationBeansException {

        if (bean == null) {
            throw new NullPointerException("argument is null.");
        }

        try {

            StringBuffer sb = new StringBuffer();
            PropertyDescriptor[] ds = getPropertyDescriptors(bean);
            for (int i = 0; i < ds.length; i++) {
                Method method = ds[i].getReadMethod();
                if (method != null) {
                    sb.append(ds[i].getName());
                    sb.append(':');
                    sb.append(Format.toString(method.invoke(bean, null)));
                    sb.append(',');
                }
            }
            if (sb.length() != 0) {
                sb.insert(0, '{');
                sb.deleteCharAt(sb.length() - 1);
                sb.append('}');
            }
            return sb.toString();

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException(e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException(e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException(e);
        }

    }

    /**
     * 指定された参照パターンに基づきプロパティまたはメソッド、配列・リストの要素に再起的にアクセスして値を返却します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @param pattern
     *            参照パターン
     * @param args
     *            参照引数
     * @return 参照パターンの示すネストされたプロパティ
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static Object getNestedProperty(Object bean, String pattern, Object[] args) throws InvocationBeansException {
        if (Eval.isOrNull(bean, pattern)) {
            throw new NullPointerException("argument is null.");
        }
        try {
            String[] nodes = splitNodes(pattern.trim());
            Object o = bean;
            for (int i = 0; i < nodes.length; i++) {
                String node = nodes[i];
                switch (node.charAt(node.length() - 1)) {
                case ')':
                    o = getMethodProperty(o, node, args);
                    break;
                case ']':
                    o = getListProperty(o, node);
                    break;
                default:
                    o = getProperty(o, node);
                    break;
                }
            }
            return o;
        } catch (RuntimeException e) {
            throw new InvocationBeansException("illegal pattern:" + pattern, e);
        }
    }

    /* TODO 適切な実装では無い、後に修正する。（書式間違いが検証出来ない、正確に区切れない） */
    private static final String nodesSpecifier;
    static {
        StringBuffer sb = new StringBuffer();
        sb.append("(?:");
        sb.append("(?:^|\\.)([_a-z-A-Z]\\w+\\((?:\"(?>\\\\\"|[^\"])*\"|'.'|[^)]+)?\\))");//Method
        sb.append("|");
        sb.append("(?:^|\\.)([_a-z-A-Z]\\w+)");//Property
        sb.append("|");
        sb.append("(\\[[0-9]+\\])");//List or Array Index
        sb.append(")");
        nodesSpecifier = sb.toString();
    }

    /* の正規表現パターン。 */
    private static final Pattern nodesPattern = Pattern.compile(nodesSpecifier);

    private static String[] splitNodes(String pattern) {
        ArrayList list = new ArrayList(4);
        Matcher m = nodesPattern.matcher(pattern);

        int i = 0;
        while (i < pattern.length()) {
            if (m.find(i)) {
                for (int j = 0; j < m.groupCount(); j++) {
                    String e = m.group(j + 1);
                    if (e != null) {
                        list.add(e);
                        break;
                    }
                }
                i = m.end();
            } else {
                break;
            }
        }

        return (String[]) list.toArray(new String[0]);
    }

    /**
     * 配列またはリストからパターン文字列の示すインデックスの要素を返します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @param pattern
     *            参照パターン
     * @return パターン文字列の示す配列またはリストの要素
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static Object getListProperty(Object bean, String pattern) throws InvocationBeansException {
        int index = getListIndex(pattern);
        if (bean instanceof List) {
            bean = ((List) bean).get(index);
        } else if (Eval.isArray(bean)) {
            bean = Array.get(bean, index);
        } else {
            throw new InvocationBeansException("no Array, illegal pattern:" + pattern);
        }
        return bean;
    }

    private static int getListIndex(String pattern) throws InvocationBeansException {
        String sindex = pattern.substring(pattern.lastIndexOf('[') + 1, pattern.length() - 1);
        try {
            return Integer.parseInt(sindex);
        } catch (NumberFormatException e) {
            throw new InvocationBeansException("no Number, illegal pattern:" + pattern);
        }
    }

    /**
     * ビーンからプロパティ名の示すプロパティを取得して返します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @param name
     *            プロパティ名
     * @return プロパティ
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static Object getProperty(Object bean, String name) throws InvocationBeansException {

        if (Eval.isOrNull(bean, name)) {
            throw new NullPointerException("argument is null.");
        }
        try {

            Class beanClass = bean.getClass();
            Method method = getMatchingAccessibleMethod(beanClass, "get" + capitalize(name));
            if (method == null) {
                method = getMatchingAccessibleMethod(beanClass, "is" + capitalize(name));
            }
            return method.invoke(bean, null);

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("illegal property:" + name, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("illegal property:" + name, e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException("illegal property:" + name, e);
        }
    }

    /* TODO 適切な実装では無い、後に修正する。（書式間違いが検証出来ない、正確に区切れない） */
    private static final String argumentsSpecifier;
    static {
        StringBuffer sb = new StringBuffer();
        sb.append("(?:^|\\s*,)\\s*");//文始または区切り
        sb.append("(");//記憶域開始
        sb.append("\"(?>\\\\\"|[^\"])*\"");//文字列
        sb.append("|");
        sb.append("[-+]?[0-9A-Fa-f.]+[BSILFD]?");//数値
        sb.append("|");
        sb.append("true");
        sb.append("|");
        sb.append("false");
        sb.append("|");
        sb.append("null");
        sb.append("|");
        sb.append("'(?:.|\\\\'|\\\\u(?:[0-9A-Fa-f]){4})'");//文字
        sb.append("|");
        sb.append("\\|(?:[-0-9 :.]{5,23})\\|");//日付
        sb.append("|");
        sb.append("%\\d+");//引数参照
        sb.append(")");//記憶域終了
        sb.append("\\s*(?:,\\s*|$)");//区切りまたは文末
        argumentsSpecifier = sb.toString();
    }

    /* 式文内のメソッド・マップの引数を分割する正規表現パターン。 */
    private static final Pattern argumentsPattern = Pattern.compile(argumentsSpecifier);

    /* 式文内のメソッド・マップの引数を分割する。 */
    private static String[] splitArguments(String node) {
        ArrayList list = new ArrayList(7);
        Matcher m = argumentsPattern.matcher(node);

        int i = 0;
        while (i < node.length()) {
            if (m.find(i)) {
                list.add(m.group(1));
                i = m.end(1);
            } else {
                break;
            }
        }

        return (String[]) list.toArray(new String[0]);
    }

    private static Object[] getMethodArguments(String pattern, Object[] args) {
        String[] srcValues = splitArguments(pattern);
        Object[] arguments = new Object[srcValues.length];
        CELBinder exps = CELBinder.getBaseTypeCreationBinder();
        for (int i = 0; i < srcValues.length; i++) {
            if (srcValues[i].charAt(0) == '%') {
                int index = Integer.parseInt(srcValues[i].substring(1));
                arguments[i] = args[index];
            } else {
                arguments[i] = exps.getValue(srcValues[i]);
            }
        }
        return arguments;
    }

    /**
     * ビーンからパターン文字列の示すメソッドを実行して戻り値を返します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @param pattern
     *            参照パターン
     * @param args
     *            参照引数
     * @return パターン文字列の示すメソッドの戻り値
     * @throws InvocationBeansException
     *             処理の呼出に失敗した場合発生する
     */
    public static Object getMethodProperty(Object bean, String pattern, Object[] args) throws InvocationBeansException {

        if (Eval.isOrNull(bean, pattern)) {
            throw new NullPointerException("argument is null.");
        }
        try {

            Class beanClass = bean.getClass();

            String methodName = pattern.substring(0, pattern.indexOf('('));
            String ptns = pattern.substring(pattern.indexOf('(') + 1, pattern.length() - 1);
            Object[] values = null;
            if (!Eval.isBlank(ptns)) {
                values = getMethodArguments(ptns, args);
            }

            if (values != null) {
                Class[] types = toTypes(values);
                Method method = getMatchingAccessibleMethod(beanClass, methodName, types);
                if (method == null) {
                    throw new InvocationBeansException();
                }
                return method.invoke(bean, values);
            } else {
                Method method = getMatchingAccessibleMethod(beanClass, methodName);
                if (method == null) {
                    throw new InvocationBeansException();
                }
                return method.invoke(bean, null);
            }

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("illegal pattern:" + pattern, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("illegal pattern:" + pattern, e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException("illegal pattern:" + pattern, e);
        }
    }

    /**
     * オブジェクトの配列を基にその要素のクラスの配列を生成して返却する。
     * 
     * @param args
     *            生成基のオブジェクトの配列
     * @return 生成基に対応するクラスの配列
     */
    public static Class[] toTypes(Object[] args) {
        Class[] types = new Class[args.length];
        for (int i = 0; i < args.length; i++) {
            types[i] = args[i].getClass();
        }
        return types;
    }

    /**
     * ビーンの <code>java.beans.PropertyDescriptor</code> を返却します。
     * 
     * @param bean
     *            ビーンのインスタンス
     * @return ビーンの <code>java.beans.PropertyDescriptor</code>
     */
    public static PropertyDescriptor[] getPropertyDescriptors(Object bean) {

        if (bean == null) {
            throw new NullPointerException("argument is null.");
        }
        return (getPropertyDescriptors(bean.getClass()));

    }

    /**
     * ビーンの <code>java.beans.PropertyDescriptor</code> を返却します。
     * 
     * @param beanClass
     *            ビーンのクラス
     * @return ビーンの <code>java.beans.PropertyDescriptor</code>
     */
    public static PropertyDescriptor[] getPropertyDescriptors(Class beanClass) {

        if (beanClass == null) {
            throw new NullPointerException("argument is null.");
        }

        PropertyDescriptor[] descriptors = null;

        BeanInfo beanInfo = null;
        try {
            beanInfo = Introspector.getBeanInfo(beanClass);
        } catch (IntrospectionException e) {
            return new PropertyDescriptor[0];
        }
        descriptors = beanInfo.getPropertyDescriptors();
        if (descriptors == null) {
            descriptors = new PropertyDescriptor[0];
        }
        return descriptors;

    }

    /**
     * アクセス可能なフィールドを検索して返します。 <br>
     * フィールドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param fieldName
     *            フィールド名
     * @return アクセス可能なコンストラクタ、発見できない場合 <code>null</code>
     */
    public static Field getAccessibleField(Class c, String fieldName) {

        if (Eval.isOrNull(c, fieldName)) {
            throw new NullPointerException("argument is null.");
        }

        Field field = null;
        try {
            field = getAccessibleField(c.getField(fieldName));
        } catch (SecurityException e) {
            //no op
        } catch (NoSuchFieldException e) {
            //no op
        }
        return field;
    }

    /**
     * パラメータが空のアクセス可能なコンストラクタを検索して返します。 <br>
     * コンストラクタが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @return アクセス可能なコンストラクタ、発見できない場合 <code>null</code>
     */
    public static Constructor getMatchingAccessibleConstructor(Class c) {
        return getMatchingAccessibleConstructor(c, new Class[0]);

    }

    /**
     * パラメータを割り当てられるアクセス可能なコンストラクタを検索して返します。 <br>
     * コンストラクタが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param parameterTypes
     *            引数のパラメータ型
     * @return アクセス可能なコンストラクタ、発見できない場合 <code>null</code>
     */
    public static Constructor getMatchingAccessibleConstructor(Class c, Class[] parameterTypes) {

        if (Eval.isOrNull(c, parameterTypes)) {
            throw new NullPointerException("argument is null.");
        }

        try {

            Constructor constructor = getAccessibleConstructor(c.getConstructor(parameterTypes));
            if (constructor != null) {
                return constructor;
            }

        } catch (SecurityException e) {
            //no op
        } catch (NoSuchMethodException e) {
            //no op
        }

        int paramSize = parameterTypes.length;
        Constructor[] constructors = c.getConstructors();
        for (int i = 0, size = constructors.length; i < size; i++) {

            Class[] constructorParams = constructors[i].getParameterTypes();
            int constructorParamSize = constructorParams.length;
            if (constructorParamSize == paramSize) {
                boolean match = true;
                for (int n = 0; n < constructorParamSize; n++) {
                    if (!isAssignmentCompatible(constructorParams[n], parameterTypes[n])) {
                        match = false;
                        break;
                    }
                }

                if (match) {
                    Constructor constructor = getAccessibleConstructor(constructors[i]);
                    if (constructor != null) {
                        return constructor;
                    }
                }
            }
        }
        return null;

    }

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの設定メソッドを検索して返します。 <br>
     * プロパティの設定メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param propertyName
     *            検索するプロパティ名
     * @param parameterType
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの設定メソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleSetProperty(Class c, String propertyName, Class parameterType) {
        return getMatchingAccessibleSetProperty(c, propertyName, new Class[] { parameterType });
    }

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの設定メソッドを検索して返します。 <br>
     * プロパティの設定メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param propertyName
     *            検索するプロパティ名
     * @param parameterTypes
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの設定メソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleSetProperty(Class c, String propertyName, Class[] parameterTypes) {
        return getMatchingAccessibleMethod(c, "set" + capitalize(propertyName), parameterTypes);
    }

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの取得メソッドを検索して返します。 <br>
     * プロパティの取得メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param propertyName
     *            検索するプロパティ名
     * @param parameterType
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの取得メソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleGetProperty(Class c, String propertyName, Class parameterType) {
        return getMatchingAccessibleGetProperty(c, propertyName, new Class[] { parameterType });
    }

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの取得メソッドを検索して返します。 <br>
     * プロパティの取得メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param propertyName
     *            検索するプロパティ名
     * @param parameterTypes
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの取得メソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleGetProperty(Class c, String propertyName, Class[] parameterTypes) {
        Method method = getMatchingAccessibleMethod(c, "get" + capitalize(propertyName), parameterTypes);
        if (method == null) {
            method = getMatchingAccessibleMethod(c, "is" + capitalize(propertyName), parameterTypes);
        }
        return method;
    }

    /**
     * パラメータを持たないアクセス可能なメソッドを検索して返します。 <br>
     * メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param methodName
     *            検索するメソッド名
     * @return アクセス可能なメソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleMethod(Class c, String methodName) {
        return getMatchingAccessibleMethod(c, methodName, new Class[0]);

    }

    /**
     * パラメータを割り当てられるアクセス可能なメソッドを検索して返します。 <br>
     * メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param methodName
     *            検索するメソッド名
     * @param parameterType
     *            引数のパラメータ型
     * @return アクセス可能なメソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleMethod(Class c, String methodName, Class parameterType) {
        return getMatchingAccessibleMethod(c, methodName, new Class[] { parameterType });

    }

    /**
     * パラメータを割り当てられるアクセス可能なメソッドを検索して返します。 <br>
     * メソッドが発見できない場合 <code>null</code> を返します。
     * 
     * @param c
     *            検索するクラス
     * @param methodName
     *            検索するメソッド名
     * @param parameterTypes
     *            引数のパラメータ型
     * @return アクセス可能なメソッド、発見できない場合 <code>null</code>
     */
    public static Method getMatchingAccessibleMethod(Class c, String methodName, Class[] parameterTypes) {

        if (Eval.isOrNull(c, methodName, parameterTypes)) {
            throw new NullPointerException("argument is null.");
        }

        try {
            Method method = c.getMethod(methodName, parameterTypes);
            try {
                method.setAccessible(true);
            } catch (SecurityException se) {
                //no op
            }
            return method;
        } catch (SecurityException se) {
            //no op
        } catch (NoSuchMethodException e) {
            //no op
        }

        int paramSize = parameterTypes.length;
        Method[] methods = c.getMethods();
        for (int i = 0, size = methods.length; i < size; i++) {
            if (methods[i].getName().equals(methodName)) {

                Class[] methodsParams = methods[i].getParameterTypes();
                int methodParamSize = methodsParams.length;
                if (methodParamSize == paramSize) {
                    boolean match = true;
                    for (int n = 0; n < methodParamSize; n++) {
                        if (!isAssignmentCompatible(methodsParams[n], parameterTypes[n])) {
                            match = false;
                            break;
                        }
                    }

                    if (match) {
                        Method method = getAccessibleMethod(methods[i]);
                        if (method != null) {
                            try {
                                method.setAccessible(true);
                            } catch (SecurityException se) {
                                //no op
                            }
                            return method;
                        }
                    }
                }
            }
        }
        return null;
    }

    /* 第一引数に対して第二引数が割り当て可能か検証します。 */
    private static boolean isAssignmentCompatible(Class fromType, Class toType) {
        //単純に割り当て可能か検証する
        if (fromType.isAssignableFrom(toType)) {
            return true;
        }
        //ラッパー型が一致する場合許可
        if (fromType.isPrimitive()) {
            Class parameterWrapperClass = Boxing.boxClass(fromType);
            if (parameterWrapperClass != null) {
                return parameterWrapperClass.equals(toType);
            }
        }
        return false;
    }

    /* フィールド自身と属するクラスまたはクラスが実装するインターフェースが公開されているか検証します。 */
    private static Field getAccessibleField(Field field) {

        if (field == null) {
            return null;
        }

        if (!Modifier.isPublic(field.getModifiers())) {
            return null;
        }

        Class c = field.getDeclaringClass();
        if (Modifier.isPublic(c.getModifiers())) {
            return field;
        }

        field = getAccessibleFieldFromInterfaceNest(c, field.getName());
        return field;

    }

    /* フィールドのクラスが実装するインターフェースが公開されているか再起的に検証します。 */
    private static Field getAccessibleFieldFromInterfaceNest(Class c, String fieldName) {

        Field field = null;

        for (; c != null; c = c.getSuperclass()) {

            Class[] interfaces = c.getInterfaces();
            for (int i = 0; i < interfaces.length; i++) {

                if (!Modifier.isPublic(interfaces[i].getModifiers())) {
                    continue;
                }

                try {
                    field = interfaces[i].getDeclaredField(fieldName);
                } catch (NoSuchFieldException e) {
                    //no op
                } catch (SecurityException e) {
                    //no op
                }
                if (field != null) {
                    break;
                }

                field = getAccessibleFieldFromInterfaceNest(interfaces[i], fieldName);
                if (field != null) {
                    break;
                }
            }
        }

        if (field != null) {
            return field;
        }

        return null;

    }

    /* メソッド自身と属するクラスまたはクラスが実装するインターフェースが公開されているか検証します。 */
    private static Method getAccessibleMethod(Method method) {

        if (method == null) {
            return null;
        }

        if (!Modifier.isPublic(method.getModifiers())) {
            return null;
        }

        Class c = method.getDeclaringClass();
        if (Modifier.isPublic(c.getModifiers())) {
            return method;
        }

        method = getAccessibleMethodFromInterfaceNest(c, method.getName(), method.getParameterTypes());
        return method;

    }

    /* コンストラクタと属するクラスが公開されているか検証します。 */
    private static Constructor getAccessibleConstructor(Constructor constructor) {

        if (constructor == null) {
            return null;
        }

        if (!Modifier.isPublic(constructor.getModifiers())) {
            return null;
        }

        Class c = constructor.getDeclaringClass();
        if (Modifier.isPublic(c.getModifiers())) {
            return constructor;
        }

        return null;

    }

    /* メソッドのクラスが実装するインターフェースが公開されているか再起的に検証します。 */
    private static Method getAccessibleMethodFromInterfaceNest(Class c, String methodName, Class[] parameterTypes) {

        Method method = null;

        for (; c != null; c = c.getSuperclass()) {

            Class[] interfaces = c.getInterfaces();
            for (int i = 0; i < interfaces.length; i++) {

                if (!Modifier.isPublic(interfaces[i].getModifiers())) {
                    continue;
                }

                try {
                    method = interfaces[i].getDeclaredMethod(methodName, parameterTypes);
                } catch (NoSuchMethodException e) {
                    //no op
                } catch (SecurityException e) {
                    //no op
                }
                if (method != null) {
                    break;
                }

                method = getAccessibleMethodFromInterfaceNest(interfaces[i], methodName, parameterTypes);
                if (method != null) {
                    break;
                }
            }
        }

        if (method != null) {
            return method;
        }

        return null;

    }

    /*
     * new instance
     */

    /**
     * パラメータを指定したコンストラクタからインスタンスを生成して返却します。
     * 
     * @param c
     * @return
     * @throws InvocationBeansException
     */
    public static Object newInstance(Class c) throws InvocationBeansException {
        return newInstance(c, new Class[0], new Object[0]);
    }

    /**
     * パラメータを指定したコンストラクタからインスタンスを生成して返却します。
     * 
     * @param c
     * @param parameterValues
     * @return
     * @throws InvocationBeansException
     */
    public static Object newInstance(Class c, Object[] parameterValues) throws InvocationBeansException {
        return newInstance(c, toTypes(parameterValues), parameterValues);
    }

    /**
     * パラメータを指定したコンストラクタからインスタンスを生成して返却します。
     * 
     * @param c
     * @param parameterTypes
     * @param parameterValues
     * @return
     * @throws InvocationBeansException
     */
    public static Object newInstance(Class c, Class[] parameterTypes, Object[] parameterValues)
            throws InvocationBeansException {

        if (Eval.isOrNull(c, parameterTypes, parameterValues)) {
            throw new NullPointerException("argument is null.");
        }

        try {

            Constructor constructor = getConstructor(c, parameterTypes);
            return constructor.newInstance(newParameterValues(parameterValues));

        } catch (SecurityException e) {
            throw new InvocationBeansException("Class#getConstructor class:" + c, e);
        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("Constructor#newInstance class:" + c, e);
        } catch (NoSuchMethodException e) {
            throw new InvocationBeansException("Class#getConstructor class:" + c, e);
        } catch (InstantiationException e) {
            throw new InvocationBeansException("Constructor#newInstance class:" + c, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("Constructor#newInstance class:" + c, e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException("Constructor#newInstance class:" + c, e);
        }
    }

    /**
     * 処理の対象となるコンストラクタオブジェクトを返却します。
     * 
     * @param c
     * @param parameterTypes
     * @return
     * @throws NoSuchMethodException
     */
    public static Constructor getConstructor(Class c, Class[] parameterTypes) throws NoSuchMethodException {

        if (Eval.isOrNull(c, parameterTypes)) {
            throw new NullPointerException("argument is null.");
        }

        Constructor constructor = BeanUtilities.getMatchingAccessibleConstructor(c, parameterTypes);
        if (constructor == null) {
            throw new NoSuchMethodException(Format.format("class:{0}, args:{1,array}", c, parameterTypes));
        }
        return constructor;
    }

    /*
     * invoke method
     */

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     * @param obj
     * @param methodName
     * @return
     * @throws InvocationBeansException
     */
    public static Object invokeMethod(Class c, Object obj, String methodName) throws InvocationBeansException {
        return invokeMethod(c, obj, methodName, new Class[0], new Object[0]);
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     * @param obj
     * @param methodName
     * @param parameterValues
     * @return
     * @throws InvocationBeansException
     */
    public static Object invokeMethod(Class c, Object obj, String methodName, Object[] parameterValues)
            throws InvocationBeansException {
        return invokeMethod(c, obj, methodName, toTypes(parameterValues), parameterValues);
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     * @param obj
     * @param methodName
     * @param parameterTypes
     * @param parameterValues
     * @return
     * @throws InvocationBeansException
     */
    public static Object invokeMethod(Class c, Object obj, String methodName, Class[] parameterTypes,
            Object[] parameterValues) throws InvocationBeansException {

        if (Eval.isOrNull(c, methodName, parameterTypes, parameterValues)) {
            throw new NullPointerException("argument is null.");
        }

        Method method = BeanUtilities.getMatchingAccessibleMethod(c, methodName, parameterTypes);

        try {

            if (Modifier.isStatic(method.getModifiers())) {
                return method.invoke(null, newParameterValues(parameterValues));
            } else {
                if (obj == null) {
                    throw new NullPointerException("argument obj is null.");
                }
                return method.invoke(obj, newParameterValues(parameterValues));
            }

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        }
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返します。
     * 
     * @param c
     * @param obj
     * @param method
     * @param parameterValues
     * @return
     * @throws InvocationBeansException
     */
    public static Object invokeMethod(Class c, Object obj, Method method, Object[] parameterValues)
            throws InvocationBeansException {

        if (Eval.isOrNull(c, method, parameterValues)) {
            throw new NullPointerException("argument is null.");
        }

        try {

            if (Modifier.isStatic(method.getModifiers())) {
                return method.invoke(null, newParameterValues(parameterValues));
            } else {
                if (obj == null) {
                    throw new NullPointerException("argument obj is null.");
                }
                return method.invoke(obj, newParameterValues(parameterValues));
            }

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        } catch (InvocationTargetException e) {
            throw new InvocationBeansException("Method#invoke class:" + c, e);
        }
    }

    /**
     * 指定されたクラスから処理の対象となるメソッドオブジェクトを返却します。
     * 
     * @param c
     * @param methodName
     * @param parameterTypes
     * @return
     * @throws NoSuchMethodException
     */
    public static Method getMethod(Class c, String methodName, Class[] parameterTypes) throws NoSuchMethodException {

        if (Eval.isOrNull(c, methodName, parameterTypes)) {
            throw new NullPointerException("argument is null.");
        }

        Method method = BeanUtilities.getMatchingAccessibleMethod(c, methodName, parameterTypes);
        if (method == null) {
            throw new NoSuchMethodException(Format.format("class:{0}, method:{1}, args:{2,array}", c, methodName,
                    parameterTypes));
        }
        return method;
    }

    /* フィールド値の取得。 */
    static Object getFieldValue(Class c, Object obj, Field field) throws InvocationBeansException {

        if (Eval.isOrNull(c, obj, field)) {
            throw new NullPointerException("argument is null.");
        }

        try {

            return field.get(obj);

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("field#get class:" + c + ", field:" + field, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("field#get class:" + c + ", field:" + field, e);
        }
    }

    /* フィールド値を設定し古い値を返却。 */
    static Object setFieldValue(Class c, Object obj, Field field, Object newValue) throws InvocationBeansException {

        if (Eval.isOrNull(c, obj, field)) {
            throw new NullPointerException("argument is null.");
        }

        try {

            Object oldValue = getFieldValue(c, obj, field);
            field.set(obj, newValue);
            return oldValue;

        } catch (IllegalArgumentException e) {
            throw new InvocationBeansException("field#set class:" + c + ", field:" + field + ", value:" + newValue, e);
        } catch (IllegalAccessException e) {
            throw new InvocationBeansException("field#set class:" + c + ", field:" + field + ", value:" + newValue, e);
        }
    }

    /*
     * helper
     */

    /* 新規のパラメータの値を生成して返却します。 */
    private static Object[] newParameterValues(Object[] srcValues) throws InvocationBeansException {

        if (srcValues == null) {
            throw new NullPointerException("argument srcValues is null.");
        }

        Object[] newValues = new Object[srcValues.length];
        for (int i = 0; i < srcValues.length; i++) {
            Object o = srcValues[i];
            if (o instanceof ClassInfo) {
                o = ((ClassInfo) o).newInstance();
            }
            newValues[i] = o;
        }
        return newValues;
    }

    /* 先頭文字を大文字に変換します。 */
    private static String capitalize(String s) {
        if (s == null || s.length() == 0) {
            return s;
        }
        char[] chars = s.toCharArray();
        chars[0] = Character.toUpperCase(chars[0]);
        return new String(chars);
    }
}
