/*
NSToolbarItemViewerEx.h

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "NSToolbarItemViewerEx.h"

@implementation NSToolbarItemViewerEx

+ (void)load
{
    [self poseAsClass:[NSToolbarItemViewer class]];
}

- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [super dealloc];
}

- (BOOL)_hasMenu
{
    // Create array of identifier which have menu
    static NSArray* _menuItemIdentifiers = nil;
    if (!_menuItemIdentifiers) {
        _menuItemIdentifiers = [[NSArray arrayWithObjects:
            SRGoBackIdentifier, SRGoForwardIdentifier, nil] retain];
    }
    
    // Check identifier
    id  item = nil;
    id  itemIdentifier = nil;
    if ([self respondsToSelector:@selector(item)]) {
        item = [self item];
        if ([item respondsToSelector:@selector(itemIdentifier)]) {
            itemIdentifier = [item itemIdentifier];
        }
    }
    if (![_menuItemIdentifiers containsObject:itemIdentifier]) {
        return NO;
    }
    
    // Check is enabled
    if (![item isEnabled]) {
        return NO;
    }
    
    return YES;
}

- (NSMenu*)_menu
{
    // Get identifier
    id  itemIdentifier = nil;
    if ([self respondsToSelector:@selector(item)]) {
        id  item;
        item = [self item];
        if ([item respondsToSelector:@selector(itemIdentifier)]) {
            itemIdentifier = [item itemIdentifier];
        }
    }
    if (!itemIdentifier) {
        return nil;
    }
    
    // Get web view
    id          windowController;
    WebView*    webView = nil;
    windowController = [[self window] windowController];
    if ([windowController respondsToSelector:@selector(selectedWebView)]) {
        webView = [windowController selectedWebView];
    }
    if (!webView) {
        return nil;
    }
    
    // Create menu
    NSMenu* menu = nil;
    // For back menu
    if ([itemIdentifier isEqualToString:SRGoBackIdentifier]) {
        // Get back list
        NSArray*    list;
        list = [[webView backForwardList] backListWithLimit:50];
        
        // Make back menu
        menu = SRCreateHistoryMenu(list, YES);
        if ([menu numberOfItems] == 0) {
            menu = nil;
        }
    }
    // For forward menu
    if ([itemIdentifier isEqualToString:SRGoForwardIdentifier]) {
        // Get forward list
        NSArray*    list;
        list = [[webView backForwardList] forwardListWithLimit:50];
        
        // Make forward menu
        menu = SRCreateHistoryMenu(list, NO);
        if ([menu numberOfItems] == 0) {
            menu = nil;
        }
    }
    
    return menu;
}

- (NSMenu*)menuForEvent:(NSEvent*)event
{
    if (![self _hasMenu]) {
        return [super menuForEvent:event];
    }
    
    return [self _menu];
}

- (void)mouseDown:(NSEvent*)event
{
    // Check has menu
    if (![self _hasMenu]) {
        [super mouseDown:event];
        return;
    }
    
    // Highlight itself
    if ([self respondsToSelector:@selector(_setHighlighted:displayNow:)]) {
        [self _setHighlighted:YES displayNow:YES];
    }
    
    // Get mouse location
    NSPoint mouseLoc;
    mouseLoc = [self convertPoint:[event locationInWindow] fromView:nil];
    
    // Wait next event a moment
    NSEvent*    waitingEvent;
    NSRect      offsetRect;
    NSDate*     expireDate;
    offsetRect = NSMakeRect(mouseLoc.x - 2.0, mouseLoc.y - 2.0, 5.0, 5.0);
    expireDate = [NSDate dateWithTimeIntervalSinceNow:0.3];
    
    while (YES) {
        waitingEvent = [NSApp nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask) 
                untilDate:expireDate 
                inMode:NSEventTrackingRunLoopMode 
                dequeue:YES];
        
        // If timer is expired, show context menu
        if (!waitingEvent) {
            break;
        }
        // If mouse up is detected, do mouse up immediately
        else if ([waitingEvent type] == NSLeftMouseUp) {
            [NSApp postEvent:waitingEvent atStart:YES];
            [super mouseDown:event];
            
            // Make highlight off
            if ([self respondsToSelector:@selector(_setHighlighted:displayNow:)]) {
                [self _setHighlighted:NO displayNow:YES];
            }
            
            return;
        }
        // Permit mouse movement up to 3 pixels
        else if ([waitingEvent type] == NSLeftMouseDragged) {
            NSPoint currentMouseLoc;
            currentMouseLoc = [self convertPoint:[waitingEvent locationInWindow] fromView:nil];
            if (!NSPointInRect(currentMouseLoc, offsetRect)) {
                [NSApp postEvent:waitingEvent atStart:YES];
                [super mouseDown:event];
                
                // Make highlight off
                if ([self respondsToSelector:@selector(_setHighlighted:displayNow:)]) {
                    [self _setHighlighted:NO displayNow:YES];
                }
                
                return;
            }
        }
    }
    
    //
    // Show poup menu
    //
    
    // Get menu
    NSMenu* menu;
    menu = [self _menu];
    if (!menu) {
        [super mouseDown:event];
        return;
    }
    
    // Move event location
    NSPoint     popupPoint;
    NSEvent*    popupEvent;
    popupPoint = [self convertPoint:NSMakePoint(0, 0) toView:nil];
    popupEvent = [NSEvent mouseEventWithType:[event type] 
            location:popupPoint 
            modifierFlags:[event modifierFlags] 
            timestamp:[event timestamp] 
            windowNumber:[event windowNumber] 
            context:[event context] 
            eventNumber:[event eventNumber] 
            clickCount:[event clickCount] 
            pressure:[event pressure]];
    
    // Register notification for menu
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    [center addObserver:self 
            selector:@selector(_menuDidEndTracking:) 
            name:NSMenuDidEndTrackingNotification 
            object:menu];
    
    // Popup context menu
    [NSMenu popUpContextMenu:menu withEvent:popupEvent forView:self];
}

- (void)_menuDidEndTracking:(NSNotification*)notification
{
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    
    // Remove notification
    [center removeObserver:self 
            name:NSMenuDidEndTrackingNotification 
            object:[notification object]];
    
    // Make highlight off
    if ([self respondsToSelector:@selector(_setHighlighted:displayNow:)]) {
        [self _setHighlighted:NO displayNow:YES];
    }
}

@end
