/* RX On-chip data flash driver
 * Copyright 2014 Yoshinori Sato <ysato@users.sorceforge.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>
#include <environment.h>
#include <flash.h>
#include <asm/io.h>
#include "rx63xflash.h"

#define ID_DATA		0x52584454
#define ID_CODE		0x52584344

#define TO_PE(address)	((address) & 0x00ffffff)

flash_info_t flash_info[] = {
	{
		.size = 0x0800,
		.sector_count = 16,
		.flash_id = ID_DATA,
		.start[0] = 0x00100000,
		.protect[0] = 1,
		.start[1] = 0x00100800,
		.protect[1] = 1,
		.start[2] = 0x00101000,
		.protect[2] = 1,
		.start[3] = 0x00101800,
		.protect[3] = 1,
		.start[4] = 0x00102000,
		.protect[4] = 1,
		.start[5] = 0x00102800,
		.protect[5] = 1,
		.start[6] = 0x00103000,
		.protect[6] = 1,
		.start[7] = 0x00103800,
		.protect[7] = 1,
		.start[8] = 0x00104000,
		.protect[8] = 1,
		.start[9] = 0x00104800,
		.protect[9] = 1,
		.start[10] = 0x00105000,
		.protect[10] = 1,
		.start[11] = 0x00105800,
		.protect[11] = 1,
		.start[12] = 0x00106000,
		.protect[12] = 1,
		.start[13] = 0x00106800,
		.protect[13] = 1,
		.start[14] = 0x00107000,
		.protect[14] = 1,
		.start[15] = 0x00107800,
		.protect[15] = 1,
	},
	{
		.size = 0x10000,
		.sector_count = 16,
		.flash_id = ID_CODE,
		.start[0] = 0xffe00000,
		.protect[0] = 1,
		.start[1] = 0xffe10000,
		.protect[1] = 1,
		.start[2] = 0xffe20000,
		.protect[2] = 1,
		.start[3] = 0xffe30000,
		.protect[3] = 1,
		.start[4] = 0xffe40000,
		.protect[4] = 1,
		.start[5] = 0xffe50000,
		.protect[5] = 1,
		.start[6] = 0xffe60000,
		.protect[6] = 1,
		.start[7] = 0xffe70000,
		.protect[7] = 1,
		.start[8] = 0xffe80000,
		.protect[8] = 1,
		.start[9] = 0xffe90000,
		.protect[9] = 1,
		.start[10] = 0xffea0000,
		.protect[10] = 1,
		.start[11] = 0xffeb0000,
		.protect[11] = 1,
		.start[12] = 0xffec0000,
		.protect[12] = 1,
		.start[13] = 0xffed0000,
		.protect[13] = 1,
		.start[14] = 0xffee0000,
		.protect[14] = 1,
		.start[15] = 0xffef0000,
		.protect[15] = 1,
	},
	{
		.size = 0x8000,
		.sector_count = 16,
		.flash_id = ID_CODE,
		.start[0] = 0xfff00000,
		.protect[0] = 1,
		.start[1] = 0xfff08000,
		.protect[1] = 1,
		.start[2] = 0xfff10000,
		.protect[2] = 1,
		.start[3] = 0xfff18000,
		.protect[3] = 1,
		.start[4] = 0xfff20000,
		.protect[4] = 1,
		.start[5] = 0xfff28000,
		.protect[5] = 1,
		.start[6] = 0xfff30000,
		.protect[6] = 1,
		.start[7] = 0xfff38000,
		.protect[7] = 1,
		.start[8] = 0xfff40000,
		.protect[8] = 1,
		.start[9] = 0xfff48000,
		.protect[9] = 1,
		.start[10] = 0xfff50000,
		.protect[10] = 1,
		.start[11] = 0xfff58000,
		.protect[11] = 1,
		.start[12] = 0xfff60000,
		.protect[12] = 1,
		.start[13] = 0xfff68000,
		.protect[13] = 1,
		.start[14] = 0xfff70000,
		.protect[14] = 1,
		.start[15] = 0xfff78000,
		.protect[15] = 1,
	},
	{
		.size = 0x4000,
		.sector_count = 30,
		.flash_id = ID_CODE,
		.start[0] = 0xfff80000,
		.protect[0] = 1,
		.start[1] = 0xfff84000,
		.protect[1] = 1,
		.start[2] = 0xfff88000,
		.protect[2] = 1,
		.start[3] = 0xfff8c000,
		.protect[3] = 1,
		.start[4] = 0xfff90000,
		.protect[4] = 1,
		.start[5] = 0xfff94000,
		.protect[5] = 1,
		.start[6] = 0xfff98000,
		.protect[6] = 1,
		.start[7] = 0xfff9c000,
		.protect[7] = 1,
		.start[8] = 0xfffa0000,
		.protect[8] = 1,
		.start[9] = 0xfffa4000,
		.protect[9] = 1,
		.start[10] = 0xfffa8000,
		.protect[10] = 1,
		.start[11] = 0xfffac000,
		.protect[11] = 1,
		.start[12] = 0xfffb0000,
		.protect[12] = 1,
		.start[13] = 0xfffb4000,
		.protect[13] = 1,
		.start[14] = 0xfffb8000,
		.protect[14] = 1,
		.start[15] = 0xfffbc000,
		.protect[15] = 1,
		.start[16] = 0xfffc0000,
		.protect[16] = 1,
		.start[17] = 0xfffc4000,
		.protect[17] = 1,
		.start[18] = 0xfffc8000,
		.protect[18] = 1,
		.start[19] = 0xfffcc000,
		.protect[19] = 1,
		.start[20] = 0xfffd0000,
		.protect[20] = 1,
		.start[21] = 0xfffd4000,
		.protect[21] = 1,
		.start[22] = 0xfffd8000,
		.protect[22] = 1,
		.start[23] = 0xfffdc000,
		.protect[23] = 1,
		.start[24] = 0xfffe0000,
		.protect[24] = 1,
		.start[25] = 0xfffe4000,
		.protect[25] = 1,
		.start[26] = 0xfffe8000,
		.protect[26] = 1,
		.start[27] = 0xfffec000,
		.protect[27] = 1,
		.start[28] = 0xffff0000,
		.protect[28] = 1,
		.start[29] = 0xffff4000,
		.protect[29] = 1,
	},
	{
		.size = 0x1000,
		.sector_count = 8,
		.flash_id = ID_CODE,
		.start[0] = 0xffff8000,
		.protect[0] = 1,
		.start[1] = 0xffff9000,
		.protect[1] = 1,
		.start[2] = 0xffffa000,
		.protect[2] = 1,
		.start[3] = 0xffffb000,
		.protect[3] = 1,
		.start[4] = 0xffffc000,
		.protect[4] = 1,
		.start[5] = 0xffffd000,
		.protect[5] = 1,
		.start[6] = 0xffffe000,
		.protect[6] = 1,
		.start[7] = 0xfffff000,
		.protect[7] = 1,
	},
};

#if defined(CONFIG_ENV_IS_IN_FLASH) || defined(CONFIG_ENV_ADDR_REDUND) || (CONFIG_SYS_MONITOR_BASE >= CONFIG_SYS_FLASH_BASE)
flash_info_t *flash_get_info(ulong base)
{
	flash_info_t *info;

	info = &flash_info[0];
	if (info->size && info->start[0] <= base &&
	    base <= info->start[0] + info->size - 1)
		return info;
	else
		return NULL;
}
#endif

unsigned long flash_sector_size(flash_info_t *info, flash_sect_t sect)
{
	return info->size;
}

static int protect_d(long sector, int prot)
{
	unsigned short key;
	unsigned long dflwe;
	unsigned short reg;
	key = (sector < 8)?DFLWE0_KEY:DFLWE1_KEY;
	dflwe = DFLWE0 + 2 * (sector / 8);
	reg = __raw_readw(dflwe) & 0x00ff;
	if (prot)
		reg &= ~(1 << (sector & 7));
	else
		reg |= (1 << (sector & 7));
	__raw_writew((key << 8)| reg, dflwe);
	return 0;
}

int flash_real_protect(flash_info_t * info, long sector, int prot)
{
	int r = 0;
	unsigned char fentry = 1 << (3 - ((info->start[sector] >> 19) & 3));
	if (info->flash_id == ID_DATA)
		r = protect_d(sector, prot);
	else {
		r = rx_flash_protect_c(TO_PE(info->start[sector]), prot, fentry);
	}
	if (r == 0)
		info->protect[sector] = prot;
	return r;
}

static int flash_erase_page_d(unsigned int addr)
{
	rx_flash_pe_mode(addr, 0x80);
	int i;
	for (i = 0; i < 2048; i += 32) {
		__raw_writeb(0x20, addr + i);
		__raw_writeb(0xd0, addr + i);
		if (rx_flash_wait_frdy() < 0) {
			rx_flash_normal_mode(addr);
			return ERR_TIMOUT;
		}
	}
	rx_flash_normal_mode(addr);
	return 0;
}

int flash_erase(flash_info_t *info, int s_first, int s_last)
{
	int i;
	if (info->flash_id == ID_DATA) {
		for (i = s_first; i <= s_last; i++) 
			if (flash_erase_page_d(info->start[i]) < 0)
				return ERR_TIMOUT;
	} else {
		for (i = s_first; i <= s_last; i++) {
			unsigned char fentry = 1 << (3 - ((info->start[i] >> 19) & 3));
			if (rx_flash_erase_page_c(TO_PE(info->start[i]), fentry) < 0)
				return ERR_TIMOUT;
		}
	}		
	return 0;
}

void flash_print_info(flash_info_t *info)
{
	int i;

	printf("RX On-chip %s flash\n",
	       (info->flash_id == ID_DATA)?"DATA":"CODE");
	for(i = 0; i < info->sector_count; i++) {
		printf("Sector %d: %ldKB (%08lx - %08lx)\n",
		       i, info->size / 1024,
		       info->start[i], info->start[i] + info->size - 1);
	}
}

static int write_d(uchar *src, ulong addr, ulong cnt)
{
	ulong dst;
	rx_flash_pe_mode(addr, 0x80);
	dst = addr & ~0x0000001;
	while(cnt > 0) {
		__raw_writeb(FCU_PROG, dst);
		__raw_writeb(1, dst);
		if (dst < addr || cnt == 0) 
			__raw_writew(0xffff, dst);
		else {
			__raw_writew(*(unsigned short *)src, dst);
			cnt -= 2;
			src += 2;
		}
		__raw_writeb(FCU_PROG_DONE, dst);
		dst += 2;
		if (rx_flash_wait_frdy() < 0) {
			rx_flash_normal_mode(addr);
			return ERR_TIMOUT;
		}
	}
	rx_flash_normal_mode(addr);
	return 0;
}

int write_buff(flash_info_t *info, uchar *src, ulong addr, ulong cnt)
{
	if (info->flash_id == ID_DATA) {
		if (write_d(src, addr, cnt) < 0)
			return ERR_TIMOUT;
	} else {
		unsigned char fentry = 1 << (3 - ((addr >> 19) & 3));
		if (rx_flash_write_c(src, TO_PE(addr), cnt, fentry) < 0)
			return ERR_TIMOUT;
	}		
	return 0;
}

unsigned char _iram_text[], _iram_text_end[], _iram_start[];

unsigned long flash_init(void)
{
	/* setup FCU RAM */
	__raw_writew(FENTRYR_KEY | 0x00, FENTRYR);
	__raw_writew(FCURAME_KEY | FCURAME_FCRME, FCURAME);
	memcpy((void *)FCURAM_TOP, (void *)FCUFIRM_TOP, FCUFIRM_SIZE);
	memcpy(_iram_text, _iram_start, _iram_text_end - _iram_text);
	/* dataflash read enable */
	__raw_writew(DFLRE0_KEY | 0xff, DFLRE0);
	__raw_writew(DFLRE1_KEY | 0xff, DFLRE1);
	return CONFIG_SYS_FLASH_SIZE + 32768;
}
