#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2013-2014, tamanegi (tamanegi@users.sourceforge.jp)
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to
# deal in the Software without restriction, including without limitation the
# rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
# IN THE SOFTWARE.

import wx
from wx import xrc

import sys
import os
import copy
import gettext
import xml.etree.ElementTree as ET
from wmutils import *

DIR_FILE = os.path.dirname( os.path.abspath( __file__ ) )
CONFIG_FILE = os.path.join( DIR_FILE, "wmxmled.conf" )
XRC_FILE = os.path.join( DIR_FILE, "wmxmled.xrc" )

LOCALEDIR = os.path.join( DIR_FILE, 'translations' )
_ = gettext.translation( domain='wmxmled',
                         localedir=LOCALEDIR,
                         fallback=True ).ugettext

"""container for main class of WMXmlEditor, a small XML editor by wxPython."""

class WMXmlEditor( wx.App ):
  ######################################################################
  #
  # Constructor and related functions. Constructor of sub-windows are
  # not here.
  #
  ######################################################################
  """Main class of WMXmlEditor."""
  def OnInit( self ):
    """kind of constructor of WMXmlEditor."""
    self.i18n = self.init_i18n() # internationalization

    self.res = xrc.XmlResource( XRC_FILE )
    self.frameMain = self.res.LoadFrame( None, "mainFrame" )
    self.frameMain.SetTitle( _("WMXMLEditor") )
    self.initializeSubWindows()
    self.initializeVariables()
    self.parseConfigFile()
    self.frameMain.Show()
    self.registerDefaultEvents()
    # parse command line argument if any
    if len( sys.argv ) > 1:
      root, ext = os.path.splitext( sys.argv[1] )
      if ext.lower() == ".xml":
        self.filename = sys.argv[1]
        self.doOpenFile()
      else:
        # three lines
        self.miscShowMessageBox( _("File extension is not .xml. You cannot open it via command line args. Try using menu.") )
    return True

  # gettext initialization for XRC
  def init_i18n( self ):
    '''initialization for i18n'''
    wx.Locale.AddCatalogLookupPathPrefix( LOCALEDIR )
    i18n = wx.Locale( wx.LANGUAGE_DEFAULT, flags=0 ) # use system locale setting
    i18n.AddCatalog( "wmxmled" )
    # created i18n object shall not be discarded
    return i18n

  # create sub-windows
  # Edit, Right-Click-Menu, Find windows
  def initializeSubWindows( self ):
    """initialize sub-windows for Edit, Right-Click-Menu, and Find."""
    menubar = self.frameMain.GetMenuBar()
    search_string = self.i18n.GetString( "&Edit" )
    #self.editMenu = menubar.GetMenu( menubar.FindMenu( "Edit" ) )
    self.editMenu = menubar.GetMenu( menubar.FindMenu( search_string ) )
    self.initializeEditWindow()
    self.initializeRightClickMenu()
    self.dialogFind = self.res.LoadDialog( None, "dialogFindXml" )
    self.initializeFindWindow()
    self.initializeAboutWindow()

  # initialize global variables
  def initializeVariables( self ):
    """initialize global variables."""
    self.filename = ""           # file name editing now
    self.lc_editing_index = -1   # for editing a single element
    self.lc_editing_indexes = [] # for editing multiple elements
    self.partiallyCommonAttribs = [] # for multiple edit

    ### whether current data is saved, this flag should be updated:
    # 1. when data is loaded from a file (-> True)
    # 2. when edit, cut, paste, remove, insert, append items (->False)
    # 3. when data is saved to a file (->True)
    self.IsSaved = True

    # TreeCtrl for XML structure
    self.tc = xrc.XRCCTRL( self.frameMain, "xmlTree" )
    # ListBox for XML text
    self.lc = xrc.XRCCTRL( self.frameMain, "xmlListCtrl" )

    # buffer for copy and paste; ElementTree will be used for this variable
    self.copybuffer = None

    ### correspondence table
    # list -> list (begin and end tags)
    self.l2l_correspondence = []
    # list -> tree
    self.l2t_correspondence = []
    self.initializeImagesForTreeCtrl()
    # undo and redo list
    self.clearUndoList()
    # unique ids for objects
    self.uniqid = 0

  # parse external config file
  def parseConfigFile( self ):
    """read external CONFIG_FILE and set global options."""
    WMXmlEditorParams.readConfigFromFile( CONFIG_FILE )

  ###################################################
  #
  # Global Event handlers and related.
  # Close button in window, menubar events.
  #
  ###################################################

  #################
  # window global #
  #################
  # default events for buttons in window, menubar, ListCtrl, and TreeCtrl
  def registerDefaultEvents( self ):
    """bind default event for buttons in window, menubar, ListCtrl, and TreeCtrl"""
    # window events
    self.frameMain.Bind( wx.EVT_SIZE, self.evWindowSize )
    self.frameMain.Bind( wx.EVT_CLOSE, self.evMenuFileQuit )
    # events invoked by menu item
    self.registerMenuEvents()
    # list control events
    self.registerListCtrlEvents()
    # tree control events
    self.registerTreeCtrlEvents()

  # menubar items
  def registerMenuEvents( self ):
    """sub func of registerDefaultEvents, bind menubar events."""
    self.registerMenuEventsFile()
    self.registerMenuEventsEdit()
    self.registerMenuEventsAbout()

  ### Clear Everything
  def clearAll( self ):
    """remove all the data in TreeCtrl and ListCtrl, and reset all values."""
    # remove all the items of TreeCtrl and ListCtrl
    self.tc.DeleteAllItems()
    self.lc.ClearAll()
    # initialize variables such as filename
    self.initializeVariables()

  def quit( self ):
    """clean memory before quit and the quit program."""
    self.Destroy()
    sys.exit()

  ##################
  # event handlers

  ### resize event; change window size manually
  def evWindowSize( self, event ):
    """window resize event handler."""
    # retrieve sizeritem objects
    sizer = self.tc.GetContainingSizer()
    tc_sizeritem = sizer.GetItem( self.tc )
    lc_sizeritem = sizer.GetItem( self.lc )
    # get current status of TreeCtrl
    tc_size = tc_sizeritem.GetSize()
    tc_size_wo_border = self.tc.GetSize()
    tc_margin = tc_size - tc_size_wo_border
    # get current status of ListCtrl
    lc_size = lc_sizeritem.GetSize()
    lc_size_wo_border = self.lc.GetSize()
    lc_margin = lc_size - lc_size_wo_border
    # calculate width ratio
    ratio = float(tc_size[0]) / float(tc_size[0]+lc_size[0])
    # get current window size
    window_size = self.frameMain.GetClientSize()
    # determine new width
    tc_new_width = int(ratio*window_size[0])
    lc_new_width = window_size[0]-tc_new_width
    # set sizeritemsize
    ## i do not why, i should consider about the margin
    tc_sizeritem_new_size = wx.Size( tc_new_width - tc_margin[0], window_size[1] - tc_margin[1] )
    lc_sizeritem_new_size = wx.Size( lc_new_width - tc_margin[0], window_size[1] - lc_margin[1] )
    tc_sizeritem.SetMinSize( tc_sizeritem_new_size )
    lc_sizeritem.SetMinSize( lc_sizeritem_new_size )
    # new window size
    tc_new_size = wx.Size( tc_new_width - tc_margin[0], window_size[1] - tc_margin[1] )
    lc_new_size = wx.Size( lc_new_width - lc_margin[0], window_size[1] - lc_margin[1] )
    self.tc.SetSize( tc_new_size )
    self.lc.SetSize( lc_new_size )
    sizer.Layout()

  # common keyboard event handler; but this is not a first event handler
  # index is an itemid of ListCtrl
  def keyboardEventHandler( self, index, keycode ):
    """worker for keyboard event."""
    if not self.lc.IsSelected( index ):
      return
    self.lc_editing_index = index
    self.lc_editing_indexes = self.listCtrlGetSelectedIndexes()
    numselected = len( self.lc_editing_indexes )
    ctrl = wx.GetKeyState( wx.WXK_CONTROL )
    if keycode == 3: # Ctrl-C; copy
      self.evPopupMenuCopy( None )
    elif keycode == 22: # Ctrl-V; paste
      # append
      self.pasteItemInBuffer( None, False, False )
    elif keycode == 24: # Ctrl-X; cut
      self.evPopupMenuCut( None )
    elif keycode == wx.WXK_DELETE or keycode == wx.WXK_BACK:
      self.evPopupMenuDelete( None )

  #############
  # File menu #
  #############
  # menubar File menu events initialization
  def registerMenuEventsFile( self ):
    """sub func of registerMenuEvents, bind File menu events to functions."""
    self.Bind( wx.EVT_MENU, self.evMenuFileNew, id=xrc.XRCID("menuFileNew") )
    self.Bind( wx.EVT_MENU, self.evMenuFileOpen, id=xrc.XRCID("menuFileOpen") )
    self.Bind( wx.EVT_MENU, self.evMenuFileClose, id=xrc.XRCID("menuFileClose") )
    self.Bind( wx.EVT_MENU, self.evMenuFileSave, id=xrc.XRCID("menuFileSave") )
    self.Bind( wx.EVT_MENU, self.evMenuFileSaveAs, id=xrc.XRCID("menuFileSaveAs") )
    self.Bind( wx.EVT_MENU, self.evMenuFileQuit, id=xrc.XRCID("menuFileQuit") )

  ##################
  # Event handlers

  # Quit
  def evMenuFileQuit( self, event ):
    """Quit event handler."""
    if self.IsSaved:
      self.quit()
    dialog = wx.MessageDialog( self.frameMain, _("Really quit without saving?"), style=wx.YES_NO|wx.NO_DEFAULT )
    try:
      if dialog.ShowModal() == wx.ID_YES:
        self.quit()
    finally:
      dialog.Destroy()

  def evDropFiles( self, event ):
    """drop file event handler (MS Windows only)."""
    # windows only
    num = event.GetNumberOfFiles()
    if num != 1:
      self.miscShowMessageBox( _("Multiple file read is not supported.") )
      return
    # get filename
    if self.askIfDiscardData():
      self.filename = event.GetFiles[0]
      self.doOpenFile()

  # New
  def evMenuFileNew( self, event = None ):
    """event handler for File menu New."""
    # NOTE: Close is usually more safer method.
    # do nothing if empty
    if self.tc.IsEmpty():
      return
    # if the data is already saved, remove data without inquiry to the user
    if not self.IsSaved:
      # show dialog
      dialog = wx.MessageDialog( self.frameMain, _("Really remove current data?"), style=wx.YES_NO|wx.NO_DEFAULT )
      try:
        if dialog.ShowModal() == wx.ID_NO:
          return
      finally:
        dialog.Destroy()
    self.clearAll()

  # Open
  def evMenuFileOpen( self, event = None ):
    """event handler for File menu Open."""
    if self.askIfDiscardData():
      self.openFile()

  # Close
  def evMenuFileClose( self, event = None ):
    """event handler for File menu Close."""
    # do nothing if filename is not specified; nothing to close
    if len( self.filename ) == 0:
      return
    if not self.IsSaved:
      self.miscShowMessageBox( _("Plase save data before close file. If you want to force to remove data, use \"New\" instead.") )
      return
    self.clearAll()

  # Save
  def evMenuFileSave( self, event = None ):
    """event handler for File menu Save."""
    if len( self.filename ) == 0:
      self.evMenuFileSaveAs()
    else:
      try:
        fh = open( self.filename, "w" )
      except IOError, ( errno, msg ):
        # failed to write data to a file
        if len( self.filename ) == 0:
          self.miscShowMessageBox( _("Please input output filename.") )
        else:
          self.miscShowMessageBox( _("Unable to open file %(file) for writing") % { "file": self.filename } )
        return
      # write to file and close it
      self.writeXml( fh )
      self.IsSaved = True
      fh.close()

  # Save As
  def evMenuFileSaveAs( self, event = None ):
    """event handler for File menu SaveAs."""
    dialog = wx.FileDialog( self.frameMain, _("filename?"), ".", "", "*.xml", wx.FD_SAVE )
    try:
      if dialog.ShowModal() == wx.ID_OK:
        self.filename = dialog.GetFilename()
        fh = open( self.filename, "w" )
      else:
        return
    except IOError, ( errno, msg ):
      # maybe failed to open file
      if len( self.filename ) == 0:
        self.miscShowMessageBox( _("Please input output filename.") )
      else:
        self.miscShowMessageBox( _("Unable to open file %(file) for writing") % { "file": self.filename } )
      return
    finally:
      dialog.Destroy()
    # write to file and close it
    self.writeXml( fh )
    self.IsSaved = True
    fh.close()

  #############
  # Utilities

  ### file operations
  def openFile( self ):
    """file open event handler."""
    dialog = wx.FileDialog( self.frameMain, _("Choose a XML file"), ".", "", "*.xml", wx.FD_OPEN )
    self.filename = ""
    try:
      if dialog.ShowModal() == wx.ID_OK:
        self.filename = dialog.GetPath()
      else:
        return
    finally:
      dialog.Destroy()
    # brief check
    if len( self.filename ) == 0 or not os.path.exists( self.filename ):
      self.miscShowMessageBox( _("Please choose a file") )
      return
    self.doOpenFile()

  def doOpenFile( self ):
    """open specified XML file."""
    ## create TreeCtrl and then ListCtrl
    try:
      xmldata = ET.parse( self.filename )
    except ET.ParseError as v:
      self.miscShowMessageBox( _("Xml parse error. Please verify xml data.") + "\n" +
                               _("Error code: ") + str(v.code) + "\n" +
                               _(" line: ") + str(v.position[0]) +
                               _(" column: ") + str(v.position[1]) )
      self.filename = ""
      return
    self.createXmlTree( ET.parse( self.filename ).getroot() )
    self.createListCtrlFromTreeData()
    self.IsSaved = True
    self.clearUndoList()

  def askIfDiscardData( self ):
    """show dialog for discard current data."""
    if not self.IsSaved:
      if not self.tc.IsEmpty() or len( self.filename ) != 0:
        dialog = wx.MessageDialog( self.frameMain, _("Discard unsaved data?"), style=wx.YES_NO|wx.NO_DEFAULT )
        try:
          if dialog.ShowModal() == wx.ID_NO:
            return False
        finally:
          dialog.Destroy()
    return True

  #############
  # Edit menu #
  #############
  # menubar Edit menu events initialization
  def registerMenuEventsEdit( self ):
    """sub func of registerMenuEvents, bind File menu events to functions."""
    ## the following line will work but the coputational cost is not negligible
    #self.Bind( wx.EVT_UPDATE_UI, self.evMenuEdit, menubar )
    ## the following line does not work ...
    #self.Bind( wx.EVT_MENU, self.resetMenuEdit, id=xrc.XRCID("menuEdit") )
    # undo and redo
    self.Bind( wx.EVT_MENU, self.evMenuEditUndo, id=xrc.XRCID("menuEditUndo") )
    self.Bind( wx.EVT_MENU, self.evMenuEditRedo, id=xrc.XRCID("menuEditRedo") )
    # find
    self.Bind( wx.EVT_MENU, self.evMenuEditFind, id=xrc.XRCID("menuEditFind") )

  ####################
  # Right click menu #
  ####################

  # Right click menu events initialization
  def initializeRightClickMenu( self ):
    """bind Right-Click-Menu events to member functions."""
    # find popup window
    self.rightClickMenu = self.res.LoadMenu( "rightClickMenu" )
    # event handlers
    self.Bind( wx.EVT_MENU, self.evPopupMenuEdit, id=xrc.XRCID("rightClickMenuEdit") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuEditAll, id=xrc.XRCID("rightClickMenuEditAll") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuCopy, id=xrc.XRCID("rightClickMenuCopy") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuCut, id=xrc.XRCID("rightClickMenuCut") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuPasteI, id=xrc.XRCID("rightClickMenuPasteInsert") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuPasteIC, id=xrc.XRCID("rightClickMenuPasteInsertChild") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuPasteA, id=xrc.XRCID("rightClickMenuPasteAppend") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuPasteAC, id=xrc.XRCID("rightClickMenuPasteAppendChild") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuDelete, id=xrc.XRCID("rightClickMenuDelete") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuInsert, id=xrc.XRCID("rightClickMenuInsert") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuInsertChild, id=xrc.XRCID("rightClickMenuInsertChild") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuAppend, id=xrc.XRCID("rightClickMenuAppend") )
    self.Bind( wx.EVT_MENU, self.evPopupMenuAppendChild, id=xrc.XRCID("rightClickMenuAppendChild") )

  ##################
  # event handlers

  # note: popup menu primary uses index of ListCtrl
  def evPopupMenuEdit( self, event ):
    """event handler for popup menu of Edit."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    self.editListCtrlItem( self.lc_editing_index )

  def evPopupMenuEditAll( self, event ):
    """event handler for popup menu of Edit Selected."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    self.editListCtrlItems( self.lc_editing_indexes )

  def evPopupMenuCopy( self, event ):
    """event handler for popup menu of Copy."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    self.copybuffer = []
    self.removeNestedIndexes()
    if len( self.lc_editing_indexes ) > 1: # multiple
      for myindex in self.lc_editing_indexes:
        self.lc_editing_index = myindex
        self.copybuffer.append( self.workerCopy() )
    else: # single copy
      self.copybuffer = [ self.workerCopy() ]

  def evPopupMenuCut( self, event ):
    """event handler for popup menu of Cut."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    # call copy and then remove
    self.evPopupMenuCopy( event )
    self.evPopupMenuDelete( event )

  def evPopupMenuPasteI( self, event ):
    """event handler for popup menu of Paste->Insert."""
    self.pasteItemInBuffer( event, True, False )

  def evPopupMenuPasteA( self, event ):
    """event handler for popup menu of Paste->Append."""
    self.pasteItemInBuffer( event, False, False )

  def evPopupMenuPasteIC( self, event ):
    """event handler for popup menu of Paste->Insert as child."""
    self.pasteItemInBuffer( event, True, True )

  def evPopupMenuPasteAC( self, event ):
    """event handler for popup menu of Paste->Append as child."""
    self.pasteItemInBuffer( event, False, True )

  def evPopupMenuDelete( self, event ):
    """event handler for popup menu Delete."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    if len( self.lc_editing_indexes ) > 1:
      self.deleteItemMultiple()
    else:
      self.deleteItem()

  def evPopupMenuInsert( self, event ):
    """event handler for popup menu Insert."""
    if self.lc_editing_index == 0:
      self.miscShowMessageBox( _("Root element must be only one.") + "\n" +
                               _("If you want to add child element, \"insert child\" or \"append child\" instead.") )
      return
    self.addElementWorker( True )

  def evPopupMenuAppend( self, event ):
    """event handler for popup menu Append."""
    if self.lc_editing_index == 0:
      self.miscShowMessageBox( _("Root element must be only one.") + "\n" +
                               _("If you want to add child element, \"insert child\" or \"append child\" instead.") )
      return
    self.addElementWorker( False )

  def evPopupMenuInsertChild( self, event ):
    """event handler for popup menu Add->Insert as child."""
    if self.lc_editing_index < 0:
      self.miscShowMessageBox( _("No element selected.") )
      return
    self.addChildElementWorker( True )

  def evPopupMenuAppendChild( self, event ):
    """event handler for popup menu Add->Append as child."""
    if self.lc_editing_index < 0:
      self.miscShowMessageBox( _("No element selected.") )
      return
    self.addChildElementWorker( False )

  #############
  # utilities

  # index (self.lc_editing_index) must be > 0 and
  # ListCtrl must not be empty
  def popupIsItemAvailable( self, index ):
    """returns whether given index is a valid one."""
    if self.lc_editing_index < 0:
      return False
    if self.lc.GetItemCount() <= 0:
      return False
    return True

  def addChildElementWorker( self, insert = True ):
    """worker for adding child element."""
    self.dialogEdit.SetTitle( _("Add new child element") )
    treeitemid, index = self.createElementByEditWindow()
    # if "Done" is clicked and accepted, insert element to the tree and
    # update ListCtrl
    if self.flag_editted:
      newxml = copy.deepcopy( self.editing_xml )
      showname = self.createTreeCtrlLabel( newxml )
      if insert:
        my_itemid = self.insertChildElement( showname, treeitemid )
      else: # append
        my_itemid = self.appendChildElement( showname, treeitemid )
      self.tc.SetItemPyData( my_itemid, [ self.tc.GetCount() - 1, newxml, self.uniqid ] )
      self.uniqid += 1
      self.createListCtrlFromTreeData()
      # register undo
      self.addActionToUndoQueue( ActionAdd( self.gatherUniqIds( my_itemid ), 0, my_itemid, newxml ) )
      # work up
      self.IsSaved = False

  def insertChildElement( self, tag, itemid ):
    """insert child element to the tree."""
    return self.tc.PrependItem( itemid, tag, self.imageFolder, self.imageFolderOpen )

  def appendChildElement( self, tag, itemid ):
    """append child element to the tree."""
    return self.tc.AppendItem( itemid, tag, self.imageFolder, self.imageFolderOpen )

  def setPopupMenuState( self, activate = True ):
    """set popup menu availability."""
    # just to shorten; dirty code
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuEdit")).Enable( activate )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuEditAll")).Enable( activate )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuCopy")).Enable( activate )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuCut")).Enable( activate )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuPaste")).Enable( activate )
    if self.tc.IsEmpty() and not self.copybuffer is None:
      self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuPaste")).Enable( True )
    # deactivate if no there is no data to paste
    if self.copybuffer is None:
      self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuPaste")).Enable( False )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuDelete")).Enable( activate )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuInsert")).Enable( True )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuInsertChild")).Enable( True )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuAppend")).Enable( True )
    self.rightClickMenu.FindItemById(xrc.XRCID("rightClickMenuAppendChild")).Enable( True )

  def workerCopy( self ):
    """worker for copy event."""
    myindex = min( self.lc_editing_index, self.l2l_correspondence[self.lc_editing_index] )
    itemid = self.l2t_correspondence[myindex]
    # uniqids will be discarded
    buf, uniqids = self.createElementTreeFromTreeCtrl( itemid )
    return buf.getroot()

  def pasteItemInBuffer( self, event, insert = True, child = True ):
    """paste event worker."""
    # brief check
    if self.copybuffer is None:
      self.miscShowMessageBox( _("no data in buffer") )
      return
    if len( self.copybuffer ) == 0:
      self.miscShowMessageBox( _("no data in buffer") )
      return
    # prepare TreeCtrl labels
    newxmls = copy.deepcopy( self.copybuffer )
    names = []
    for newxml in newxmls:
      names.append( self.createTreeCtrlLabel( newxml ) )
    numadd = len( names )
    # get ListCtrl index
    if len( self.l2l_correspondence ) > 0:
      index = min( self.lc_editing_index, self.l2l_correspondence[self.lc_editing_index] )
    else:
      index = -1

    if insert:
      names.reverse()
      newxmls.reverse()

    # add xml in copybuffer
    my_itemids = []
    if index >= 0:
      itemid = self.l2t_correspondence[index]
    if index < 0 and child: # i do not know what to do
      self.miscShowMessageBox( _("cannot create child element.") )
      return
    elif index < 0 and self.tc.IsEmpty(): # add root element
      if numadd > 1:
        self.miscShowMessageBox( _("cannot add multiple root element.") )
        return
      my_itemids.append( self.tc.AddRoot( names[0], self.imageFolder, self.imageFolderOpen ) )
    else:
      if insert and child: # insert child
        for name in names:
          my_itemids.append( self.insertChildElement( name, itemid ) )
      elif insert and not child: # insert
        if index == 0:
          self.miscShowMessageBox( _("Cannot insert another root element.") )
          return
        else:
          my_itemids.append( self.insertElement( names[0], itemid ) )
          for i in range( 1, numadd ):
            my_itemids.append( self.insertElement( names[i], my_itemids[-1] ) )
      elif not insert and child: # append child
        for name in names:
          my_itemids.append( self.appendChildElement( names[i], itemid ) )
      elif not insert and not child: # append
        if index == 0:
          self.miscShowMessageBox( _("Cannot insert another root element.") )
          return
        else:
          my_itemids.append( self.appendElement( names[0], itemid ) )
          for i in range( 1, numadd ):
            my_itemids.append( self.appendElement( names[i], my_itemids[-1] ) )
    # convert
    myxmls = []
    for i in range( 0, numadd ):
      myxmls.append( WMXmlElement( newxmls[i] ) )
      # use new uniq id
      self.tc.SetItemPyData( my_itemids[i], [ self.tc.GetCount() - 1, myxmls[-1], self.uniqid ] )
      self.uniqid += 1
      self.createXmlTreeRecursive( self.copybuffer[i], my_itemids[i] )
    self.createListCtrlFromTreeData()
    
    if numadd == 1:
      # add to the undo queue
      self.addActionToUndoQueue( ActionAdd( self.gatherUniqIds( my_itemids[0] ), 0, my_itemids[0], myxmls[0] ) )
    else:
      uniqids = []
      for i in range( 0, numadd ):
        uniqids.append( self.gatherUniqIds( my_itemids[i] ) )
      self.addActionToUndoQueue( ActionAdds( uniqids, 0, my_itemids, myxmls ) )
    self.IsSaved = False
    # focus on newly created item
    self.focusTreeCtrlItem( my_itemids[-1] )

  # preprocess for deleting item
  def preRemove( self ):
    """preparation for deleting item."""
    if not self.popupIsItemAvailable( self.lc_editing_index ):
      return
    myindex = self.lc_editing_index
    itemid = self.l2t_correspondence[myindex]
    # remember item position for undoing
    pos = self.calculateItemPosition( itemid )
    if not self.tc.GetNextSibling( itemid ):
      pos = -1
    ## we need Element, not WMXmlElement
    myxml, uniqids = self.createElementTreeFromTreeCtrl( itemid )
    myxml = myxml.getroot()
    parent = self.tc.GetItemParent( itemid )
    if parent.IsOk():
      uniqids.insert( 0, self.tc.GetItemPyData(parent)[2] )
    else:
      uniqids.insert( 0, None )
    return uniqids, pos, itemid, myxml

  def deleteItem( self ):
    """delete an item specified by lc_editing_index."""
    uniqids, pos, itemid, myxml = self.preRemove()
    # find next item
    nextitem = self.tc.GetNextSibling(itemid)
    parent = self.tc.GetItemParent(itemid)
    # remove item
    if self.deleteTreeCtrlItem( itemid ):
      # add to undo queue
      self.addActionToUndoQueue( ActionDel( uniqids, pos, None, myxml ) )
      # store removed data for undo
      # reset flag
      self.IsSaved = False
      # force update data
      self.createListCtrlFromTreeData()
      if nextitem.IsOk():
        self.focusTreeCtrlItem( nextitem )
      elif parent.IsOk():
        self.focusTreeCtrlItem( parent )

  def deleteItemMultiple( self ):
    """delete items specified by lc_editing_indexes."""
    # remove nested indexes
    self.removeNestedIndexes()
    # sort indexes
    self.lc_editing_indexes.sort()
    self.lc_editing_indexes.reverse()
    # use self.lc_editing_indexes to delete selected items
    uniqidss = []
    poss = []
    myxmls = []
    for index in self.lc_editing_indexes:
      self.lc_editing_index = index
      uniqids, pos, itemid, myxml = self.preRemove()
      if self.deleteTreeCtrlItem( itemid ):
        uniqidss.append( uniqids )
        poss.append( pos )
        myxmls.append( myxml )
    self.addActionToUndoQueue( ActionDels( uniqidss, poss, None, myxmls ) )
    self.IsSaved = False
    self.createListCtrlFromTreeData()
    # do not focus next item here

  # remove nested elements from lc_editing_indexes
  def removeNestedIndexes( self ):
    """remove nested index from lc_editing_indexes."""
    num = len(self.lc_editing_indexes)
    itemids = []
    for i in range( 0, num ):
      ind = self.lc_editing_indexes[i]
      ind = min( ind, self.l2l_correspondence[ind] )
      itemids.append( self.l2t_correspondence[ind] )
      self.lc_editing_indexes[i] = ind

    newone = []
    for i in range( 0, num ):
      if not self.isNestedItem( itemids[i] ):
        newone.append( self.lc_editing_indexes[i] )

    # replace to new data
    self.lc_editing_indexes = list(set(newone))

  def isNestedItem( self, itemid ):
    """returns whether itemid is nested item."""
    parent = self.tc.GetItemParent( itemid )
    while parent.IsOk():
      ind = self.tc.GetItemPyData( parent )[0]
      if ind in self.lc_editing_indexes:
        return True
      parent = self.tc.GetItemParent( parent )
    return False

  def addElementWorker( self, insert = True ):
    """worker for adding new element to the tree."""
    self.dialogEdit.SetTitle( _("Add new element") )
    treeitemid, index = self.createElementByEditWindow()
    # if "Done" is clicked and accepted, insert element to the tree and
    # update ListCtrl
    if self.flag_editted:
      newxml = copy.deepcopy( self.editing_xml )
      showname = self.createTreeCtrlLabel( newxml )
      if index < 0: # add new element root
        my_itemid = self.tc.AddRoot( showname, self.imageFolder, self.imageFolderOpen )
      else:
        if insert:
          my_itemid = self.insertElement( showname, treeitemid )
        else: # append
          my_itemid = self.appendElement( showname, treeitemid )
      self.tc.SetItemPyData( my_itemid, [ self.tc.GetCount() - 1, newxml, self.uniqid ] )
      self.uniqid += 1
      self.createListCtrlFromTreeData()
      # add to undo queue
      self.addActionToUndoQueue( ActionAdd( self.gatherUniqIds( my_itemid ), 0, my_itemid, newxml ) )
      # work-up
      self.IsSaved = False

  def insertElement( self, tag, itemid ):
    """insert element just before itemid."""
    itempos = self.calculateItemPosition( itemid )
    return self.tc.InsertItemBefore( self.tc.GetItemParent( itemid ), itempos, tag, self.imageFolder, self.imageFolderOpen )

  def appendElement( self, tag, itemid ):
    """append element just after itemid."""
    return self.tc.InsertItem( self.tc.GetItemParent( itemid ), itemid, tag, self.imageFolder, self.imageFolderOpen )

  ###################################################
  #
  # Edit sub-window
  #
  ###################################################
  # Edit window preparation
  def initializeEditWindow( self ):
    """Edit window constructor."""
    self.dialogEdit = self.res.LoadDialog( None, "dialogEditXml" )
    ## do not register here, events are activated when Edit window will be shown
    #self.registerEditWindowEvents()

  def registerEditWindowEvents( self ):
    """Bind Edit window events to class member functions."""
    # text boxes
    self.Bind( wx.EVT_TEXT, self.evEditTextElementName, id=xrc.XRCID("textCtrlElementName") )
    self.Bind( wx.EVT_TEXT, self.evEditTextInnerText, id=xrc.XRCID("textCtrlXMLInner") )
    self.Bind( wx.EVT_TEXT, self.evEditTextAttrValue, id=xrc.XRCID("textCtrlAttrValue") )
    # choice
    self.Bind( wx.EVT_CHOICE, self.evEditChooseAttr, id=xrc.XRCID("choiceAttr") )
    # buttons
    self.Bind( wx.EVT_BUTTON, self.evEditRemoveButton, id=xrc.XRCID("buttonRemoveEdit") )
    self.Bind( wx.EVT_BUTTON, self.evEditAddNewAttrButton, id=xrc.XRCID("buttonAddNewAttr") )
    self.Bind( wx.EVT_BUTTON, self.evEditDoneButton, id=xrc.XRCID("buttonDoneEdit") )
    self.Bind( wx.EVT_BUTTON, self.evEditCancelButton, id=xrc.XRCID("buttonCancelEdit") )

  # unbind Edit window events
  def unregisterEditWindowEvents( self ):
    """unbind Edit window events."""
    self.Unbind( wx.EVT_TEXT, id=xrc.XRCID("textCtrlElementName") )
    self.Unbind( wx.EVT_TEXT, id=xrc.XRCID("textCtrlXMLInner") )
    self.Unbind( wx.EVT_TEXT, id=xrc.XRCID("textCtrlAttrValue") )
    # choice
    self.Unbind( wx.EVT_CHOICE, id=xrc.XRCID("choiceAttr") )
    # buttons
    self.Unbind( wx.EVT_BUTTON, id=xrc.XRCID("buttonRemoveEdit") )
    self.Unbind( wx.EVT_BUTTON, id=xrc.XRCID("buttonAddNewAttr") )
    self.Unbind( wx.EVT_BUTTON, id=xrc.XRCID("buttonDoneEdit") )
    self.Unbind( wx.EVT_BUTTON, id=xrc.XRCID("buttonCancelEdit") )

  ##############################
  # Edit window event handlers #
  ##############################

  def evMenuEditUndo( self, event ):
    """event handler for undo (menu or shortcut)."""
    if len( self.undo ) <= 0: # this would not happen...
      return
    if self.undo[0].undo( self.tc, self ):
      # pop the action from undo and insert to redo
      action = self.undo.pop(0)
      self.addActionToRedoQueue( action.reverse() )
      self.createListCtrlFromTreeData()

  def evMenuEditRedo( self, event ):
    """event handler for redo (menu or shortcut)."""
    if len( self.redo ) <= 0: # this would not happen...
      return
    self.redo[0].undo( self.tc, self )
    # pop the action from redo and inser back to undo
    action = self.redo.pop(0)
    self.addActionToUndoQueue( action.reverse() )
    self.createListCtrlFromTreeData()

  def evMenuEditFind( self, event ):
    """event handler for Find."""
    self.dialogFind.Show()
    self.dialogFind.Raise()

  def evEditRemoveButton( self, event ):
    """event handler for Remove button in Edit window."""
    choice = xrc.XRCCTRL( self.dialogEdit, "choiceAttr" )
    # fail-safe
    if choice.GetCount() == 0:
      return
    removing_attr = choice.GetString( choice.GetSelection() )
    self.editing_xml.attrib.pop( removing_attr )
    if removing_attr in self.partiallyCommonAttribs:
      self.partiallyCommonAttribs.remove( removing_attr )
    self.rebuildChoice()

  def evEditAddNewAttrButton( self, event ):
    """event handler for Add New Attribute button in Edit window."""
    # attribute name
    tctl_new_attr_name = xrc.XRCCTRL( self.dialogEdit, "textCtrlNewAttrName" )
    new_attr_name = tctl_new_attr_name.GetValue().strip()
    # attribute value
    tctl_new_attr_val = xrc.XRCCTRL( self.dialogEdit, "textCtrlNewAttrVal" )
    new_attr_val = tctl_new_attr_val.GetValue().strip()
    if len( new_attr_name ) == 0:
      self.miscShowMessageBox( _("Please input attribute name.") )
      return
    if self.editing_xml.attrib.has_key( new_attr_name ):
      dialog = wx.MessageDialog( self.dialogEdit, _("overwrite existing value?"), style=wx.YES_NO|wx.NO_DEFAULT )
      try:
        if dialog.ShowModal() == wx.ID_NO:
          return
      finally:
        dialog.Destroy()
    self.editing_xml.attrib[ new_attr_name ] = new_attr_val
    self.rebuildChoice()
    # clear inputted value
    tctl_new_attr_name.ChangeValue("")
    tctl_new_attr_val.ChangeValue("")

  def evEditDoneButton( self, event ):
    """event handler for Done button in Edit window."""
    # reject if element name is empty
    if len(self.editing_xml.tag.strip()) == 0:
      self.miscShowMessageBox( _("Empty element name is not acceptable.") )
      return
    # copy edtitted data to the kept data
    if not self.keep_xml is None:
      if isinstance( self.keep_xml, list ): # multiple edit
        # skip if data is only "*"
        if self.editing_xml.tag.strip() != "*":
          for xml in self.keep_xml:
            xml.tag = self.editing_xml.tag.strip()
        # complicated...
        for attr, val in self.editing_xml.attrib.items():
          if val != "*":
            for xml in self.keep_xml:
              if self.isNotCommonAttrib( attr ) and not xml.attrib.has_key(attr):
                continue
              xml.attrib[attr] = val
        # check deleted attributes
        for xml in self.keep_xml:
          for key in xml.attrib.keys():
            if not self.editing_xml.attrib.has_key( key ):
              xml.attrib.pop( key )
        # this might be potentially problematic
        if not self.editing_xml.text is None:
          if self.editing_xml.text.strip() != "*":
            for xml in self.keep_xml:
              xml.text = self.editing_xml.text
      else: # standard
        self.keep_xml.tag = self.editing_xml.tag.strip()
        self.keep_xml.text = self.editing_xml.text
        self.keep_xml.tail = self.editing_xml.tail
        self.keep_xml.attrib = self.editing_xml.attrib
    self.dialogEdit.Hide()
    # mark editted
    self.IsSaved = False
    self.flag_editted = True

  def evEditCancelButton( self, event ):
    """event handler for Cancel button in Edit window."""
    self.dialogEdit.Hide()

  def evEditChooseAttr( self, event ):
    """event handler for choosing attribute in Edit window."""
    choice = xrc.XRCCTRL( self.dialogEdit, "choiceAttr" )
    tctl = xrc.XRCCTRL( self.dialogEdit, "textCtrlAttrValue" )
    n = event.GetSelection()
    choice_str = choice.GetString(n)
    self.highlightChoiceItem( choice, choice_str )
    tctl.ChangeValue( self.editing_xml.attrib[choice_str] )

  def evEditTextElementName( self, event ):
    """event handler for editing XML element name in Edit window."""
    tctl = xrc.XRCCTRL( self.dialogEdit, "textCtrlElementName" )
    self.editing_xml.tag = tctl.GetValue().strip()

  def evEditTextInnerText( self, event ):
    """event handler for editing XML inner text in Edit window."""
    tctl = xrc.XRCCTRL( self.dialogEdit, "textCtrlXMLInner" )
    self.editing_xml.text = tctl.GetValue()

  def evEditTextAttrValue( self, event ):
    """event handler for editing XML attribute value in Edit window."""
    choice = xrc.XRCCTRL( self.dialogEdit, "choiceAttr" )
    tctl = xrc.XRCCTRL( self.dialogEdit, "textCtrlAttrValue" )
    self.editing_xml.attrib[choice.GetString(choice.GetCurrentSelection())] = tctl.GetValue().strip()

  #########################
  # Edit window utilities #
  #########################

  def createElementByEditWindow( self ):
    """create new element and returns its itemid and ListCtrl index."""
    self.flag_editted = False
    if self.lc_editing_index >= 0:
      index = min( self.lc_editing_index, self.l2l_correspondence[self.lc_editing_index] )
      itemid = self.l2t_correspondence[index]
    else: # add new root element
      index = -1
      itemid = None
    # flag
    # create new myxml element and edit it
    self.editing_xml = WMXmlElement()
    self.keep_xml = None
    self.invokeEditWindow()
    return itemid, index

  def isNotCommonAttrib( self, attr ):
    """return whether given attribute is a common attribute."""
    return ( attr in self.partiallyCommonAttribs )

  # myxmls and itemids are list[WMXmlElement] and list[TreeItemId], respectively
  def showEditWindow( self, myxml, itemid ):
    """show Edit window for XML data of itemid."""
    # flag whether the element is editted
    # are there any simple ways to handle it?
    self.flag_editted = False
    self.partiallyCommonAttribs = []
    # remember id of the editing xml, though unused currently
    self.editing_itemid = itemid
    ### extended for multiple element edit
    if isinstance(myxml,list):
      self.editing_xml = self.generateXmlDataForEdit( myxml )
      self.dialogEdit.SetTitle( _("Edit elements") )
    else:
      self.editing_xml = copy.deepcopy(myxml)
      self.dialogEdit.SetTitle( _("Edit element") )
    self.keep_xml = myxml
    orig_data = copy.deepcopy( myxml )
    # just do it
    self.invokeEditWindow()
    if self.flag_editted:
      if isinstance(myxml,list):
        self.addActionToUndoQueue( ActionEdits( itemid, orig_data ), True )
      else:
        self.addActionToUndoQueue( ActionEdit( itemid, orig_data ), True )
    # reset flag once more
    self.partiallyCommonAttribs = []

  def generateXmlDataForEdit( self, myxmls ):
    """generate XML data for editing of myxmls."""
    ret = WMXmlElement()
    tags = []
    attrs = {}
    self.partiallyCommonAttribs = []
    texts = []
    count = len( myxmls )
    for xml in myxmls:
      # element name
      tags.append( xml.tag )
      # attributes
      for attr, val in xml.attrib.items():
        if attrs.has_key( attr ):
          attrs[attr].append( val )
        else:
          attrs[attr] = [ val ]
      # inner text
      texts.append( xml.text )
    ### check uniqueness
    # tag
    if len(set(tags)) > 1:
      ret.tag = "*"
    else: # shall be 1
      ret.tag = tags[0]
    # attrib
    for attr, vals in attrs.items():
      if len(set(vals)) > 1:
        ret.attrib[attr] = "*"
      else:
        ret.attrib[attr] = vals[0]
      if len(vals) != count:
        self.partiallyCommonAttribs.append( attr )
    # text
    if len(set(texts)) > 1:
      ret.text = "*"
    else:
      ret.text = texts[0]
    return ret

  # prepare items in the Edit Window and show it modal
  def invokeEditWindow( self ):
    """preactions for Edit window."""
    self.prepareEditWindowItems()
    self.registerEditWindowEvents()
    self.dialogEdit.ShowModal()
    self.unregisterEditWindowEvents()

  def prepareEditWindowItems( self ):
    """preparation for Edit window."""
    ### gather objects
    # element name
    tctl_elem = xrc.XRCCTRL( self.dialogEdit, "textCtrlElementName" )
    # inner text
    tctl_inner = xrc.XRCCTRL( self.dialogEdit, "textCtrlXMLInner" )
    # attributes
    choice = xrc.XRCCTRL( self.dialogEdit, "choiceAttr" )
    # new attribute
    tctl_new_attr_name = xrc.XRCCTRL( self.dialogEdit, "textCtrlNewAttrName" )
    tctl_new_attr_val = xrc.XRCCTRL( self.dialogEdit, "textCtrlNewAttrVal" )
    ### initialize objects
    # element name
    # this may not happen, though
    if self.editing_xml.tag is None:
      # NOTE: use ChangeValue instead of SetValue not to dispatch EVT_TEXT
      tctl_elem.ChangeValue( "" )
    else:
      tctl_elem.ChangeValue( self.editing_xml.tag )
    # inner text
    if self.editing_xml.text is None:
      tctl_inner.ChangeValue( "" )
    else:
      tctl_inner.ChangeValue( self.editing_xml.text )
    # choice
    self.rebuildChoice()
    # new attribute
    tctl_new_attr_name.ChangeValue( "" )
    tctl_new_attr_val.ChangeValue( "" )

  def rebuildChoice( self ):
    """rebuild Choice items in Edit window."""
    choice = xrc.XRCCTRL( self.dialogEdit, "choiceAttr" )
    choice.Clear()
    tctl_attr = xrc.XRCCTRL( self.dialogEdit, "textCtrlAttrValue" )
    but_remove = xrc.XRCCTRL( self.dialogEdit, "buttonRemoveEdit" )
    for key in self.editing_xml.attrib.keys():
      choice.Append( key )
    if choice.GetCount() != 0:
      choice.SetSelection(0)
      choice_str = choice.GetString(choice.GetSelection())
      self.highlightChoiceItem( choice, choice_str )
      tctl_attr.ChangeValue( self.editing_xml.attrib[choice_str] )
      choice.Enable( True )
      tctl_attr.Enable( True )
      but_remove.Enable( True )
    else:
      tctl_attr.ChangeValue( "" )
      # if no attribute is available, disable TextCtrl and Choice
      choice.Enable( False )
      tctl_attr.Enable( False )
      but_remove.Enable( False )

  def highlightChoiceItem( self, choice, string ):
    """highlight choice item if choosed one is a common attribute."""
    # highlight text if not a common attribute
    if string in self.partiallyCommonAttribs:
      choice.SetForegroundColour( WMXmlEditorParams.LC_emphasis_color )
    else:
      choice.SetForegroundColour( wx.NullColour )

  def resetMenuEdit( self, event = None ):
    """event handler for Edit in Edit menu."""
    self.editMenu.FindItemById( xrc.XRCID( "menuEditUndo" ) ).Enable( len( self.undo ) > 0 )
    self.editMenu.FindItemById( xrc.XRCID( "menuEditRedo" ) ).Enable( len( self.redo ) > 0 )

  def editListCtrlItem( self, index ):
    """edit a ListCtrl item."""
    myindex = min(index,self.l2l_correspondence[index])
    # search object corresponding to begin tag
    treeitemid = self.l2t_correspondence[myindex]
    myxml = self.tc.GetItemPyData( treeitemid )[1]
    # remember tag name of the element
    savetag = myxml.tag
    need_end_tag_save = self.isEndTagRequired( myxml, treeitemid )
    # XML data is obtained, show (modal) data to a Panel
    self.showEditWindow( myxml, treeitemid )
    # is needed to update text label in TreeCtrl and ListCtrl
    need_end_tag = self.isEndTagRequired( myxml, treeitemid )
    ### Check whether ListCtrl items are updated
    # 1: update if existing end tag is removed or end tag is newly added
    # 2: update if element name is modified and end tag is required
    if need_end_tag_save ^ need_end_tag or ( savetag != myxml.tag and need_end_tag ):
      # rebuild ListCtrl
      self.createListCtrlFromTreeData()
    else:
      self.lc.SetItemText( myindex, self.myxml2string( myxml, need_end_tag, self.getIndentItem( treeitemid ) ) )
    # update TreeCtrl label
    self.tc.SetItemText( treeitemid, self.createTreeCtrlLabel( myxml ) )

  # multiple version of editListCtrlItem above
  def editListCtrlItems( self, indexes ):
    """edit multiple ListCtrl items."""
    # unique index list (original list may have duplication)
    myindexes = []
    for index in indexes:
      myindexes.append( min(index,self.l2l_correspondence[index]) )
    myindexes = list(set(myindexes))
    # retrieve itemids
    itemids = []
    for index in myindexes:
      itemids.append( self.l2t_correspondence[index] )
    # remember tag names
    myxmls = []
    savetags = []
    for itemid in itemids:
      myxmls.append( self.tc.GetItemPyData( itemid )[1] )
      savetags.append( myxmls[-1].tag )
    # begin editing
    self.showEditWindow( myxmls, itemids )
    # force update list control label
    self.createListCtrlFromTreeData()
    # update tree control items
    for itemid in itemids:
      self.tc.SetItemText( itemid, self.createTreeCtrlLabel( self.tc.GetItemPyData( itemid )[1] ) )

  ###################################################
  #
  # Find sub-window
  #
  ###################################################
  # Find window initialization
  def initializeFindWindow( self ):
    """bind Find window events to member functions."""
    self.Bind( wx.EVT_BUTTON, self.evFindSearchButton, id=xrc.XRCID("buttonSearchFind") )
    self.Bind( wx.EVT_BUTTON, self.evFindCloseButton, id=xrc.XRCID("buttonCloseFind") )
    self.Bind( wx.EVT_BUTTON, self.evFindClearButton, id=xrc.XRCID("buttonClearFind") )

  ##################
  # event handlers #
  ##################

  def evFindSearchButton( self, event ):
    """event handler for Search button in Find window."""
    # if the items are empty, do nothing
    if self.verifyFindItems():
      return
    # check whether this is refine search
    cbox = xrc.XRCCTRL( self.dialogFind, "checkRefineFind" )
    refine_search = cbox.GetValue()
    # deselect all items if not refine search
    if not refine_search:
      self.tc.UnselectAll()
      self.deselectAllItemsListCtrl()
    # build query
    query = self.genXmlQuery()
    matched = [] # terrible waste ... orz
    last_selected = -1
    # search over TreeCtrl item (wmxml info belongs to TreeCtrlItem)
    for itemid in self.allItemIdGenerator():
      itemdata = self.tc.GetItemPyData( itemid )
      # if in refine search and item is not selected neither in ListCtrl nor
      # TreeCtrl, skip this item
      if refine_search and not ( self.tc.IsSelected(itemid) or self.lc.IsSelected(itemdata[0]) ):
        continue
      if query.match( itemdata[1] ):
        # select it
        ####self.tc.SelectItem( itemid )
        self.selectListCtrlItem( itemdata[0] )
        matched.append( itemid )
        last_selected = itemdata[0]
      elif refine_search:
        # deselect it
        ####self.tc.UnselectItem( itemid )
        self.selectListCtrlItem( itemdata[0], 0 )
    if last_selected >= 0:
      self.lc.Focus( last_selected )
    ### selection state of items in TreeCtrl should be done here in a lump
    ### at least in wx@Linux-GTK, selecting items in TreeCtrl and ListCtrl
    ### alternatively does not work fine for TreeCtrl items.
    ### I do not know why. What's this?
    # deselect all
    self.tc.UnselectAll()
    for itemid in matched:
      self.tc.SelectItem( itemid )

  def evFindCloseButton( self, event ):
    """event handler for Close button in Find window."""
    self.dialogFind.Hide()

  def evFindClearButton( self, event ):
    """event handler for Clear button in Find window."""
    if WMXmlEditorParams.FIND_clear_wo_query:
      self.clearFindWindowItems()
    else:
      dialog = wx.MessageDialog( self.dialogFind, _("Really remove query?"), style=wx.YES_NO|wx.NO_DEFAULT )
      try:
        if dialog.ShowModal() == wx.ID_YES:
          self.clearFindWindowItems()
      finally:
        dialog.Destroy()

  #############
  # utilities #
  #############

  def genXmlQuery( self ):
    """generate XML query using items in Find window."""
    ret = WMXmlQuery()
    ret.setElementQuery( xrc.XRCCTRL( self.dialogFind, "textCtrlElementFind" ).GetValue(),
                         xrc.XRCCTRL( self.dialogFind, "checkICTagFind" ).GetValue() )
    ret.setAttrQuery( xrc.XRCCTRL( self.dialogFind, "textCtrlAttrNameFind" ).GetValue(),
                      xrc.XRCCTRL( self.dialogFind, "checkICAttrNameFind" ).GetValue(),
                      xrc.XRCCTRL( self.dialogFind, "textCtrlAttrValFind" ).GetValue(),
                      xrc.XRCCTRL( self.dialogFind, "checkICAttrValFind" ).GetValue() )
    ret.setInnerTextQuery( xrc.XRCCTRL( self.dialogFind, "textCtrlInnerTextFind" ).GetValue(),
                           xrc.XRCCTRL( self.dialogFind, "checkICInnerTextFind" ).GetValue() )
    return ret

  def verifyFindItems( self ):
    """returns whether current query is not empty."""
    if not self.isNullXRCObjectValue( self.dialogFind, "textCtrlElementFind" ):
      return False
    if not self.isNullXRCObjectValue( self.dialogFind, "textCtrlAttrNameFind" ):
      return False
    if not self.isNullXRCObjectValue( self.dialogFind, "textCtrlAttrValFind" ):
      return False
    if not self.isNullXRCObjectValue( self.dialogFind, "textCtrlInnerTextFind" ):
      return False
    return True

  def isNullXRCObjectValue( self, parent, name ):
    """returns whether value of given XRC object is empty."""
    tctl = xrc.XRCCTRL( parent, name )
    if len( tctl.GetValue().strip() ) > 0:
      return False
    return True

  def clearFindWindowItems( self ):
    """clear query items in Find window."""
    # reset checkbox
    for name in [ "checkRefineFind", "checkICTagFind", "checkICAttrNameFind",
                  "checkICAttrValFind", "checkICInnerTextFind" ]:
      cbox = xrc.XRCCTRL( self.dialogFind, name )
      cbox.SetValue( False )
    # clear values in TextCtrls
    for name in [ "textCtrlElementFind", "textCtrlAttrNameFind",
                  "textCtrlAttrValFind", "textCtrlInnerTextFind" ]:
      tctl = xrc.XRCCTRL( self.dialogFind, name )
      tctl.SetValue( "" )

  ###################################################
  #
  # TreeCtrl related
  #
  ###################################################
  # prepare images used for TreeCtrl
  def initializeImagesForTreeCtrl( self ):
    """prepare images used for TreeCtrl items."""
    # images for TreeCtrl
    imageList = wx.ImageList(16,16)
    self.imageFolder = imageList.Add( wx.ArtProvider.GetBitmap( wx.ART_FOLDER, wx.ART_OTHER, (16,16) ) )
    self.imageFolderOpen = imageList.Add( wx.ArtProvider.GetBitmap( wx.ART_FOLDER_OPEN, wx.ART_OTHER, (16,16) ) )
    self.imageNew = imageList.Add( wx.ArtProvider.GetBitmap( wx.ART_NEW, wx.ART_OTHER, (16,16) ) )
    self.tc.AssignImageList( imageList )

  # TreeCtrl (left pane) events definition
  def registerTreeCtrlEvents( self ):
    """sub func of registerDefaultEvents, bind TreeCtrl events."""
    self.tc.Bind( wx.EVT_TREE_ITEM_ACTIVATED, self.evTreeCtrlActivate )
    ## remove since this is annoying
    #self.tc.Bind( wx.EVT_TREE_ITEM_EXPANDED, self.evTreeCtrlActivate )
    ## this event is not dispatched when TreeCtrl is empty...
    self.tc.Bind( wx.EVT_TREE_ITEM_RIGHT_CLICK, self.evTreeCtrlRightClick )
    self.tc.Bind( wx.EVT_RIGHT_DOWN, self.evTreeCtrlRightClickSub )
    # keyboard event
    self.tc.Bind( wx.EVT_TREE_KEY_DOWN, self.evTreeCtrlKeyDown )
    # drop files (same as DROP_FILES event of ListCtrl )
    self.tc.Bind( wx.EVT_DROP_FILES, self.evDropFiles )

  ### create tree control
  # generate XML tree structure
  def createXmlTree( self, xmlroot ):
    """create XML tree (TreeCtrl) using given xmlroot."""
    # initialize tree control
    self.tc.DeleteAllItems()
    self.uniqid = 0
    showname = self.createTreeCtrlLabel( xmlroot )
    root_itemid = self.tc.AddRoot( showname, self.imageFolder, self.imageFolderOpen )
    num = self.tc.GetCount() - 1
    # first element of the data is tentative value
    self.tc.SetItemPyData( root_itemid, [ num, WMXmlElement( xmlroot ), self.uniqid ] )
    self.uniqid += 1
    self.createXmlTreeRecursive( xmlroot, root_itemid )

  def createXmlTreeRecursive( self, element, itemid ):
    """create XML tree recursively."""
    for child in element:
      showname = self.createTreeCtrlLabel( child )
      child_itemid = self.tc.AppendItem( itemid, showname, self.imageFolder, self.imageFolderOpen )
      num = self.tc.GetCount() - 1
      self.tc.SetItemPyData( child_itemid, [ num, WMXmlElement( child ), self.uniqid ] )
      self.uniqid += 1
      self.createXmlTreeRecursive( child, child_itemid )

  ##################
  # event handlers #
  ##################

  # open tree or double click
  def evTreeCtrlActivate( self, event ):
    """event handler for TreeCtrl item double click."""
    self.deselectAllItemsListCtrl()
    itemid = event.GetItem()
    # corresponding item in ListCtrl
    myitem = self.tc.GetItemPyData( itemid )[0]
    self.selectListCtrlItems( myitem )
    # update ListCtrl status
    self.lc.Focus( myitem )
    # update TreeCtrl status
    self.tc.SelectItem( itemid )

  # almost equivalent to evListCtrlRightClick event
  def evTreeCtrlRightClick( self, event ):
    """event handler for right click on TreeCtrl."""
    itemid = event.GetItem()
    itemids = self.treeCtrlGetSelectedItems()
    if not itemid.IsOk(): # may be waste
      self.lc_editing_index = -1
      self.lc_editing_indexes = []
    else:
      self.lc_editing_index = self.tc.GetItemPyData( itemid )[0]
      for myid in itemids:
        self.lc_editing_indexes.append( self.tc.GetItemPyData( myid )[0] )
    self.setPopupMenuState( self.lc_editing_index >= 0 )
    self.frameMain.PopupMenu( self.rightClickMenu )

  def evTreeCtrlRightClickSub( self, event ):
    """sub function for evTreeCtrlRightClick."""
    if not self.tc.IsEmpty():
      event.Skip()
      return
    self.lc_editing_index = -1
    self.lc_editing_indexes = []
    self.setPopupMenuState( self.lc_editing_index >= 0 )
    self.frameMain.PopupMenu( self.rightClickMenu )

  def evTreeCtrlKeyDown( self, event ):
    """keyboard event handler for TreeCtrl"""
    itemid = self.tc.GetSelection()
    ## this event cannot get valid itemid... why?
    #itemid = event.GetItem()
    if not itemid.IsOk():
      return
    index = self.tc.GetItemPyData( itemid )[0]
    keycode = event.GetKeyCode()
    self.keyboardEventHandler( index, keycode )

  #############
  # utilities #
  #############

  def findItemIdByUniqId( self, uniqid ):
    """find TreeCtrl itemid having given uniqid."""
    for myid in self.allItemIdGenerator():
      if myid.IsOk():
        if self.tc.GetItemPyData(myid)[2] == uniqid:
          return myid
    return None

  def gatherUniqIds( self, itemid ):
    """gather uniqids of itemid and its descendant."""
    if not itemid.IsOk():
      return None
    ret = [self.tc.GetItemPyData(itemid)[2]]
    self.gatherUniqIdsRecursive( itemid, ret )
    tmpid = self.tc.GetItemParent( itemid )
    if tmpid.IsOk():
      ret.insert( 0, self.tc.GetItemPyData( tmpid )[2] )
    else:
      ret.insert( 0, None )
    return ret

  def gatherUniqIdsRecursive( self, itemid, ret ):
    """sub function of gatherUniqIds."""
    if self.tc.ItemHasChildren( itemid ):
      child_itemid, cookie = self.tc.GetFirstChild( itemid )
      while child_itemid.IsOk():
        ret.append( self.tc.GetItemPyData(child_itemid)[2] )
        self.gatherUniqIdsRecursive( child_itemid, ret )
        child_itemid, cookie = self.tc.GetNextChild( itemid, cookie )

  # all itemid generator
  def allItemIdGenerator( self ):
    """TreeCtrl itemid generator."""
    rootid = self.tc.GetRootItem()
    if rootid.IsOk():
      yield rootid
      for val in self.childItemIdGeneratorRecursive( rootid ):
        yield val

  def childItemIdGeneratorRecursive( self, itemid ):
    """TreeCtrl itemid generator for given itemid and its descendant."""
    if self.tc.ItemHasChildren( itemid ):
      citemid, cookie = self.tc.GetFirstChild( itemid )
      while citemid.IsOk():
        yield citemid
        if self.tc.ItemHasChildren( citemid ):
          for val in self.childItemIdGeneratorRecursive( citemid ):
            yield val
        citemid, cookie = self.tc.GetNextChild( itemid, cookie )

  # get item position in TreeCtrl
  def calculateItemPosition( self, itemid ):
    """returns item position in TreeCtrl of given itemid."""
    myparent = self.tc.GetItemParent( itemid )
    for child_id, pos in self.childIdGenerator( myparent ):
      if child_id.IsOk():
        if child_id == itemid:
          return pos
    # if correspondending child does not exist, return None
    return None

  # generator
  def childIdGenerator( self, parent ):
    """generator for children ids of given parent."""
    child_itemid, cookie = self.tc.GetFirstChild( parent )
    counter = 0
    while child_itemid.IsOk():
      yield child_itemid, counter
      child_itemid, cookie = self.tc.GetNextChild( parent, cookie )
      counter += 1

  # not event
  def deleteTreeCtrlItem( self, itemid ):
    """delete a TreeCtrl item given by itemid."""
    if not itemid.IsOk():
      self.miscShowMessageBox( _("unexisting item id is requested to delete?") )
      return False
    self.tc.Delete( itemid )
    return True

  # equivalent function to listCtrlGetSelectedIndexes
  def treeCtrlGetSelectedItems( self ):
    """returns list of selected items in TreeCtrl."""
    ret = []
    for itemid in self.allItemIdGenerator():
      if itemid.IsOk():
        if self.tc.IsSelected( itemid ):
          ret.append( itemid )
    return ret

  def createTreeCtrlLabel( self, myxml ):
    """generate label for a TreeCtrl item."""
    if WMXmlEditorParams.TC_showattr_name:
      if len( myxml.attrib ) > 0:
        return myxml.tag + "(" + self.genAttrList( myxml, WMXmlEditorParams.TC_showattr_value ) + ")"
    return myxml.tag

  def genAttrList( self, myxml, genValue = False ):
    """generate attribute list of label for a TreeCtrl item."""
    ret = ""
    # attribute and value must be string
    for attr, val in myxml.attrib.items():
      if len( ret ) > 0:
        ret += ","
      ret += attr + ":" + val
    return ret

  def reassignUniqIDsRecursive( self, itemid, uniqids ):
    """re-assign unique ids for each item."""
    localcounter = 1
    for val in self.childItemIdGeneratorRecursive( itemid ):
      localcounter += 1
      # item order shall not be changed
      curdata = self.tc.GetItemPyData( val )
      self.tc.SetItemPyData( val, [ curdata[0], curdata[1], uniqids[localcounter] ] )

  def isEndTagRequired( self, myxml, itemid ):
    """returns whether end tag is required for given myxml."""
    if not myxml.text is None:
      if len( myxml.text ) > 0:
        return True
    if self.tc.ItemHasChildren( itemid ):
      return True
    return False

  def getIndentItem( self, itemid ):
    """returns the indent depth of given itemid."""
    ret = -1
    tmpid = itemid
    while tmpid.IsOk():
      ret += 1
      tmpid = self.tc.GetItemParent( tmpid )
    return( ret )

  ###################################################
  #
  # ListCtrl related
  #
  ###################################################
  # ListCtrl (right pane) events definition
  def registerListCtrlEvents( self ):
    """sub func of registerDefaultEvents, bind ListCtrl events to functions."""
    self.registerListCtrlSelectEvent()
    # corresponds to double click
    self.lc.Bind( wx.EVT_LIST_ITEM_ACTIVATED, self.evListCtrlDoubleClick )
    # right click
    self.lc.Bind( wx.EVT_LIST_ITEM_RIGHT_CLICK, self.evListCtrlRightClick )
    # key down
    self.lc.Bind( wx.EVT_LIST_KEY_DOWN, self.evListCtrlKeyDown )
    # drop files
    self.lc.Bind( wx.EVT_DROP_FILES, self.evDropFiles )

  # ListCtrl (rightpane) select/deselect events
  def registerListCtrlSelectEvent( self ):
    """sub func of registerListCtrlEvents, bind ListCtrl select/deselect events."""
    self.lc.Bind( wx.EVT_LIST_ITEM_SELECTED, self.evListCtrlSelectionOn )
    self.lc.Bind( wx.EVT_LIST_ITEM_DESELECTED, self.evListCtrlSelectionOff )

  # unused currently. just reserved.
  def unregisterListCtrlSelectEvent( self ):
    """unbind ListCtrl select/deselect events, unused currently."""
    self.lc.Unbind( wx.EVT_LIST_ITEM_SELECTED )
    self.lc.Unbind( wx.EVT_LIST_ITEM_DESELECTED )

  # create ListCtrl from pre-built TreeCtrl
  def createListCtrlFromTreeData( self ):
    """generate ListCtrl (right pane) from TreeCtrl data."""
    # initialize ListCtrl
    self.lc.ClearAll()
    self.l2l_correspondence = [] # begin-end tag correspondence list
    self.l2t_correspondence = [] # ListCtrl->TreeCtrl index correspondence
    self.lc.InsertColumn( 0, "XML" )
    # counter; indicating number of tags (including both begin and end tags)
    self.counter = 0
    # get root xml element and create field
    root_itemid = self.tc.GetRootItem()
    if not root_itemid.IsOk():
      return
    self.l2l_correspondence.append( self.counter )
    self.l2t_correspondence.append( root_itemid )
    root_element = self.tc.GetItemPyData( root_itemid )[1]
    # replace index
    self.tc.GetItemPyData( root_itemid )[0] = self.counter
    self.appendBeginTag( root_element, root_itemid, 0 )
    self.createListCtrlFromTreeDataRecursive( root_itemid, 1 )
    # if this element should have explicit end tag
    if self.isEndTagRequired( root_element, root_itemid ):
      self.counter += 1
      self.appendEndTag( root_element, root_itemid, 0 )
      self.l2l_correspondence[0] = self.counter
      # end tag has the same corresponding object as the begin tag
      self.l2l_correspondence.append(0)
      self.l2t_correspondence.append( root_itemid )
    self.lc.SetColumnWidth( 0, wx.LIST_AUTOSIZE )

  def createListCtrlFromTreeDataRecursive( self, itemid, indent ):
    """sub func of createListCtrlFromTreeData."""
    next_indent = indent + 1
    if self.tc.ItemHasChildren( itemid ):
      child_itemid, cookie = self.tc.GetFirstChild( itemid )
      while child_itemid.IsOk():
        self.counter += 1
        child_element = self.tc.GetItemPyData( child_itemid )[1]
        # replace index
        self.tc.GetItemPyData( child_itemid )[0] = self.counter
        self.l2l_correspondence.append( self.counter )
        self.l2t_correspondence.append( child_itemid )
        mynum = self.appendBeginTag( child_element, child_itemid, indent )
        self.createListCtrlFromTreeDataRecursive( child_itemid, next_indent )
        if self.isEndTagRequired( child_element, child_itemid ):
          self.appendEndTag( child_element, child_itemid, indent )
          self.counter += 1
          self.l2l_correspondence[mynum] = self.counter
          self.l2l_correspondence.append(mynum)
          self.l2t_correspondence.append( child_itemid )
        child_itemid, cookie = self.tc.GetNextChild( itemid, cookie )

  ##################
  # event handlers #
  ##################

  # single click (select) on ListCtrl
  def evListCtrlSelectionOn( self, event ):
    """ListCtrl select event handler."""
    self.workerListCtrlSelection( event.GetIndex() )

  # single click (deselect) on ListCtrl
  def evListCtrlSelectionOff( self, event ):
    """ListCtrl deselect event handler."""
    self.workerListCtrlSelection( event.GetIndex(), 0 )

  # double click on ListCtrl
  def evListCtrlDoubleClick( self, event ):
    """ListCtrl double click event handler."""
    self.editListCtrlItem( event.GetIndex() )

  # right click on ListCtrl
  def evListCtrlRightClick( self, event ):
    """event handler for right click on ListCtrl."""
    # remember editing index for child functions
    self.lc_editing_index = event.GetIndex()
    self.lc_editing_indexes = self.listCtrlGetSelectedIndexes()
    # reset menu state; disable some items when ListCtrl is empty
    self.setPopupMenuState( self.lc_editing_index >= 0 )
    self.frameMain.PopupMenu( self.rightClickMenu )

  def evListCtrlKeyDown( self, event ):
    """keyboard event handler for ListCtrl."""
    index = event.GetIndex()
    keycode = event.GetKeyCode()
    self.keyboardEventHandler( index, keycode )

  #############
  # utilities #
  #############

  def selectListCtrlItems( self, index, switch = wx.LIST_STATE_SELECTED ):
    """select given ListCtrl items."""
    self.unregisterListCtrlSelectEvent()
    minnum = min( index, self.l2l_correspondence[index] )
    maxnum = max( index, self.l2l_correspondence[index] )
    # select child elements, too
    for i in range( minnum, maxnum + 1 ):
      # not to use Select, be care on recursive event invoke and focus change
      self.lc.SetItemState( i, switch, wx.LIST_STATE_SELECTED )
    self.registerListCtrlSelectEvent()
    return minnum

  def selectListCtrlItem( self, index, switch = wx.LIST_STATE_SELECTED ):
    """select a given ListCtrl item."""
    self.unregisterListCtrlSelectEvent()
    minnum = min( index, self.l2l_correspondence[index] )
    maxnum = max( index, self.l2l_correspondence[index] )
    self.lc.SetItemState( minnum, switch, wx.LIST_STATE_SELECTED )
    if minnum != maxnum:
      self.lc.SetItemState( maxnum, switch, wx.LIST_STATE_SELECTED )
    self.registerListCtrlSelectEvent()

  def deselectAllItemsListCtrl( self ):
    """deselect all the ListCtrl items."""
    self.unregisterListCtrlSelectEvent()
    num = self.lc.GetItemCount()
    for i in range( 0, num ):
      self.lc.SetItemState( i, 0, wx.LIST_STATE_SELECTED )
    self.registerListCtrlSelectEvent()

  def workerListCtrlSelection( self, sel, switch = wx.LIST_STATE_SELECTED ):
    """worker function for evListCtrlSelectionOn and evListCtrlSelectionOff."""
    minnum = self.selectListCtrlItems( sel, switch )
    treeobj = self.l2t_correspondence[minnum]

    self.tc.UnselectAll()
    #### not available...?
    #self.tc.SetFocusedItem( treeobj )
    self.tc.SelectItem( treeobj )

  def appendBeginTag( self, xml, tree_xml, indent = 0 ):
    """create begin tag for TreeCtrl item."""
    need_end_tag = self.isEndTagRequired( xml, tree_xml )
    i = self.lc.Append( [self.myxml2string( xml, need_end_tag, indent )] )
    if xml.text != None:
      if len( xml.text.strip() ) > 0:
        self.lc.SetItemTextColour( i, WMXmlEditorParams.LC_emphasis_color )
    return i

  def appendEndTag( self, xml, tree_xml, indent = 0 ):
    """create end tag for TreeCtrl item."""
    if xml.text is None and not self.tc.ItemHasChildren( tree_xml ):
      return
    i = self.lc.Append( [self.xmlEndTag( xml, indent )] )
    if xml.text != None:
      if len( xml.text.strip() ) > 0:
        self.lc.SetItemTextColour( i, WMXmlEditorParams.LC_emphasis_color )
    return i

  # has_child consider both of child elements and inner text
  def myxml2string( self, element, need_end_tag, indent = 0 ):
    """convert xml data to string, and return it."""
    ret = "<" + element.tag
    for attr, val in element.items():
      ret += " " + attr + "=\"" + val + "\""
    if need_end_tag:
      ret += ">"
    else:
      ret += "/>"
    return self.xmlIndent(indent) + ret

  def xmlEndTag( self, xml, indent = 0 ):
    """returns xml end tag for given xml data and indent depth."""
    return self.xmlIndent(indent) + "</" + xml.tag + ">"

  def xmlIndent( self, indent ):
    """returns leading spaces of indent in ListCtrl."""
    return " " * ( WMXmlEditorParams.LC_indent_size * indent )

  def listCtrlGetSelectedIndexes( self ):
    """returns selected indexes in ListCtrl."""
    ret = []
    selected = self.lc.GetFirstSelected()
    while selected >= 0:
      ret.append( selected )
      selected = self.lc.GetNextSelected( selected )
    return ret

  ###################################################
  #
  # Undo/Redo
  #
  ###################################################

  # reset undo/redo queue items
  def clearUndoList( self ):
    """clear undo/redo queue items."""
    self.undo = []
    self.redo = []
    self.resetMenuEdit()

  def addActionToUndoQueue( self, action, remove = False ):
    """add given action to undo queue."""
    self.addActionToQueue( action, self.undo )
    if remove:
      self.redo = []

  def addActionToRedoQueue( self, action, remove = False ):
    """add given action to redo queue."""
    self.addActionToQueue( action, self.redo )
    if remove:
      self.undo = []

  def addActionToQueue( self, action, queue ):
    """worker for adding action to undo/redo queue."""
    queue.insert( 0, action )
    # trim list size if too big
    if len( queue ) > WMXmlEditorParams.UNDO_queue_size:
      del queue[WMXmlEditorParams.UNDO_queue_size:]
    self.resetMenuEdit()

  ###################################################
  #
  # About Window
  #
  ###################################################

  def initializeAboutWindow( self ):
    version_string = "unknown" # not for gettext
    self.dialogAbout = self.res.LoadDialog( None, "dialogAbout" )
    # replace version number if available
    ver = xrc.XRCCTRL( self.dialogAbout, "versionString" )
    ver.SetLabel( version_string )
    self.Bind( wx.EVT_BUTTON, self.evAboutCloseButton, id=xrc.XRCID("buttonCloseAbout") )

  def registerMenuEventsAbout( self ):
    self.Bind( wx.EVT_MENU, self.evMenuAboutAbout, id=xrc.XRCID("menuAboutAbout") )

  ##################
  # event handlers #
  ##################

  def evMenuAboutAbout( self, event ):
    self.dialogAbout.Show()

  def evAboutCloseButton( self, event ):
    self.dialogAbout.Hide()

  ###################################################
  #
  # misc functions
  #
  ###################################################

  def writeXml( self, fh ):
    """write Xml data to given file handler, fh."""
    xmltree, uniqids = self.createElementTreeFromTreeCtrl( self.tc.GetRootItem() )
    if xmltree is None:
      self.miscShowMessageBox( _("no data to write.") )
    else:
      xmltree.write( fh, encoding="UTF-8" )

  # create ElementTree instance of the element corresponding to itemid in
  # TreeCtrl and its children
  def createElementTreeFromTreeCtrl( self, itemid ):
    """returns ElementTree object of itemid and its descendants."""
    # if data is empty do nothing and return None
    if not itemid.IsOk():
      return None, None
    element_data = self.tc.GetItemPyData( itemid )
    element = self.createElementFromWMXmlElement( element_data[1] )
    # first item of uniqids is itemid of its parent, and the second is itemid
    # of the selected element
    uniqids = [element_data[2]]
    self.createElementFromTreeCtrlRecursive( itemid, element, uniqids )
    ret = ET.ElementTree( element )
    return ret, uniqids

  def createElementFromTreeCtrlRecursive( self, itemid, element, uniqids ):
    """worker for createElementTreeFromTreeCtrl."""
    if self.tc.ItemHasChildren( itemid ):
      child_itemid, cookie = self.tc.GetFirstChild( itemid )
      while child_itemid.IsOk():
        element_data = self.tc.GetItemPyData( child_itemid )
        element.append( self.createElementFromWMXmlElement( element_data[1] ) )
        uniqids.append( element_data[2] )
        self.createElementFromTreeCtrlRecursive( child_itemid, element[-1], uniqids )
        child_itemid, cookie = self.tc.GetNextChild( itemid, cookie )

  def createElementFromWMXmlElement( self, wmxml ):
    """convert WMXmlElement to ElementTree.Element, and returns it."""
    ret = ET.Element( wmxml.tag, wmxml.attrib )
    # text and tail should NOT be added to the constructor
    # Note that "copy" is NOT used here
    ret.text = wmxml.text
    ret.tail = wmxml.tail
    return ret

  def focusTreeCtrlItem( self, itemid ):
    """focus on selected TreeCtrl item."""
    # TreeCtrl
    self.tc.UnselectAll()
    self.tc.SelectItem( itemid )
    # ListCtrl
    index = self.tc.GetItemPyData( itemid )[0]
    self.deselectAllItemsListCtrl()
    self.selectListCtrlItems( index )
    self.lc.Focus( index )

  def miscShowMessageBox( self, msg, msgtype = 'Info' ):
    """show messabe box."""
    wx.MessageBox( msg, msgtype )

if __name__ == "__main__":
  app = WMXmlEditor( False )
  app.MainLoop()
