/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2009 Sony Ericsson Mobile Communications AB.         *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Mats Wilson (mats.wilson@sonyericsson.com)
 */

package com.sonyericsson.eventstream.telephonyplugin;

import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.ServiceIntentCmd;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.Telephony;
import com.sonyericsson.eventstream.telephonyplugin.utils.MockOriginatedSourceHelper;
import com.sonyericsson.eventstream.telephonyplugin.utils.MockTestCursor;
import com.sonyericsson.eventstream.telephonyplugin.utils.ReflectionUtilities;

import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.os.Handler;
import android.provider.ContactsContract.PhoneLookup;
import android.test.ServiceTestCase;
import android.util.Log;

import junit.framework.Assert;

public class UTelephonyPluginTests extends ServiceTestCase<TelephonyPluginService> {

    private static final int TIMEOUT = 10 * 1000;

    private static final String REGISTRATION_EVENT_MESSAGE = "registration_message";

    private static final String EVENT_MESSAGE_UNDERTEST = "test_message";

    private static final String FRIEND_NAME_FIRST = "test_testing_first";

    private static final String FRIEND_NAME_SECOND = "test_testing_second";

    private static final int REGISTRATION_EVENT_ID = 0;

    private static final int EVENT_UNDERTEST_ID = 1;

    TelephonyPluginService mService;

    public UTelephonyPluginTests() {
        super(TelephonyPluginService.class);
    }

    @Override
    protected void setUp() throws Exception {
        getContext().getContentResolver().delete(
                PluginConstants.EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
        getContext().getContentResolver().delete(
                PluginConstants.EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI, null, null);
        getContext().getContentResolver().delete(
                PluginConstants.EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI, null, null);
        getContext().getContentResolver().delete(
                PluginConstants.EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null);

        bindService(new Intent());
        mService = getService();

        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        if (mService != null) {
            if (mService.mContactsContentObserver != null) {
                mService.mContactsContentObserver.closeExistingBulkTimer();
            }
            if (mService.mMessageContentObserver != null) {
                mService.mMessageContentObserver.closeExistingBulkTimer();
            }
        }
        super.tearDown();
    }

    /**
     * Verify that the plug-in is registered correct and its source.
     * Will also check that sync if performed after the registration
     */
    public void testPluginRegistration() {
        performRegistration(mService);
    }

    /**
     * Verify that message on change trigger a new event
     */
    public void testOnChangeMessageObserver() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();

        performRegistration(mService);

        // Create a mock cursor with one sms item
        MockTestCursor cursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(cursor);
        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        mService.new MessagingContentObserver(new Handler()).onChange(true);
        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added", isEventAdded(EVENT_UNDERTEST_ID));
    }

    /**
     * Verify that message on change trigger a new friend
     */
    public void testOnChangeFriendObserver() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();

        performRegistration(mService);

        // Create a mock cursor with one sms item
        MockTestCursor smsCursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(smsCursor);
        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        // Refresh the service events
        Intent intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);
        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));

        MockTestCursor friendCursor = getMockFriendCursorOneItem(FRIEND_NAME_FIRST);
        helper.setAddressLookupCursor(friendCursor);

        mService.new ContactsContentObserver(new Handler()).onChange(true);
        waitForFriendAdded(FRIEND_NAME_FIRST);
        assertTrue("Friend with displayname should be added", isFriendAdded(FRIEND_NAME_FIRST));
    }

    /**
     * Verify that message on change trigger a new friend
     */
    public void testMappedContactChangeName() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();
        Intent intent = new Intent();

        performRegistration(mService);

        // Create a mock cursor with one sms item
        MockTestCursor smsCursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(smsCursor);
        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        MockTestCursor friendCursor = getMockFriendCursorOneItem(FRIEND_NAME_FIRST);
        helper.setAddressLookupCursor(friendCursor);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        // Perform service refresh
        intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);

        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));

        waitForFriendAdded(FRIEND_NAME_FIRST);
        assertTrue("Friend with displayname should be added", isFriendAdded(FRIEND_NAME_FIRST));

        friendCursor = getMockFriendCursorOneItem(FRIEND_NAME_SECOND);
        helper.setAddressLookupCursor(friendCursor);
        smsCursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(smsCursor);

        mService.new ContactsContentObserver(new Handler()).onChange(true);
        waitForFriendAdded(FRIEND_NAME_SECOND);
        assertTrue("Friend with displayname should be added", isFriendAdded(FRIEND_NAME_SECOND));
    }

    /**
     * Verify that a deleted mapped contact is removed
     */
    public void testMappedContactDeleted() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();

        performRegistration(mService);

        String address = helper.getAddressFromSmsId(null, 0);

        // Create a mock cursor with one sms item
        MockTestCursor smsCursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(smsCursor);

        MockTestCursor friendCursor = getMockFriendCursorOneItem(FRIEND_NAME_FIRST);
        helper.setAddressLookupCursor(friendCursor);

        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        // Perform service refresh
        Intent intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);

        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));

        waitForFriendAdded(FRIEND_NAME_FIRST);
        assertTrue("Friend with displayname should be added", isFriendAdded(FRIEND_NAME_FIRST));

        friendCursor = new MockTestCursor();
        friendCursor.expectedData("moveToFirst", "").andReturn(false);
        friendCursor.expectedData("getCount", "").andReturn(0);
        helper.setAddressLookupCursor(friendCursor);

        mService.new ContactsContentObserver(new Handler()).onChange(true);
        waitForFriendAdded(address);
        assertTrue("Friend with displayname should be added", isFriendAdded(address));
    }

    /**
     * Verify that refresh intent trigger a new event
     */
    public void testRefreshEventData() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();
        Intent intent = new Intent();

        performRegistration(mService);

        // Create a mock cursor with one sms item
        MockTestCursor cursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(cursor);
        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);
        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));
    }

    /**
     * Verify that two messages with matching number should be linked to the same friend.
     */
    public void testRefreshEventDataMatchingNumber() throws IllegalArgumentException, IllegalAccessException {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();
        Intent intent = new Intent();

        performRegistration(mService);

        int sourceId = mService.getSourceId();
        ContentValues values = new ContentValues();
        values.put(PluginConstants.EventStream.FriendColumns.SOURCE_ID, sourceId);
        values.put(PluginConstants.EventStream.FriendColumns.FRIEND_KEY, "012345678");
        getContext().getContentResolver().insert(
                PluginConstants.EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                values);

        // Create a mock cursor with one sms item
        MockTestCursor cursor = getMockEventCursorTwoItem(
                EVENT_UNDERTEST_ID,
                EVENT_UNDERTEST_ID+1);
        helper.setSourceMessageCursor(cursor);
        helper.setMessageText(EVENT_MESSAGE_UNDERTEST);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);
        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));
        waitForEventAdded(EVENT_UNDERTEST_ID+1);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID+1));
        Cursor eventCursor = null;
        try {
            eventCursor = getEvent(EVENT_UNDERTEST_ID);
            String friendKey1 = "";
            String friendKey2 = "";
            if (eventCursor != null && eventCursor.moveToFirst()) {
                friendKey1 = eventCursor.getString(
                        eventCursor.getColumnIndexOrThrow(
                                PluginConstants.EventStream.FriendColumns.FRIEND_KEY));
            }
            if (eventCursor != null) {
                eventCursor.close();
            }
            eventCursor = getEvent(EVENT_UNDERTEST_ID+1);
            if (eventCursor != null && eventCursor.moveToFirst()) {
                friendKey2 = eventCursor.getString(
                        eventCursor.getColumnIndexOrThrow(
                                PluginConstants.EventStream.FriendColumns.FRIEND_KEY));
            }
            assertEquals("the events should be linked to the same friend", friendKey1, friendKey2);
        } finally {
            eventCursor.close();
        }
    }
    /**
     * Verify that message on change trigger a new event
     */
    public void testChangeAfterMessagingItemDeleted() {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();
        Intent intent;

        performRegistration(mService);

        // Create a mock cursor with one sms item
        MockTestCursor cursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID);
        helper.setSourceMessageCursor(cursor);

        String expectedText = EVENT_MESSAGE_UNDERTEST;
        helper.setMessageText(expectedText);

        try {
            ReflectionUtilities.getField(mService, "mSourceHelper").set(mService,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        // Perform refresh
        intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
        getService().onStartCommand(intent, 0, 0);

        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added:", isEventAdded(EVENT_UNDERTEST_ID));

        // Create new mock content and replace the old, only this item should exist.
        cursor = getMockEventCursorOneItem(EVENT_UNDERTEST_ID + 1);
        helper.setSourceMessageCursor(cursor);
        expectedText = EVENT_MESSAGE_UNDERTEST + "_second";
        helper.setMessageText(expectedText);

        if (mService.mMessageContentObserver != null) {
            mService.mMessageContentObserver.onChange(true);
        } else {
            mService.mMessageContentObserver =
                mService.new MessagingContentObserver(new Handler());
        }
        waitForEventAdded(EVENT_UNDERTEST_ID);
        assertTrue("Event with message should be added", isEventAdded(EVENT_UNDERTEST_ID));
    }

    /**
     * Create a event mock cursor which item have eventId
     * @param eventId The id that the items should have
     * @return A mock cursor
     */
    private MockTestCursor getMockEventCursorOneItem(int eventId) {
        MockTestCursor cursor = new MockTestCursor();
        cursor.expectedData("moveToFirst", "").andReturn(true);
        cursor.expectedData("getCount", "").andReturn(1);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ID).andReturn(1);
        cursor.expectedData("getInt", "1").andReturn(eventId);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ADDRESS).andReturn(2);
        cursor.expectedData("getString", "2").andReturn("12345678");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_BODY).andReturn(3);
        cursor.expectedData("getString", "3").andReturn("My message");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_DATE).andReturn(4);
        cursor.expectedData("getLong", "4").andReturn(123L);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_TYPE).andReturn(5);
        cursor.expectedData("getInt", "5").andReturn(Telephony.Sms.MESSAGE_TYPE_INBOX);
        return cursor;
    }

    private MockTestCursor getMockEventCursorTwoItem(int eventId1, int eventId2) {
        MockTestCursor cursor = new MockTestCursor();
        cursor.expectedData("moveToFirst", "").andReturn(true);
        cursor.expectedData("getCount", "").andReturn(2);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ID).andReturn(1);
        cursor.expectedData("getInt", "1").andReturn(eventId1);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ADDRESS).andReturn(2);
        cursor.expectedData("getString", "2").andReturn("012345678");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_BODY).andReturn(3);
        cursor.expectedData("getString", "3").andReturn("My message");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_DATE).andReturn(4);
        cursor.expectedData("getLong", "4").andReturn(123L);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_TYPE).andReturn(5);
        cursor.expectedData("getInt", "5").andReturn(Telephony.Sms.MESSAGE_TYPE_INBOX);
        cursor.expectedData("moveToNext", "").andReturn(true);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ID).andReturn(1);
        cursor.expectedData("getInt", "1").andReturn(eventId2);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_ADDRESS).andReturn(2);
        cursor.expectedData("getString", "2").andReturn("+4612345678");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_BODY).andReturn(3);
        cursor.expectedData("getString", "3").andReturn("My message2");
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_DATE).andReturn(4);
        cursor.expectedData("getLong", "4").andReturn(1234L);
        cursor.expectedData("getColumnIndexOrThrow", Telephony.Sms.COLUMN_TYPE).andReturn(5);
        cursor.expectedData("getInt", "5").andReturn(Telephony.Sms.MESSAGE_TYPE_INBOX);
        return cursor;
    }

    /**
     * Create a friend mock cursor containing one item
     * @param name The name of the friend
     * @return A mock cursor
     */
    private MockTestCursor getMockFriendCursorOneItem(String name) {
        MockTestCursor friendCursor = new MockTestCursor();
        friendCursor.expectedData("moveToFirst", "").andReturn(true);
        friendCursor.expectedData("getCount", "").andReturn(1);
        friendCursor.expectedData("getColumnIndexOrThrow", PhoneLookup._ID).andReturn(1);
        friendCursor.expectedData("getInt", "1").andReturn(1);
        friendCursor.expectedData("getColumnIndexOrThrow", PhoneLookup.DISPLAY_NAME).andReturn(2);
        friendCursor.expectedData("getString", "2").andReturn(name);
        return friendCursor;
    }

    private void waitForEventAdded(int eventId) {
        int delay = 0;
        do {
            delay += 200;
            try {
                Thread.sleep(200);
                Thread.yield();
            } catch (InterruptedException e) {
                assertTrue("Wait failed", false);
            }
        } while (!isEventAdded(eventId) && delay <= TIMEOUT);
    }

    private boolean isEventAdded(int eventId) {
        Cursor c = null;
        try {
            c = getEvent(eventId);
            if (c != null && c.getCount() == 1) {
                return true;
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return false;
    }

    private Cursor getEvent(int eventId) {
        Cursor c = null;
        c = getContext().getContentResolver().query(
                PluginConstants.EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                null, "event_key=?", new String[]{""+eventId}, null);
        return c;
    }

   private void waitForFriendAdded(String displayName) {
       int delay = 0;
       do {
           delay += 200;
           try {
               Thread.sleep(200);
               Thread.yield();
           } catch (InterruptedException e) {
               assertTrue("Wait failed", false);
           }
       } while (!isFriendAdded(displayName) && delay <= TIMEOUT);
   }

    private boolean isFriendAdded(String displayName) {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(
                    PluginConstants.EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    null,
                    PluginConstants.EventStream.FriendColumns.DISPLAY_NAME + "=?",
                    new String[]{displayName},
                    null);
            if (c != null && c.getCount() == 1) {
                return true;
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return false;
    }

   private void waitForRegistration() {
       int delay = 0;
       do {
           delay += 200;
           try {
               Thread.sleep(200);
               Thread.yield();
           } catch (InterruptedException e) {
               assertTrue("Wait failed", false);
           }
       } while (!isRegistrated() && delay <= TIMEOUT);
   }

    private boolean isRegistrated() {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(
                    PluginConstants.EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null, null,
                    null);
            if (c != null && c.getCount() > 0) {
                if (isSourceRegistrated()) {
                    return true;
                }
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return false;
    }

    private boolean isSourceRegistrated() {
        Cursor c = null;
        try {
            c = getContext().getContentResolver().query(
                    PluginConstants.EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI, null, null, null,
                    null);
            if (c != null && c.getCount() > 0) {
                return true;
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return false;
    }

    private void performRegistration(TelephonyPluginService service) {
        MockOriginatedSourceHelper helper = new MockOriginatedSourceHelper();
        Intent intent = new Intent();

        // Create a mock cursor with one sms item
        MockTestCursor cursor = getMockEventCursorOneItem(REGISTRATION_EVENT_ID);
        helper.setSourceMessageCursor(cursor);

        helper.setMessageText(REGISTRATION_EVENT_MESSAGE);

        try {
            ReflectionUtilities.getField(service, "mSourceHelper").set(service,
                    helper);
        } catch (Exception e) {
            e.printStackTrace();
        }

        intent = new Intent();
        intent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REGISTER_PLUGIN);
        getService().onStartCommand(intent, 0, 0);
        waitForRegistration();
        assertTrue("Should be registered", isRegistrated());
        waitForEventAdded(REGISTRATION_EVENT_ID);
        assertTrue("Event with message should be added:" + REGISTRATION_EVENT_MESSAGE, isEventAdded(REGISTRATION_EVENT_ID));
    }
    /*
     * Check that database values are correctly updated on locale change:
     */
    public void testLocaleChange() {

        String junkPluginName = "junk plugin";
        String junkSourceName = "junk source";
        String expectedPluginName = getContext().getResources().getString(
                R.string.ts_messaging_application_name);

        String expectedSourceName = getContext().getResources().getString(
                R.string.ts_messaging_plugin_name);

        testPluginRegistration();
        //write junk values to database:
        setLocaleColumns(junkPluginName, junkSourceName);

        //verify junk values written
        checkLocaleColumns(expectedPluginName, expectedSourceName, true);

        // trigger locale change
        Intent serviceIntent = new Intent();
        serviceIntent.setComponent(new ComponentName(getContext(), TelephonyPluginService.class));
        serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY,
                                ServiceIntentCmd.LOCALE_CHANGED);
        getContext().startService(serviceIntent);

        sleep(2);//give the service a chance to write new values to the database..
        // check new values in database are as expected.
        checkLocaleColumns(expectedPluginName, expectedSourceName, false);
    }

    private void setLocaleColumns(String pluginName, String sourceName) {

        putStringValueInColumn(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                                        EventStream.PluginColumns.NAME,
                                        pluginName);

        putStringValueInColumn(EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI,
                                  EventStream.SourceColumns.NAME,
                                        sourceName);
    }

     private void checkLocaleColumns(String pluginName, String sourceName, boolean shouldFail) {

         checkExpectedStringValue(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                                              EventStream.PluginColumns.NAME,
                                              pluginName,
                                              shouldFail);

         checkExpectedStringValue(EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI,
                                              EventStream.SourceColumns.NAME,
                                              sourceName,
                                              shouldFail);
     }

     private void putStringValueInColumn (Uri tableUri, String columnName, String value) {

         ContentValues values = new ContentValues();
         int result;

         values.put(columnName, value);

         try {
             result = getContext().getContentResolver().update(tableUri, values, null, null);
                 if (result != 1) {
                          Log.e(Config.LOG_TAG, "Failed to update column: " + columnName + "in: " + tableUri +
                                  "result= " + result);
                 }
         } catch (SQLException exception) {
         Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
         throw new RuntimeException(exception);
         } catch (SecurityException exception) {
         Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
         throw new RuntimeException(exception);
         }
     }

     /*
     * Compares actual value in column with expected value. If shouldFail == true, then the function will throw an
     * assert if the values match and vice versa.
     */
     private void checkExpectedStringValue(Uri tableUri, String columnName, String expectedValue, boolean shouldFail) {

         Cursor cursor = null;

         try {
             ContentResolver contentResolver = getContext().getContentResolver();
             cursor = contentResolver.query(tableUri, null, null, null, null);

             if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
             // Check the value in field columnName
             String tempValue = cursor.getString(cursor.getColumnIndexOrThrow(columnName));
             if (shouldFail) {
                 Assert.assertTrue(!expectedValue.equals(tempValue));
             } else {
                 Assert.assertTrue(expectedValue.equals(tempValue));
             }
             }
         } finally {
             if (cursor != null) {
                 cursor.close();
             }
         }
     }

     private long getSourceId(Context context) {
         long sourceId = -1;
         Cursor cursor = null;

         try {
             cursor = context.getContentResolver().query(
                     EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI, new String[] {
                         EventStream.SourceColumns._ID
                     }, null, null, null);
             if (cursor != null && cursor.moveToFirst()) {
                 sourceId = cursor.getLong(cursor.getColumnIndex(EventStream.SourceColumns._ID));
             }
         } catch (SQLException exception) {
             Log.w(Config.LOG_TAG, "Failed to query source");
             throw new RuntimeException(exception);
         } catch (SecurityException exception) {
             Log.w(Config.LOG_TAG, "Failed to query source");
             throw new RuntimeException(exception);
         } finally {
             if (cursor != null) {
                 cursor.close();
             }
         }
         return sourceId;
     }

     public void sleep(int seconds) {
         try {
             Thread.sleep(seconds * 1000);
         } catch (InterruptedException e) {
             e.printStackTrace();
         }
     }
}
