/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.simpleui.utils;

import com.sonyericsson.simpleui.R;
import com.sonyericsson.simpleui.activity.SimpleUiInternal.Config;
import com.sonyericsson.simpleui.provider.EventStreamContentProvider;

import android.app.Activity;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.provider.ContactsContract;
import android.util.Log;
import android.widget.ImageView;

public class ImageManager {

    private static ImageManager sInstance;
    private Context mContext;
    private static Bitmap mDefaultEventImage;
    private static String LOG_TAG = "SimpleUiImageManager";
    private static ImageDownloader mImageDownloader;

    private ImageManager(Context context) {
        mContext = context;
    }

    public static ImageManager getInstance(Activity activity) {
        if (sInstance == null) {
            sInstance = new ImageManager(activity.getApplicationContext());
        }
        mDefaultEventImage = BitmapFactory.decodeResource(activity.getResources(),
                R.drawable.user_default_picture_1);
        if (mDefaultEventImage == null) {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Failed to decode default background image!!");
            }
        }
        mImageDownloader = new ImageDownloader(mDefaultEventImage);
        return sInstance;
    }

    /**
     * Figures out which image to show together with the event message.
     * Rules go like this:
     * -display the image in Event table image_uri column
     * -if that is empty, check if the Friend is listed in Contacts.
     * -if the Friend is in Contacts, get the contact image
     * -if the contact image is also empty, show the default image.
     *
     * @param view The ImageView to attach the drawable or bitmap to
     * @param cursor The cursor containing the necessary information for getting the correct image
     */
    public void getEventImage(ImageView view, Cursor cursor) {
        if (cursor != null) {
            Uri imageUri = Uri.parse("");
            String imageUriColumn = cursor.getString(cursor.getColumnIndex(
                    EventStreamContentProvider.EventColumns.IMAGE_URI));
            Log.d(LOG_TAG, "getEventImage for imageUriColumn = " + imageUriColumn);
            if (imageUriColumn != null) {
                imageUri = Uri.parse(imageUriColumn);
                Log.d(LOG_TAG, "imageUri = " + imageUri);

                // 1. Check if the imageUri is empty
                if (!Uri.EMPTY.equals(imageUri)) {
                    // 2. Is the image already in the in-memory cache?
                    // If not, need to trigger a download
                    boolean loaded = mImageDownloader.getImageFromCache(imageUri, view);
                    if (!loaded) {
                        // Trigger a download
                        mImageDownloader.triggerDownload(mContext, imageUri, view);
                    }
                }
            } else {
                // 3. If empty, is the Friend of this message also in Contacts?
                // If so, try to get its image from there
                String friendKey = cursor.getString(cursor.getColumnIndexOrThrow(
                        EventStreamContentProvider.EventColumns.FRIEND_KEY));

                if (friendKey != null) {
                    int sourceId = cursor.getInt(cursor.getColumnIndexOrThrow(
                            EventStreamContentProvider.EventColumns.SOURCE_ID));

                    String contactReference = getContactReference(friendKey, sourceId);

                    if (contactReference == null) {
                        // Since there is no contact information for this Friend,
                        // use the default image
                        Log.d(LOG_TAG, "No associated Contact info so using default image");
                        view.setImageBitmap(mDefaultEventImage);
                    } else {
                        // If there is a match and the profile image URI exists, use it
                        // Otherwise, use the default image
                        Log.d(LOG_TAG, "Getting contact image for raw contact: " + contactReference);
                        Uri contactUri = null;

                        if (contactReference.startsWith(ContactsContract.RawContacts.CONTENT_URI.toString())) {
                            contactUri = DBUtilities.getContactUri(mContext, contactReference);
                        } else {
                            contactUri = Uri.parse(contactReference);
                        }
                        if (contactUri != null) {
                            getContactImage(contactUri, view);
                        }
                    }
                } else {
                    // Since there is no associated "Friend", use the default image
                    Log.d(LOG_TAG, "No associated Friend so using default image");
                    view.setImageBitmap(mDefaultEventImage);
                }
            }
        }
    }

    private void getContactImage(Uri contactUri, ImageView view) {
        if (contactUri != null) {
            Uri contactPhotoUri = Uri.withAppendedPath(contactUri, ContactsContract.Contacts.Photo.CONTENT_DIRECTORY);
            boolean loaded = mImageDownloader.getImageFromCache(contactPhotoUri, view);
            if (!loaded) {
                Log.d(LOG_TAG, "Contact image not in in-memory cache, so need to download");
                mImageDownloader.triggerDownload(mContext, contactPhotoUri, view);
            }
        }
    }

    private String getContactReference(String friendKey, int sourceId) {
      String contactReference = null;
      Cursor friendCursor = null;
      String[] friendProjection = new String[] {EventStreamContentProvider.FriendColumns.CONTACTS_REFERENCE};
      StringBuilder selection = new StringBuilder();

      selection.append(EventStreamContentProvider.FriendColumns.FRIEND_KEY);
      selection.append("=? AND ");
      selection.append(EventStreamContentProvider.FriendColumns.SOURCE_ID);
      selection.append("=? ");
      String friendSelection = selection.toString();

      String[] friendSelectionArgs = new String[2];
      friendSelectionArgs[0] = friendKey;
      friendSelectionArgs[1] = new Integer(sourceId).toString();

      String sortOrder = null;

      try {
          friendCursor = mContext.getContentResolver().query(
                  EventStreamContentProvider.Uris.FRIENDS_CONTENT_URI,
                  friendProjection, friendSelection, friendSelectionArgs,
                  sortOrder);
          if (friendCursor != null && friendCursor.moveToFirst()) {
              contactReference = friendCursor.getString(
                      friendCursor.getColumnIndexOrThrow(EventStreamContentProvider.FriendColumns.CONTACTS_REFERENCE));
          }
      } catch (SQLException e) {
          Log.e(Config.LOG_TAG, "Couldn't query for friend key: " + friendKey + " and source-id:" + sourceId);
      } finally {
          if (friendCursor != null) {
              friendCursor.close();
          }
      }

      return contactReference;
    }
}
