/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file FacebookCommunication.java
 *
 * @author
 */
package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.API_KEY_PARAM;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LOG_TAG;
import static com.sonyericsson.eventstream.facebookplugin.Constants.SESSION_KEY_PARAM;
import static com.sonyericsson.eventstream.facebookplugin.Constants.TIME_PARAM;
import static com.sonyericsson.eventstream.facebookplugin.Constants.UID_PARAM;
import static com.sonyericsson.eventstream.facebookplugin.Constants.URL_PARAM;
import static com.sonyericsson.eventstream.facebookplugin.Constants.SIG_PARAM;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;

import org.apache.http.NameValuePair;
import org.apache.http.message.BasicNameValuePair;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

public class FacebookSecurity {

    /** Uri for authentication **/
    private final static String AUTH_URI_STRING = "http://touch.facebook.com/auth.php";

    /** The time to deeplink is expired. */
    private final static long EXPIRE_TIME = 60;

    private static final String DIGEST_ALGORITHM = "MD5";

    private static final String ENCODING = "UTF-8";

    public String generateAPIKey() {
        CredentialHandler handler = CredentialHandler.getInstance();
        return decode(handler.getCredential(CredentialHandler.FACEBOOK_API_KEY));
    }

    public String generateSecretKey() {
        CredentialHandler handler = CredentialHandler.getInstance();
        return decode(handler.getCredential(CredentialHandler.FACEBOOK_SECRET_KEY));
    }

    public Uri generateDeepLink(Context context, String uid, String destUriString) {
        String time = String.valueOf(System.currentTimeMillis()/1000 + EXPIRE_TIME);
        String facebook_source_appkey = generateAPIKey();
        String session_key = new Settings(context).getSessionKey();
        Uri.Builder builder = Uri.parse(AUTH_URI_STRING).buildUpon();

        if (context == null || uid == null || destUriString == null) {
            return null;
        }

        // If we for some reason can't retrieve the needed parameters, skip
        // deep linking and have the user login instead...
        if (session_key == null) {
            return Uri.parse(destUriString);
        }

        List<NameValuePair> parameters = new ArrayList<NameValuePair>();

        parameters.add(new BasicNameValuePair(API_KEY_PARAM, facebook_source_appkey));
        parameters.add(new BasicNameValuePair(SESSION_KEY_PARAM, session_key));
        parameters.add(new BasicNameValuePair(TIME_PARAM, time));
        parameters.add(new BasicNameValuePair(UID_PARAM, uid));
        parameters.add(new BasicNameValuePair(URL_PARAM, destUriString));
        parameters.add(new BasicNameValuePair(SIG_PARAM, generateDeepLinkSignature(context, parameters)));

        for (NameValuePair pair : parameters) {
            builder.appendQueryParameter(pair.getName(), pair.getValue());
        }

        return builder.build();
    }

    public String generateAuthenticationSignature(List<NameValuePair> parameters) {
        final String secretKey = generateSecretKey();

        return generateSignature(parameters, secretKey);
    }

    private String generateDeepLinkSignature(Context context, List<NameValuePair> parameters) {
        final String sessionSecret = new Settings(context).getSessionSecret();

        if (sessionSecret != null) {
            return generateSignature(parameters, sessionSecret);
        } else {
            return "";
        }
    }

    private String generateSignature(List<NameValuePair> parameters, String appendingString) {
        try {
            StringBuilder builder = new StringBuilder();
            for (NameValuePair parameter : parameters) {
                builder.append(parameter.getName()).append('=').append(parameter.getValue());
            }
            if (appendingString != null) {
                builder.append(appendingString);
            }

            MessageDigest algorithm = MessageDigest.getInstance(DIGEST_ALGORITHM);

            algorithm.reset();
            algorithm.update(builder.toString().getBytes(ENCODING));
            byte messageDigest[] = algorithm.digest();
            BigInteger values = new BigInteger(1, messageDigest);
            String hashString = values.toString(16);
            // The leading zeroes of the hash are lost when we convert it to
            // a BigInteger. Replace them with new ones.
            int leadingZeroes = messageDigest.length * 2 - hashString.length();
            StringBuffer resultString = new StringBuffer(hashString.length() + leadingZeroes);

            for (int i = 0; i < leadingZeroes; i++) {
                resultString.append("0");
            }
            resultString.append(hashString);

            return resultString.toString();
        } catch (NoSuchAlgorithmException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error generating method signature, " + DIGEST_ALGORITHM + " not available.");
                e.printStackTrace();
            }
        } catch (UnsupportedEncodingException e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error generating method signature, invalid encoding.");
                e.printStackTrace();
            }
        }
        return null;
    }

    /**
     * Decode a text string.
     *
     * @param cipherText    text to decrypt
     * @return              the decrypted text
     */
    private String decode(String cipherText) {
        byte[] plainText = null;
        byte[] cipherTextArray = new BigInteger(cipherText, 16).toByteArray();
        SecretKeySpec secretKeySpec = new SecretKeySpec(new byte[] {
            0x07, 0x17, 0x02, 0x0f, 0x13, 0x05, 0x15, 0x12, 0x08, 0x00, 0x01,
            0x03, 0x16, 0x0c, 0x0d, 0x11, 0x10, 0x04, 0x06, 0x09, 0x14, 0x0a,
            0x0b, 0x0e}, "AES");

        try {
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding", "BC");
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec);

            plainText = cipher.doFinal(cipherTextArray);
        } catch (Exception e) {
            throw new RuntimeException("Failed to decrypt key", e);
        }

        // return the text, and remove trailing spaces
        try {
            return new String(plainText, ENCODING).trim();
        } catch (UnsupportedEncodingException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
    }
}