/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin;

import android.content.Context;
import android.content.SharedPreferences;
import android.test.InstrumentationTestCase;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;

public class UPreferenceHelperTests extends InstrumentationTestCase {

    Context mContext;

    @Override
    protected void setUp() throws Exception {
        super.setUp();
        mContext = getInstrumentation().getTargetContext();

        // Clear all values
        SharedPreferences preferences = mContext.getSharedPreferences(
                PreferencesHelper.CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.clear();
                editor.commit();
            }
        }
    }

    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * Test that the disclaimer is not accepted
     */
    public void testHasNotAcceptedDisclaimer() {
        assertEquals("Disclaimer accepted should not be set", false, PreferencesHelper
                .hasAcceptedDisclaimer(mContext));
    }

    /**
     * Test that the disclaimer value can be accepted
     */
    public void testHasAcceptedDisclaimer() {
        PreferencesHelper.setAcceptedDisclaimer(mContext);

        assertEquals("Disclaimer accepted should be set", true, PreferencesHelper
                .hasAcceptedDisclaimer(mContext));
    }

    /**
     * Test that if the token is not set.
     */
    public void testHasNoToken() {
        assertEquals("Token should not be set", false, PreferencesHelper.hasToken(mContext));
    }

    /**
     * Test that if the token is set.
     */
    public void testSetToken() {
        PreferencesHelper.setToken(mContext, "token", "token");
        assertEquals("Token should be set", true, PreferencesHelper.hasToken(mContext));
    }

    /**
     * The token should not be set if null is inserted
     */
    public void testSetNullToken() {
        PreferencesHelper.setToken(mContext, null, "token");
        assertEquals("Token should not be set", false, PreferencesHelper.hasToken(mContext));

        PreferencesHelper.setToken(mContext, "token", null);
        assertEquals("Token should not be set", false, PreferencesHelper.hasToken(mContext));
    }

    /**
     * Test to get a set token
     */
    public void testHasValidToken() {
        PreferencesHelper.setToken(mContext, "token", "secret");

        assertEquals("Token key should be set", "token", PreferencesHelper.getTokenKey(mContext));

        assertEquals("Token secret key should be set", "secret", PreferencesHelper
                .getTokenSecretKey(mContext));
    }

    /**
     * Test to set and get the paging since id
     */
    public void testSetAndGetPagingSinceId() {
        String unsupported = "unsupported";
        PreferencesHelper.setPagingSinceId(mContext, unsupported, 10);

        assertEquals("Paging for unsupported should not be set", 0, PreferencesHelper
                .getPagingSinceId(mContext, unsupported));

        PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, 20);

        assertEquals("Paging for message should not be set", 0, PreferencesHelper.getPagingSinceId(
                mContext, TwitterConf.PAGING_TYPE_MESSAGE));

        assertEquals("Paging for status should be set", 20, PreferencesHelper.getPagingSinceId(
                mContext, TwitterConf.PAGING_TYPE_STATUS));

        PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, 30);

        assertEquals("Paging for message should be set", 30, PreferencesHelper.getPagingSinceId(
                mContext, TwitterConf.PAGING_TYPE_MESSAGE));
    }

    /**
     * Test to clear authentication values
     */
    public void testClearAuthentication() {

        PreferencesHelper.setAcceptedDisclaimer(mContext);
        PreferencesHelper.setToken(mContext, "token", "secret");
        PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, 10);
        PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, 20);

        PreferencesHelper.clearAuthentication(mContext);

        assertEquals("Accepted disclaimer should not be cleared", true, PreferencesHelper
                .hasAcceptedDisclaimer(mContext));
        assertEquals("Token should be cleared", false, PreferencesHelper.hasToken(mContext));
        assertEquals("Paging for status should be cleared", 0, PreferencesHelper.getPagingSinceId(
                mContext, TwitterConf.PAGING_TYPE_STATUS));
        assertEquals("Paging for message should be cleared", 0, PreferencesHelper.getPagingSinceId(
                mContext, TwitterConf.PAGING_TYPE_MESSAGE));
    }
}
