/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY_PARAMETER;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.Instrumentation;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Intent;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.test.ActivityInstrumentationTestCase2;
import android.util.Log;
import android.util.SparseArray;
import android.view.WindowManager.BadTokenException;
import android.widget.Button;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;
import com.sonyericsson.eventstream.facebookplugin.FacebookPluginApplication.State;
import com.sonyericsson.eventstream.facebookplugin.util.DatabaseHelper;

import java.lang.reflect.Field;

import junit.framework.Assert;

public class FFacebookPluginConfigTests extends
        ActivityInstrumentationTestCase2<FacebookPluginConfig> {

    private FacebookPluginConfig mActivity;
    private Facebook mFacebook;
    private FacebookPluginApplication mApplication;
    private volatile boolean mLoggedIn;

    public FFacebookPluginConfigTests() {
        super("com.sonyericsson.eventstream.facebookplugin", FacebookPluginConfig.class);
    }

    @Override
    protected void setUp() throws Exception {
        mFacebook = FacebookFactory.getFacebook(getInstrumentation().getTargetContext());
        Field f = Facebook.class.getDeclaredField("mSettings");
        f.setAccessible(true);
        f.set(mFacebook, mTestSettings);
        super.setUp();

        // Reset settings...
        new Settings(getInstrumentation().getTargetContext()).removeSettings();
    }

    @Override
    protected void tearDown() throws Exception {
        // Kill all dialogs
        mActivity.removeDialog(FacebookPluginConfig.DIALOG_DISCLAIMER);
        mActivity.removeDialog(FacebookPluginConfig.DIALOG_ERROR);
        mActivity.removeDialog(FacebookPluginConfig.DIALOG_LOGIN);
        mActivity.removeDialog(FacebookPluginConfig.DIALOG_PROGRESS);
        super.tearDown();
    }

    public void testShowDisclaimer() throws SecurityException, IllegalArgumentException,
            NoSuchFieldException, IllegalAccessException, InterruptedException {
        mActivity = getActivity();
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_ERROR));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGIN));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGOUT));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_PROGRESS));
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
    }

    public void testLoginDialogShowing() {
        new Settings(getInstrumentation().getTargetContext()).setHasAcceptedDisclaimer(true);

        mActivity = getActivity();
        getInstrumentation().waitForIdleSync();
        mApplication = (FacebookPluginApplication) mActivity.getApplication();

        mApplication.removeStateListener(mActivity);
        mApplication.setState(State.NOT_CONFIGURED);
        getInstrumentation().callActivityOnDestroy(mActivity);
        mActivity.setInitialStates();

        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_ERROR));
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGIN));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGOUT));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_PROGRESS));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
    }

    public void testShowLogout() throws SecurityException, IllegalArgumentException,
            NoSuchFieldException, IllegalAccessException {
        // The user has accepted the disclaimer...
        new Settings(getInstrumentation().getTargetContext()).setHasAcceptedDisclaimer(true);
        new Settings(getInstrumentation().getTargetContext()).setAuthenticationToken("BAD_TOKEN");

        mActivity = getActivity();
        getInstrumentation().waitForIdleSync();
        mApplication = (FacebookPluginApplication) mActivity.getApplication();

        mApplication.removeStateListener(mActivity);
        mApplication.setState(State.AUTHENTICATED);
        getInstrumentation().callActivityOnDestroy(mActivity);
        mActivity.setInitialStates();

        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_ERROR));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGIN));
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGOUT));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_PROGRESS));
    }

    public void testShowProgress() throws SecurityException, NoSuchFieldException,
            IllegalArgumentException, IllegalAccessException, InterruptedException {
        new Settings(getInstrumentation().getTargetContext()).setHasAcceptedDisclaimer(true);
        new Settings(getInstrumentation().getTargetContext()).setAuthenticationToken("BAD_TOKEN");

        mActivity = getActivity();
        getInstrumentation().waitForIdleSync();
        mApplication = (FacebookPluginApplication) mActivity.getApplication();

        mApplication.removeStateListener(mActivity);
        mApplication.setState(State.AUTHENTICATION_IN_PROGRESS);
        getInstrumentation().callActivityOnDestroy(mActivity);
        mActivity.setInitialStates();

        mActivity.onResume();

        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_ERROR));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGIN));
        assertFalse(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGOUT));
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_PROGRESS));
    }

    public void testDisclaimerDialogsOnPause() throws SecurityException, IllegalArgumentException,
            NoSuchFieldException, IllegalAccessException, InterruptedException {
        mActivity = getActivity();
        getInstrumentation().waitForIdleSync();
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
        mActivity.onPause();
        getInstrumentation().waitForIdleSync();
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_DISCLAIMER));
    }

    public void testLoginFail() throws SecurityException, NoSuchFieldException,
            IllegalArgumentException, IllegalAccessException {
        Instrumentation instr = getInstrumentation();
        mActivity = getActivity();

        // Fake login procedure...
        FacebookNotification progressNotification = new FacebookNotification(State.AUTHENTICATION_IN_PROGRESS);
        mActivity.onStateChange(progressNotification);
        instr.waitForIdleSync();
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_PROGRESS));

        // Fail!
        FacebookNotification failedNotification = new FacebookNotification(State.AUTHENTICATION_FAILED);
        mActivity.onStateChange(failedNotification);
        instr.waitForIdleSync();
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_ERROR));
    }

    public void testAcceptDisclaimer() throws IllegalArgumentException, SecurityException,
            IllegalAccessException, NoSuchFieldException {
        mActivity = getActivity();
        Instrumentation instr = getInstrumentation();
        instr.waitForIdleSync();

        AlertDialog disclaimer = (AlertDialog) getDialog(FacebookPluginConfig.DIALOG_DISCLAIMER);
        assertNotNull(disclaimer);
        final Button okButton = disclaimer.getButton(AlertDialog.BUTTON_POSITIVE);
        mActivity.runOnUiThread(new Runnable() {
            public void run() {
                okButton.performClick();
            }
        });
        instr.waitForIdleSync();
        assertTrue(isDialogShowing2(FacebookPluginConfig.DIALOG_LOGIN));
    }

    public void testStateChangeAfterFinish() {
        mActivity = getActivity();
        mActivity.finish();
        Instrumentation instr = getInstrumentation();
        instr.waitForIdleSync();
        boolean exceptionThrown = false;
        try {

            mActivity.onStateChange(new FacebookNotification(FacebookPluginApplication.State.AUTHENTICATION_IN_PROGRESS));
            instr.waitForIdleSync();
        } catch (BadTokenException e) {
            exceptionThrown = true;
        }
        assertFalse("Failed testcase: forbidden to execute in UIThread after the activity was finished!", exceptionThrown);
    }

    private void setLocaleColumns(String configName, String configText) {
        Cursor cursor = null;
        try {
            ContentResolver contentResolver = getInstrumentation().getTargetContext().getContentResolver();
            cursor = contentResolver.query(EventStreamConstants.SOURCE_PROVIDER_URI, null, null,
                    null, null);

            if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
                ContentValues values = new ContentValues();
                values.put(EventStreamConstants.PluginTable.NAME, configName);
                values.put(EventStreamConstants.PluginTable.CONFIGURATION_TEXT, configText);
                contentResolver
                        .update(EventStreamConstants.PLUGIN_PROVIDER_URI, values, null, null);
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    private void checkLocaleColumns(String configName, String configText, boolean shouldFail) {
        Cursor cursor = null;
        try {
            ContentResolver contentResolver = getInstrumentation().getTargetContext().getContentResolver();
            cursor = contentResolver.query(EventStreamConstants.PLUGIN_PROVIDER_URI, null, null,
                    null, null);

            if (cursor != null && cursor.moveToFirst() && cursor.getCount() == 1) {
                // Check the name
                String tempName = cursor.getString(cursor
                        .getColumnIndexOrThrow(EventStreamConstants.PluginTable.NAME));

                if (shouldFail) {
                    Assert.assertTrue(!configName.equals(tempName));
                } else {
                    Assert.assertTrue(configName.equals(tempName));
                }

                // Check the text
                String tempText = cursor
                        .getString(cursor
                                .getColumnIndexOrThrow(EventStreamConstants.PluginTable.CONFIGURATION_TEXT));

                if (shouldFail) {
                    Assert.assertTrue(!configText.equals(tempText));
                } else {
                    Assert.assertTrue(configText.equals(tempText));
                }
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    public void testLocaleChange() {

        String junkServiceName = "junk service name";
        String junkConfigText = "junk config text";
        String expectedServiceName = getInstrumentation().getTargetContext().getResources().getString(
                                    R.string.ts_facebook_service_name);
        String expectedConfigText = getInstrumentation().getTargetContext().getResources().getString(
                                    R.string.facebook_register_txt);

        mActivity = getActivity();

        //write junk values to database:
        setLocaleColumns(junkServiceName, junkConfigText);

        //verify junk values written
        checkLocaleColumns(expectedServiceName, expectedConfigText, true);
        // trigger locale change
        Intent serviceIntent = new Intent(Constants.LOCALE_CHANGED_INTENT);
        serviceIntent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
        getInstrumentation().getTargetContext().startService(serviceIntent);

        sleep(2);

        // check new values in database are as expected.
        checkLocaleColumns(expectedServiceName, expectedConfigText, false);
    }

    private boolean isDialogShowing2(int dialogId) {
        boolean result = true;
        try {
            mActivity.dismissDialog(dialogId);
        } catch (Exception exception) {
            result = false;
        }

        return result;
    }

    @SuppressWarnings("unchecked")
    private Dialog getDialog(int id) throws IllegalArgumentException, IllegalAccessException,
            SecurityException, NoSuchFieldException {
        Field f = Activity.class.getDeclaredField("mManagedDialogs");
        f.setAccessible(true);
        SparseArray managedDialogs = (SparseArray)f.get(mActivity);
        Object dialogWrapper = managedDialogs.get(id);
        if (dialogWrapper instanceof Dialog) {
            return (Dialog)dialogWrapper;
        } else if (dialogWrapper != null) {
            // Gingerbread
            Field fields[] = dialogWrapper.getClass().getDeclaredFields();
            for (Field unknownField : fields) {
                unknownField.setAccessible(true);
                Object instanceField = unknownField.get(dialogWrapper);
                if (instanceField instanceof Dialog) {
                    return (Dialog)instanceField;
                }
            }
        }
        return null;
    }

    private void sleep(int seconds) {
        try {
            Thread.sleep(seconds * 1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    private Facebook.Settings mTestSettings = new Facebook.Settings() {

        public String getAuthenticationToken() {
            if (mLoggedIn) {
                return "dummy_token";
            } else {
                return null;
            }
        }

        public String getOwnId() {
            // TODO Auto-generated method stub
            return null;
        }

        public Long getYoungestTimestamp() {
            // TODO Auto-generated method stub
            return null;
        }

        public void setAuthenticationToken(String token) {
            // TODO Auto-generated method stub

        }

        public void setOwnId(String id) {
            // TODO Auto-generated method stub

        }

        public void setSessionKey(String sessionKey) {
            // TODO Auto-generated method stub

        }

        public void setSessionSecret(String sessionSecret) {
            // TODO Auto-generated method stub

        }

        public void setYoungestTimestamp(Long newValue) {
            // TODO Auto-generated method stub

        }

    };

}
