/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file FacebookCommunication.java
 *
 * @author
 */
package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.ACCESS_TOKEN_PREFS_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.CONFIG_STORE;
import static com.sonyericsson.eventstream.facebookplugin.Constants.DISPLAY_NAME_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.HAS_ACCEPTED_DISCLAIMER;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LAST_COMMUNICATION_WITH_SERVER_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.NEWS_FEED_SINCE_PREFS_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.OWN_ID_PREFS_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.SESSION_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.SESSION_SECRET;

import android.content.Context;
import android.content.SharedPreferences;

public class Settings implements Facebook.Settings {
    private static final int MODE_PRIVATE = android.content.Context.MODE_PRIVATE;
    private Context mContext = null;

    public Settings(Context context) {
        mContext = context;
    }

    /**
     * Remove all settings
     */
    public void removeSettings() {
        setAuthenticationToken(null);
        setYoungestTimestamp(null);
        setLastCommunicationTime(null);
        setOwnId(null);
        setSessionKey(null);
        setSessionSecret(null);
        setHasAcceptedDisclaimer(null);
        setDisplayName(null);
    }

    /**
     * Get the users own facebook id.
     * Empty string ("") is returned if id is not set
     *
     * @return the facebook id or "" if not set
     */
    public String getOwnId() {
        return getString(Constants.OWN_ID_PREFS_KEY, "");
    }

    /**
     * Set the own id
     * null will remove the setting
     *
     * @param id the new id
     */
    public void setOwnId(String id) {
        if (id == null) {
            removeSetting(OWN_ID_PREFS_KEY);
        } else {
            setString(OWN_ID_PREFS_KEY, id);
        }
    }

    /**
     * Get the display (real) name of the logged in user
     *
     * @return display name or "" if not set
     */
    public String getDisplayName() {
        return getString(DISPLAY_NAME_KEY, "");
    }

    /**
     * Set the display (real) name of the logged in user
     *
     * @param displayName display name
     */
    public void setDisplayName(String displayName) {
        if (displayName == null) {
            removeSetting(DISPLAY_NAME_KEY);
        } else {
            setString(DISPLAY_NAME_KEY, displayName);
        }
    }

    /**
     * Get the authentication token.
     * If not stored then null is returned
     *
     * @return the authentication token
     */
    public String getAuthenticationToken() {
        return getString(ACCESS_TOKEN_PREFS_KEY, null);
    }

    /**
     * Set the authentication token
     * null will remove the setting
     *
     * @param token the new token
     */
    public void setAuthenticationToken(final String token) {
        if (token == null) {
            removeSetting(ACCESS_TOKEN_PREFS_KEY);
        } else {
            setString(ACCESS_TOKEN_PREFS_KEY, token);
        }
    }

    /**
     * Get the youngest timestamp
     * If not stored then 0 is returned
     *
     * @return the youngest timestamp
     */
    public Long getYoungestTimestamp() {
        return getLong(NEWS_FEED_SINCE_PREFS_KEY, 0L);
    }

    /**
     * Set the youngest timestamp
     * null will remove the setting
     *
     * @param the new value
     */
    public void setYoungestTimestamp(Long newValue) {
        if (newValue == null) {
            removeSetting(NEWS_FEED_SINCE_PREFS_KEY);
        } else {
            setLong(NEWS_FEED_SINCE_PREFS_KEY, newValue);
        }
    }

    /**
     * Set the last communication time
     * null will remove the setting
     *
     * @param time the new time
     */
    public void setLastCommunicationTime(Long time) {
        if (time == null) {
            removeSetting(LAST_COMMUNICATION_WITH_SERVER_KEY);
        } else {
            setLong(LAST_COMMUNICATION_WITH_SERVER_KEY, time);
        }
    }

    /**
     * Get the last communication time
     * Default value value is 0L
     *
     * @return the last time of communication
     */
    public Long getLastCommunicationTime() {
        return getLong(LAST_COMMUNICATION_WITH_SERVER_KEY, 0L);
    }

    /**
     * Set the session key
     *
     * @param sessionKey the new session key
     */
    public void setSessionKey(String sessionKey) {
        if (sessionKey == null) {
            removeSetting(SESSION_KEY);
        } else {
            setString(SESSION_KEY, sessionKey);
        }
    }

    /**
     * Get the session key
     *
     * @return the session key or null if not found
     */
    public String getSessionKey() {
        return getString(SESSION_KEY, null);
    }

    public void setSessionSecret(String sessionSecret) {
        if (sessionSecret == null) {
            removeSetting(SESSION_SECRET);
        } else {
            setString(SESSION_SECRET, sessionSecret);
        }
    }

    public String getSessionSecret() {
        return getString(SESSION_SECRET, null);
    }


    public boolean hasAcceptedDisclaimer() {
        int value = getInteger(HAS_ACCEPTED_DISCLAIMER, 0);

        return (value != 0);
    }

    public void setHasAcceptedDisclaimer(Boolean value) {
        if (value == null) {
            removeSetting(HAS_ACCEPTED_DISCLAIMER);
        } else if (value) {
            setInteger(HAS_ACCEPTED_DISCLAIMER, 1);
        } else {
            setInteger(HAS_ACCEPTED_DISCLAIMER, 0);
        }
    }

    /**
     * Remove a setting
     *
     * @param key the setting to remove
     */
    private void removeSetting(String key) {
        SharedPreferences preferences = mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE);
        preferences.edit().remove(key).commit();
    }

    /**
     * Sets a new string setting
     *
     * @param key the key, null not allowed
     * @param value the new value, null not allowed
     */
    private void setString(String key, String value) {
        SharedPreferences preferences = mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE);

        preferences.edit().putString(key, value).commit();
    }

    /**
     * Get a stored string setting
     *
     * @param key the key for the setting
     * @param defaultValue the default value if key not found
     * @return the setting
     */
    private String getString(String key, String defaultValue) {
        SharedPreferences preferences = mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE);

        return preferences.getString(key, defaultValue);
    }

    /**
     * Store a long setting
     *
     * @param key the key for the setting
     * @param value the value for the setting
     */
    private void setLong(String key, Long value) {
        SharedPreferences preferences = mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE);

        preferences.edit().putLong(key, value).commit();
    }

    /**
     * Get a stored long setting
     *
     * @param key the key for the setting
     * @param defaultValue the default value if key not found
     * @return the setting
     */
    private Long getLong(String key, long defaultValue) {
        return mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE).getLong(key, defaultValue);
    }

    /**
     * Get a stored integer setting
     *
     * @param key
     * @param defaultValue
     * @return
     */
    private Integer getInteger(String key, Integer defaultValue) {
        return mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE).getInt(key, defaultValue);
    }

    /**
     * Set an integer
     * @param key
     * @param value
     */
    private void setInteger(String key, int value) {
        SharedPreferences preferences = mContext.getSharedPreferences(CONFIG_STORE, MODE_PRIVATE);

        preferences.edit().putInt(key, value).commit();
    }
}
