/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.utils;

import android.content.Context;

import twitter4j.DirectMessage;
import twitter4j.PagableResponseList;
import twitter4j.Status;
import twitter4j.User;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterCommunication;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException.StatusCode;
import junit.framework.Assert;

import java.util.List;

public class MockTwitterCommunication extends TwitterCommunication {

    public static final long DEFAULT_PAGING_ID = 10;

    private String mMethodName = "";

    private int mMethodCallCount = 0;

    private boolean mReturnError = false;

    private boolean mPerformThrowExceptionOnCall = false;

    private StatusCode mResponseCode = StatusCode.INTERNAL_ERROR;

    Context mContext;

    private User mOwnUser;

    private List<Status> mStatuses;

    private List<DirectMessage> mMessages;

    private PagableResponseList<User> mUsers;

    public MockTwitterCommunication(Context context) {
        super(context);
    }

    /**
     * Set a method to track for call count.
     * @param name Method name
     */
    public void trackMethodCalls(String methodName) {
        mMethodCallCount = 0;
        mMethodName = methodName;
    }

    /**
     * Get the call count for the set method
     * @return
     */
    public int getMethodCalls() {
        return mMethodCallCount;
    }

    /**
     * Will check if the method is called and increase the call counter
     * @param name
     */
    private void checkTrackedMethodCalled(String name) {
        if (mMethodName.equals(name)) {
            ++mMethodCallCount;
        }
    }

    /**
     * Will set is null should be returned when Twitter is called.
     * If throw exception is set it has higher prio.
     * @param returnNull
     */
    public void returnErrorOnCall(boolean returnError) {
        mReturnError = returnError;
    }

    /**
     * Set if exception should be thrown or not when called
     * @param throwException
     * @param code
     */
    public void performThrowExceptionOnCall(boolean throwException, StatusCode code) {
        mPerformThrowExceptionOnCall = throwException;
        mResponseCode = code;
    }

    /**
     * Will check if exception should be thrown
     * @throws TwitterPluginException
     */
    public void throwTwitterPluginException() throws TwitterPluginException {
        if (mPerformThrowExceptionOnCall) {
            throw new TwitterPluginException(mResponseCode);
        }
    }

    @Override
    public boolean authenticateTwitterAccount(String userName, String password)
            throws TwitterPluginException {
        checkTrackedMethodCalled("authenticateTwitterAccount");
        throwTwitterPluginException();

        if (userName == null || password == null || userName.length() == 0 || password.length() == 0) {
            return false;
        }
        if (mReturnError) {
            return false;
        }
        return true;
    }

    @Override
    public void updateStatus(String status) throws TwitterPluginException {
        checkTrackedMethodCalled("updateStatus");
        throwTwitterPluginException();
        Assert.assertNotNull(status);
    }

    @Override
    public void setPagingSinceForStatus(List<Status> statuses) {
        checkTrackedMethodCalled("setPagingSinceForStatus");
        Assert.assertNotNull(statuses);
    }


    @Override
    public void setPagingSinceForMessages(List<DirectMessage> messages) {
        checkTrackedMethodCalled("setPagingSinceForMessages");
        Assert.assertNotNull(messages);
    }

    @Override
    public List<Status> getHomeTimeline() throws TwitterPluginException {
        checkTrackedMethodCalled("getHomeTimeline");
        throwTwitterPluginException();

        if (mReturnError) {
            return null;
        }
        return mStatuses;
    }

    @Override
    public List<DirectMessage> getDirectMessages() throws TwitterPluginException {
        checkTrackedMethodCalled("getDirectMessages");
        throwTwitterPluginException();

        if (mReturnError) {
            return null;
        }
        return mMessages;
    }

    @Override
    public PagableResponseList<User> getFriendsList() throws TwitterPluginException {
        checkTrackedMethodCalled("getFriendsList");
        throwTwitterPluginException();

        if (mReturnError) {
            return null;
        }
        return mUsers;
    }

    @Override
    public void logout() {
        checkTrackedMethodCalled("logout");
    }

    @Override
    public User getOwnUser() throws TwitterPluginException {
        checkTrackedMethodCalled("getOwnUser");
        throwTwitterPluginException();

        if (mReturnError) {
            return null;
        }
        return mOwnUser;
    }

    /**
     * Set the own user to mock the calls
     * @param ownUser
     */
    public void setOwnUser(User ownUser) {
        mOwnUser = ownUser;
    }

    /**
     * Set the statuses to mock the calls
     * @param statuses
     */
    public void setStatuses(List<Status> statuses) {
        mStatuses = statuses;
    }

    /**
     * Set the messages to mock the call
     * @param messages
     */
    public void setMessages(List<DirectMessage> messages) {
        mMessages = messages;
    }

    /**
     * Set the message and status users
     * @param users
     */
    public void setFriends(PagableResponseList<User> users) {
        mUsers =  users;
    }
}
