/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;

import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.utility.NotificationHelper;

import junit.framework.Assert;

import org.easymock.EasyMock;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.api.easymock.PowerMock;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;


@RunWith(PowerMockRunner.class)
@PrepareForTest({NotificationHelper.class, NotificationManager.class,
    Notification.class, PendingIntent.class, Context.class, System.class})
public class TestNotificationHelper {

    private static final int NOTIFICATION_ID = 1;

    private static final String title = "Title";
    private static final String message = "Message";
    private static final String error = "Error";

    /**
     * Test set up
     */
    @Before
    public void setUp() {

    }

    /**
     * Test to set notification
     * @throws Exception
     */
    @Test
    public void testSetNotification() throws Exception {

        // Mock objects
        Context context = PowerMock.createMock(Context.class);
        NotificationManager nm = PowerMock.createMock(NotificationManager.class);
        Notification notification = PowerMock.createMock(Notification.class);
        PowerMock.mockStatic(PendingIntent.class);
        PowerMock.mockStatic(System.class);
        PendingIntent intent = PowerMock.createMock(PendingIntent.class);

        // Expect calls
        long time = 0;
        EasyMock.expect(System.currentTimeMillis()).andReturn(time).once();
        try {
            PowerMock.expectNew(Notification.class, 0, null, time).andReturn(notification).once();
        } catch (Exception e) {
            Assert.fail("Could not create notification class: " + e.getMessage());
        }

        EasyMock.expect(context.getSystemService(Context.NOTIFICATION_SERVICE)).andReturn(nm).once();
        EasyMock.expect(PendingIntent.getActivity(context, 0, null, Intent.FLAG_ACTIVITY_NEW_TASK)).andReturn(intent).once();

        notification.setLatestEventInfo(context, title, message, intent);
        nm.notify(NOTIFICATION_ID, notification);
        PowerMock.expectLastCall().once();

        PowerMock.replayAll();

        NotificationHelper.notifyInStatusBar(context, 0, title, message);

        PowerMock.verifyAll();
    }

    /**
     * Test to set invalid notification
     * @throws Exception
     */
    @Test
    public void testSetInvalidNotification()  throws Exception {
        Context context = PowerMock.createMock(Context.class);
        NotificationHelper.notifyInStatusBar(context, 0, null, message);
        NotificationHelper.notifyInStatusBar(context, 0, title, null);
        NotificationHelper.notifyInStatusBar(null, 0, title, message);
    }

    /**
     * Set a error notification
     * @throws Exception
     */
    @Test
    public void testSetErrorNotification()  throws Exception {
        // Mock objects
        Context context = PowerMock.createMock(Context.class);
        Resources resource = PowerMock.createMock(Resources.class);

        try {
            PowerMock.mockStatic(NotificationHelper.class, NotificationHelper.class.getMethod(
                    "removeFromStatusBar", Context.class), NotificationHelper.class.getMethod(
                    "notifyInStatusBar", Context.class, int.class, String.class, String.class));
        } catch (SecurityException e) {
            Assert.fail("SecurityException: " + e.getMessage());
        } catch (NoSuchMethodException e) {
            Assert.fail("NoSuchMethodException: " + e.getMessage());
        }

        NotificationHelper.removeFromStatusBar(context);
        PowerMock.expectLastCall().once();

        EasyMock.expect(context.getResources()).andReturn(resource).once();
        EasyMock.expect(resource.getString(R.string.ts_twitter_notify_fetch_failed_title_txt))
                .andReturn(title).once();

        NotificationHelper.notifyInStatusBar(context, R.drawable.notif_data_fetch_failed_icn,
                title, error);
        PowerMock.expectLastCall().once();

        PowerMock.replayAll();
        NotificationHelper.notifyError(context, error);
        PowerMock.verifyAll();
    }

    /**
     * Test to set a invalid notification
     * @throws Exception
     */
    @Test
    public void testSetInvalidErrorNotification()  throws Exception {
        Context context = PowerMock.createMock(Context.class);
        NotificationHelper.notifyError(context, null);
        NotificationHelper.notifyError(null, message);
    }
    /**
     * Test to remove the notification
     * @throws Exception
     */
    @Test
    public void testRemoveNotification()  throws Exception {
        // Mock objects
        Context context = PowerMock.createMock(Context.class);
        NotificationManager nm = PowerMock.createMock(NotificationManager.class);

        EasyMock.expect(context.getSystemService(Context.NOTIFICATION_SERVICE)).andReturn(nm).once();
        nm.cancel(NOTIFICATION_ID);
        PowerMock.expectLastCall().once();

        PowerMock.replayAll();
        NotificationHelper.removeFromStatusBar(context);
        PowerMock.verifyAll();
    }
}
