/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/
package com.sonyericsson.eventstream.twitterplugin.twitter;

import android.content.Context;
import android.util.Log;

import twitter4j.TwitterException;

import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;

@SuppressWarnings("serial")
public class TwitterPluginException extends Exception {

    public enum StatusCode {
        NOT_MODIFIED,
        INTERNAL_ERROR,
        UNAUTHORIZED,
        SERVER_UNAVAILABLE
    }

    private StatusCode mExceptionValue = StatusCode.UNAUTHORIZED;

    public TwitterPluginException(TwitterException e) {
        mExceptionValue = translateErrorCause(e);
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "exception value: " + mExceptionValue.name());
        }
    }

    public TwitterPluginException(StatusCode code) {
        if (code != null) {
            mExceptionValue = code;
        } else {
            mExceptionValue = StatusCode.INTERNAL_ERROR;
        }
    }

    /**
     * Return the status code, ERRORCAUSE_...
     * @return the error code ERRORCAUSE_OK or other
     */
    public StatusCode getStatusCode() {
        return mExceptionValue;
    }

    /**
     * Will translate a StatusCode into a notification message
     * @param context The Context
     * @param code the StatusCode to translate
     * @return String if exist else null
     */
    public String getStatusCodeNotificationMessage(Context context) {
        if (context == null) {
            return null;
        }

        switch (mExceptionValue) {
            case INTERNAL_ERROR:
                return context.getString(R.string.ts_twitter_notify_fetch_failed_operation_failed_txt);

            case NOT_MODIFIED:
                return null;

            case SERVER_UNAVAILABLE:
                return context.getString(R.string.ts_twitter_notify_fetch_failed_servererror_txt);

            case UNAUTHORIZED:
                return context.getString(R.string.ts_twitter_notify_fetch_failed_invalid_access_txt);

            default:
                return null;
        }
    }

    /**
     * Will translate the TwitterException into internal ERRORCAUSE_
     * @param e The twitter exception to translate
     * @return ErrorCause.ERRORCAUSE_UNAUTHORIZED or other
     */
    private StatusCode translateErrorCause(TwitterException e) {
        if (e == null) {
            return StatusCode.INTERNAL_ERROR;
        }

        switch (e.getStatusCode()) {
            case TwitterException.BAD_REQUEST:
            case TwitterException.FORBIDDEN:
            case TwitterException.NOT_ACCEPTABLE:
            case TwitterException.NOT_FOUND:
            case TwitterException.OK:
                return StatusCode.INTERNAL_ERROR;

            case TwitterException.ENHANCE_YOUR_CLAIM:
            case TwitterException.BAD_GATEWAY:
            case TwitterException.INTERNAL_SERVER_ERROR:
            case TwitterException.SERVICE_UNAVAILABLE:
                return StatusCode.SERVER_UNAVAILABLE;

            case TwitterException.UNAUTHORIZED:
                return StatusCode.UNAUTHORIZED;

            case TwitterException.NOT_MODIFIED:
                return StatusCode.NOT_MODIFIED;

            default:
                return StatusCode.INTERNAL_ERROR;
        }
    }
}
