/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.service;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventStreamIntentActions;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventStreamIntentExtraParams;


/**
 * This BroadcastReceiver listens for incoming broadcasts from Event Stream.
 */
public class EventStreamListener extends BroadcastReceiver {

    private static final int SCREEN_NAME_INDEX = 0;
    public static final String TWITTER_STATUS_VIEW_URI = "http://mobile.twitter.com/";

    @Override
    public void onReceive(Context context, Intent intent) {
        launchTwitterService(context, intent);
    }

    public void launchTwitterService(Context context, Intent intent) {
        if (context == null) {
            return;
        }

        // Check if the intent is ok
        if (!validateIntent(intent)) {
            return;
        }

        String action = intent.getAction();
        Intent serviceIntent = new Intent();
        if (EventStreamIntentActions.REFRESH.equals(action)) {
            serviceIntent.setAction(PluginConstants.TWITTER_REFRESH_REQUEST);
        } else if (EventStreamIntentActions.REGISTER.equals(action)) {
            serviceIntent.setAction(PluginConstants.TWITTER_REGISTER_PLUGIN);
        } else if (EventStreamIntentActions.VIEW_EVENT.equals(action)) {
            launchWebIntent(context, intent);
        } else if (EventStreamIntentActions.STATUS_UPDATE.equals(action)) {
            String statusMessage = intent.getExtras().getString(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE);
            if (statusMessage != null && statusMessage.length() > 0) {
                serviceIntent.setAction(PluginConstants.TWITTER_UPDATE_STATUS);
                serviceIntent.putExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE, statusMessage);
            }
        }
        context.startService(serviceIntent);
    }

    /**
     * Check if the sent intent is valid, also need to contain extra plug-in key
     * and the value need to match the internally stored key.
     * @param intent The intent sent
     * @return true if valid else false
     */
    protected boolean validateIntent(Intent intent) {
        if (intent == null) {
            return false;
        }

        String action = intent.getAction();
        if (action == null) {
            return false;
        }

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Twitter service intent: " + action);
        }

        // Only check the key if not registration is sent
        if (EventStreamIntentActions.REGISTER.equals(action)) {
            return true;
        } else {
            Bundle extras = intent.getExtras();
            if (extras != null) {
                String key = extras.getString(EventStreamIntentExtraParams.EXTRA_PLUGIN_KEY);
                if (key != null) {
                    return key.equals(PluginConstants.PLUGIN_IDENTIFIER);
                }
            }
        }

        return false;
    }

    /**
     * Function to launch the web browser via intent to show Twitter events
     * @param context The context
     * @param intent The intent from the service
     */
    private void launchWebIntent(Context context, Intent intent) {
        Bundle bundle = intent.getExtras();
        String screenName = bundle.getString(EventStreamIntentExtraParams.EXTRA_EVENT_KEY);
        String[] eventKeyData = screenName.split(PluginConstants.EVENT_KEY_DATA_SEPARATOR);
        if (eventKeyData != null && eventKeyData[SCREEN_NAME_INDEX] != null) {
            Uri uri = Uri.withAppendedPath(Uri.parse(TWITTER_STATUS_VIEW_URI), eventKeyData[SCREEN_NAME_INDEX]);
            Intent webViewIntent = new Intent(Intent.ACTION_VIEW, uri);
            webViewIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(webViewIntent);
        }
    }
}
