/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.testplugin;

import android.app.Service;
import android.content.ContentValues;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.IBinder;
import android.util.Config;
import android.util.Log;

import com.sonyericsson.eventstream.provider.EventStreamProvider;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.Timer;
import java.util.TimerTask;

public class GenerateContentService extends Service {

    public static final String TESTPLUGINSOURCE_ACTION_GENERATE_CONTENT = "com.sonyericsson.eventstream.testplugin.GENERATE_CONTENT";

    private static final String LOG_TAG = "TestPlugin[Generate]";

    // Counter for id of events written to Event Stream.
    private long mEventId = 0;

    private static final String PREF_NAME = "TestPluginAlpha";

    private static final String EVENT_ID_PREF_KEY = "EventId-";

    private SharedPreferences mPrefs;

    private final ArrayList<Timer> mTimers = new ArrayList<Timer>();

    private String[] mContentArray;

    private static final String appResUri = "android.resource://com.sonyericsson.eventstream.testplugin/";

    private int mMsgNum = 0;

    // @TODO Test odd characters.

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        int return_value = super.onStartCommand(intent, flags, startId);

        if (Config.DEBUG) {
            Log.i(LOG_TAG, "onStartCommand called");
        }

        if (intent == null) {
            if (Config.DEBUG) {
                Log.i(LOG_TAG, "ERROR! Empty intent!");
            }
            return -1;
        }

        // Pull out all the drawables from the resource folder and store their
        // identifiers in an array
        mContentArray = new String[10];
        mContentArray[0] = appResUri + Integer.toString(R.drawable.android);
        mContentArray[1] = appResUri + Integer.toString(R.drawable.basketball);
        mContentArray[2] = appResUri + Integer.toString(R.drawable.duke);
        mContentArray[3] = appResUri + Integer.toString(R.drawable.firefox_logo);
        mContentArray[4] = "http://baconhaikus.files.wordpress.com/2008/12/kevin_bacon.jpg?access_token=55772661571|1174491402945fc7283b5eec-604262514|dJQDC28yT_Et-SbUurjgtF7tSOk.&type=large";
        mContentArray[5] = "http://blogs.smh.com.au/gadgetsonthego/android.jpg";
        mContentArray[6] = "http://stuffthathappens.com/blog/wp-content/uploads/2008/03/android-case-mod.jpg";
        mContentArray[7] = "http://www.hemagazine.com/files/Howard%20stringerand3D.jpg";
        mContentArray[8] = "http://images.intomobile.com/wp-content/uploads/2010/09/bert-nordberg-sony-ericsson-ceo1.jpg";
        mContentArray[9] = "http://cdn.slashgear.com/wp-content/uploads/2010/02/18feb10sonyericssogb820.jpg";

        // Read event counter value from stored data.
        mPrefs = this.getSharedPreferences(PREF_NAME, 0);
        mEventId = mPrefs.getLong(EVENT_ID_PREF_KEY, 0);

        if (TESTPLUGINSOURCE_ACTION_GENERATE_CONTENT.equals(intent.getAction())) {
            // Generate "hello" event and setup task and timer for
            // generating more events for this Event Stream source.
            long default_source_id = -100;
            long source_id = intent.getLongExtra("source_id", default_source_id);

            int default_start_code = -101;
            int start_code = intent.getIntExtra("start_code", default_start_code);

            if (source_id != default_source_id) {
                if (start_code == 2) {
                    // A source was just added, so add the "Hello, World!" event
                    if (Config.DEBUG) {
                        Log.i(LOG_TAG, "New auto source added (Total: " + (mTimers.size() + 1)
                                + "). Adding default 1st event...");
                    }
                    addHelloEvent((int)source_id);
                }

                // Set up timer for next auto generated event.
                Timer timer = new Timer(Integer.toString((int)source_id), true);
                ScheduleTimerTask scheduleTask = new ScheduleTimerTask((int)source_id, timer);
                timer.scheduleAtFixedRate(scheduleTask, 5000, 10000);
                mTimers.add(timer);
            }
        }
        return return_value;
    }

    /**
     * We'll have zero or one task per source. When timer expires this task will
     * insert event into Event Stream provider. It will collect the data for the
     * event from predefined lists.
     *
     * @author 23007672
     */
    class ScheduleTimerTask extends TimerTask {

        // This task "belongs" to this source.
        private int mSourceId = 0;

        // Observe! Task and timer have circular reference.
        // When task fails we need to remove "its" timer from
        // list ("mTimers") in parent class ("GenerateContentService").
        private Timer mTimer = null;

        // Each auto-source will loop through
        // list of messages, friends and content.
        private int msgIndex = 0;

        private int friendIndex = 0;

        private int contentIndex = 0;

        // Constructor
        ScheduleTimerTask(int sourceId, Timer timer) {
            super();
            mSourceId = sourceId;
            mTimer = timer;
        }

        @Override
        public void run() {
            // Generate content and add an event.
            if (!addEvent(mSourceId)) {
                // Adding events no longer works.
                // (Source and/or plug-in was maybe removed from provider?)
                // Abort this task and timer.
                // Find timer in list.
                Iterator<Timer> it = mTimers.iterator();
                while (it.hasNext()) {
                    Timer t = it.next();
                    if (t.equals(mTimer)) {
                        if (Config.DEBUG) {
                            Log.d(LOG_TAG, "Event task failed. Abort next task (and timer).");
                        }
                        it.remove();
                        mTimer = null;
                        this.cancel();
                        t.cancel();
                        break;
                    }
                }
            }
        }

        /**
         * Add event with info from predefined lists.
         *
         * @TODO Bulk insert test.
         * @param sourceId Sender id to be use in insert.
         * @return true on success else false.
         */
        private boolean addEvent(int sourceId) {
            boolean result = false;
            String backgroundImageUri = "";

            ContentValues values = new ContentValues();
            values.put(EventStreamProvider.Event.COLUMN_SOURCE_ID, sourceId);
            values.put(EventStreamProvider.Event.COLUMN_PUBLISHED_TIME, new Date().getTime());

            // TODO: this can be improved by randomly setting these values as
            // well to test
            // the queries made by the UI.
            values.put(EventStreamProvider.Event.COLUMN_OUTGOING, 0);
            values.put(EventStreamProvider.Event.COLUMN_PERSONAL, 0);

            values.put(EventStreamProvider.Event.COLUMN_STATUS_ICON_URI,
                    "android.resource://com.sonyericsson.eventstream.testplugin/"
                            + Integer.toString(R.drawable.status_icon));

            // Yes, inefficient to get entire array each time.
            String[] msgArray = getResources().getStringArray(R.array.msgArray);
            msgIndex = (msgIndex + 1) % msgArray.length;
            values.put(EventStreamProvider.Event.COLUMN_MESSAGE, msgArray[msgIndex]);
            // Left for easy debug :)

            // values.put(EventStreamProvider.Event.COLUMN_MESSAGE, "Msg no: " +
            // mMsgNum);
            // System.out.println("Msg no: " + mMsgNum);
            // mMsgNum++;
            contentIndex = (contentIndex + 1) % mContentArray.length;
            backgroundImageUri = mContentArray[contentIndex];
            if (Config.DEBUG) {
                Log.i(LOG_TAG, "Drawable name retrieved " + backgroundImageUri);
            }
            values.put(EventStreamProvider.Event.COLUMN_IMAGE_URI, backgroundImageUri);

            String[] friendArray = getResources().getStringArray(R.array.friendArray);
            friendIndex = (friendIndex + 1) % friendArray.length;
            values.put(EventStreamProvider.Event.COLUMN_TITLE, friendArray[friendIndex]);

            Uri uri = null;
            try {
                uri = getContentResolver().insert(EventStreamProvider.URIs.EVENTS, values);
            } catch (IllegalArgumentException e) {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Insert event caused exception: " + e.getMessage());
                }
                uri = null;
            }

            if (uri != null) {
                result = true;
                mEventId++;
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Auto-source (id: " + sourceId
                            + ") added event to Event Stream.");
                }
            } else {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Event was not added (from source: " + sourceId + ")");
                }
            }

            return result;
        }
    }

    /**
     * Add a default "Hello, World!" event for source.
     */
    private void addHelloEvent(int sourceId) {
        String backgroundImageUri = appResUri + Integer.toString(R.drawable.android);
        ContentValues values = new ContentValues();
        values.put(EventStreamProvider.Event.COLUMN_SOURCE_ID, sourceId);
        values.put(EventStreamProvider.Event.COLUMN_MESSAGE, "Hello, I'm source no " + sourceId
                + " from test plug-in.");
        values.put(EventStreamProvider.Event.COLUMN_TITLE, "System");
        values.put(EventStreamProvider.Event.COLUMN_PUBLISHED_TIME, new Date().getTime());
        values.put(EventStreamProvider.Event.COLUMN_IMAGE_URI, backgroundImageUri);
        values.put(EventStreamProvider.Event.COLUMN_OUTGOING, 0);
        values.put(EventStreamProvider.Event.COLUMN_PERSONAL, 0);
        values.put(EventStreamProvider.Event.COLUMN_STATUS_ICON_URI,
                "android.resource://com.sonyericsson.eventstream.testplugin/"
                        + Integer.toString(R.drawable.status_icon));

        getContentResolver().insert(EventStreamProvider.URIs.EVENTS, values);

        mEventId++;

        if (Config.DEBUG) {
            Log.i(LOG_TAG, "Source " + sourceId + " added Hello World event to Event Stream.");
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        // Remove timers
        Iterator<Timer> it = mTimers.iterator();
        while (it.hasNext()) {
            Timer t = it.next();
            t.cancel();
            it.remove();
        }
        // Save internal incremented event id.
        SharedPreferences.Editor pe = mPrefs.edit();
        pe.putLong(EVENT_ID_PREF_KEY, mEventId);
        pe.commit();
    }

}
