/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.telephonyplugin;

import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.ServiceIntentCmd;
import com.sonyericsson.eventstream.telephonyplugin.PluginConstants.Telephony;

import android.app.Service;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.ContentObserver;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.net.Uri.Builder;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.IBinder;
import android.provider.ContactsContract;
import android.provider.ContactsContract.Contacts;
import android.provider.ContactsContract.PhoneLookup;
import android.util.Log;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

/**
 * @auth Erik Hellman <erik.hellman@sonyericsson.com>
 */
public class TelephonyPluginService extends Service {

    /** Content observer for messaging content */
    protected MessagingContentObserver mMessageContentObserver;

    /** Content observer for contact content */
    protected ContactsContentObserver mContactsContentObserver;

    /** Value of a invalid source id */
    private static final int INVALID_SOURCE_ID = -1;

    /** Keeps the handler thread */
    private HandlerThread mHandlerThread;

    /** Keeps the Handler */
    private Handler mHandler;

    /** Keeps a helper to handle data from the sources */
    private static OriginatedSourceHelper mSourceHelper = new OriginatedSourceHelper();

    private static final int BULK_INSERT_MAX_COUNT = 50;

    private static final int BULK_INSERT_DELAY = 20; //ms

    @Override
    public void onCreate() {
        super.onCreate();
        mHandlerThread = new HandlerThread("TelephonyPluginHandler");
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper());
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        int status = super.onStartCommand(intent, flags, startId);
        if (intent != null && intent.getExtras() != null) {

            String serviceCommand = intent.getExtras().getString(ServiceIntentCmd.SERVICE_COMMAND_KEY);
            if (serviceCommand != null) {

                if (ServiceIntentCmd.REFRESH_REQUEST.equals(serviceCommand)) {
                    Runnable runnable = new Runnable() {
                        public void run() {
                            int sourceId = getSourceId();
                            if (sourceId != INVALID_SOURCE_ID) {
                                syncTelephonyAndEventStream(sourceId);
                            }
                        }
                    };
                    mHandler.post(runnable);
                } else if (ServiceIntentCmd.REGISTER_PLUGIN.equals(serviceCommand)) {
                    Runnable runnable = new Runnable() {
                        public void run() {
                            if (pluginRegistration()) {
                                // Only perform sync the first registration
                                int sourceId = getSourceId();
                                if (sourceId != INVALID_SOURCE_ID) {
                                    syncTelephonyAndEventStream(sourceId);
                                }
                            }
                        }
                    };
                    mHandler.post(runnable);
                } else if (ServiceIntentCmd.UPDATE_FRIENDS.equals(serviceCommand)) {
                    Runnable runnable = new Runnable() {
                        public void run() {
                            int sourceId = getSourceId();
                            if (sourceId != INVALID_SOURCE_ID) {
                                updateFriends(sourceId);
                            }
                        }
                    };
                    mHandler.post(runnable);
                } else if (ServiceIntentCmd.VIEW_EVENT.equals(serviceCommand)) {
                    String friendKey = intent.getExtras().getString(EventStream.EVENTSTREAM_VIEW_INTENT_FRIEND_KEY_DATA);
                    if (friendKey != null) {
                        launchMessageIntent(this, friendKey);
                    }
                } else if (ServiceIntentCmd.LOCALE_CHANGED.equals(serviceCommand)) {
                    Runnable runnable = new Runnable() {
                        public void run() {
                            handleLocaleChanged();
                        }
                    };
                    mHandler.post(runnable);
                }
            }
        }

        // Register Conversation content observer
        if (mMessageContentObserver == null) {
            mMessageContentObserver = new MessagingContentObserver(new Handler());
        }

        // Register Contacts content observer
        if (mContactsContentObserver == null) {
            mContactsContentObserver = new ContactsContentObserver(new Handler());
        }

        getContentResolver().unregisterContentObserver(mMessageContentObserver);
        getContentResolver().unregisterContentObserver(mContactsContentObserver);

        getContentResolver().registerContentObserver(Telephony.TELEPHONY_MMS_SMS_URI, true,
                mMessageContentObserver);
        getContentResolver().registerContentObserver(Contacts.CONTENT_URI, true,
                mContactsContentObserver);

        return status;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (mContactsContentObserver != null) {
            getContentResolver().unregisterContentObserver(mContactsContentObserver);
            mContactsContentObserver.closeExistingBulkTimer();
        }

        if (mMessageContentObserver != null) {
            getContentResolver().unregisterContentObserver(mMessageContentObserver);
            mMessageContentObserver.closeExistingBulkTimer();
        }
    }

    /**
     * Retrieve the source id if it exist
     *
     * @return The source id if exist else -1
     */
    protected int getSourceId() {
        int sourceId = INVALID_SOURCE_ID;
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI,
                    new String[] {
                        EventStream.SourceColumns._ID
                    }, null, null, null);
            if (cursor != null && cursor.moveToFirst() && cursor.getCount() > 0) {
                sourceId = cursor.getInt(cursor
                        .getColumnIndexOrThrow(EventStream.SourceColumns._ID));
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return sourceId;
    }

    /**
     * Will update plugin name and source name when locale is changed
     * (for some reason, plugin name is called application name and source name
     * is called plugin name in the resource file for this plugin...)
     */
    private void handleLocaleChanged() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Handling locale change.");
        }
        String pluginName = getResources().getString(
                    R.string.ts_messaging_application_name);

        String sourceName = getResources().getString(
                    R.string.ts_messaging_plugin_name);

        ContentValues values = new ContentValues();
        values.put(EventStream.PluginColumns.NAME,pluginName);

        setLocaleColumns(pluginName,sourceName);

        try {
            int result = getContentResolver().update(
                    EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                    values,
                    null,
                    null);

            if (result != 1) {
                Log.e(Config.LOG_TAG, "Failed to update plugin name");
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update plugin name");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update plugin name");
            throw new RuntimeException(exception);
        }

        values.put(EventStream.SourceColumns.NAME, sourceName);

        try {
            int result = getContentResolver().update(
                    EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI,
                    values,
                    EventStream.SourceColumns._ID + " = " + getSourceId(),
                    null);

            if (result != 1) {
                Log.e(Config.LOG_TAG, "Failed to update source name");
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update source name");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update source name");
            throw new RuntimeException(exception);
        }
    }

    /**
     * Will perform a new plug-in registration.
     * @return true if the plug-in is registered for the first time
     */
    public boolean pluginRegistration() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Start refreshing plugin registration.");
        }
        boolean firstTimeRegister = false;

        ContentResolver contentResolver = getContentResolver();
        Cursor pluginCursor = null;

        ContentValues regValues = new ContentValues();
        regValues.put(EventStream.PluginColumns.NAME, getResources().getString(
                R.string.ts_messaging_plugin_name));

        Builder iconUriBuilder = new Uri.Builder().scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
                .authority(getPackageName()).appendPath(Integer.toString(R.drawable.icon));
        regValues.put(EventStream.PluginColumns.ICON_URI, iconUriBuilder.toString());

        regValues.put(EventStream.PluginColumns.API_VERSION,
                EventStream.EVENTSTREAM_PLUGIN_VERSION);

        regValues.put(EventStream.PluginColumns.CONFIGURATION_STATE,
                EventStream.EVENTSTREAM_CONFIGURATION_STATE);

        try {
            pluginCursor = contentResolver.query(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null,
                    null, null, null);
            // Check for previous registration
            if (pluginCursor != null && pluginCursor.moveToNext()) {
                int updated = contentResolver.update(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                        regValues, null, null);
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Plugin registration updated: " + (updated == 1));
                }
            } else { // First time registration
                Uri uri = contentResolver.insert(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                        regValues);
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Plugin registration created: " + uri);
                }

                firstTimeRegister = true;
            }
        } finally {
            if (pluginCursor != null) {
                pluginCursor.close();
            }
        }

        ContentValues smsSourceValues = new ContentValues();

        smsSourceValues.put(EventStream.SourceColumns.NAME, getResources().getString(
                R.string.ts_messaging_plugin_name));
        smsSourceValues.put(EventStream.SourceColumns.ICON_URI, iconUriBuilder.toString());

        smsSourceValues.put(EventStream.SourceColumns.LAYOUT_ORDER, 1);

        // Create or update the plug-in source
        int sourceId = getSourceId();
        if (sourceId == INVALID_SOURCE_ID) {
            smsSourceValues.put(EventStream.SourceColumns.ENABLED, 1);
            Uri smsServiceUri = contentResolver.insert(
                    EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI, smsSourceValues);

            sourceId = Integer.parseInt(smsServiceUri.getLastPathSegment());
            if (Config.DEBUG || sourceId < 0) {
                Log.d(Config.LOG_TAG, "Source registration created: " + sourceId);
            }
        } else {
            int updated = contentResolver.update(EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI, smsSourceValues,
                    "_id = " + sourceId, null);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Source registration updated: " + (updated == 1));
            }
        }

        return firstTimeRegister;
    }

    /**
     * Synchronize the messaging content with the event stream engine.
     */
    public void syncTelephonyAndEventStream(int sourceId) {
        ContentResolver contentResolver = getContentResolver();

        // Get all message events from the engine
        Map<Integer, Event> events = getEventStreamMessagingEvents();


        // Sync SMS events
        Cursor smsCursor = null;
        try {
            smsCursor = mSourceHelper.getSourceMessages(this, null);
            if (smsCursor != null && smsCursor.moveToFirst()) {
                // Create a bulk list of content values
                ArrayList<ContentValues> bulkValues = new ArrayList<ContentValues>();

                do {
                    int smsId = smsCursor.getInt(smsCursor.getColumnIndexOrThrow(
                            Telephony.Sms.COLUMN_ID));
                    Event event = events.get(smsId);
                    if (event != null) {
                        // Remove this event from our list of known events.
                        events.remove(smsId);
                    } else {
                        // New event - insert into engine
                        ContentValues values = new ContentValues();

                        String address = smsCursor.getString(smsCursor
                                .getColumnIndexOrThrow(Telephony.Sms.COLUMN_ADDRESS));
                        String eventText = mSourceHelper.getMessageText(
                                smsCursor, OriginatedSourceHelper.MESSAGE_TYPE_SMS);
                        Long published = smsCursor.getLong(smsCursor
                                .getColumnIndexOrThrow(Telephony.Sms.COLUMN_DATE));
                        int messageType = smsCursor.getInt(smsCursor
                                .getColumnIndexOrThrow(Telephony.Sms.COLUMN_TYPE));

                        // Check which message type this is, support inbox/outbox.
                        if (messageType == Telephony.Sms.MESSAGE_TYPE_INBOX) {
                            values.put(EventStream.EventColumns.OUTGOING, 0);
                            values.put(EventStream.EventColumns.LAYOUT_TYPE, PluginConstants.LayoutType.RECEIVED);
                            values.put(EventStream.EventColumns.PERSONAL, 1);
                        } else if (messageType == Telephony.Sms.MESSAGE_TYPE_SENT) {
                            values.put(EventStream.EventColumns.OUTGOING, 1);
                            values.put(EventStream.EventColumns.LAYOUT_TYPE, PluginConstants.LayoutType.SENT);
                            values.put(EventStream.EventColumns.PERSONAL, 0);
                        } else {
                            continue;
                        }

                        String friendKey = insertOrUpdateFriend(address, sourceId);
                        if (friendKey == null) {
                            continue;
                        }

                        values.put(EventStream.EventColumns.FRIEND_KEY, friendKey);
                        values.put(EventStream.EventColumns.SOURCE_ID, sourceId);
                        values.put(EventStream.EventColumns.EVENT_KEY, String.valueOf(smsId));
                        values.put(EventStream.EventColumns.MESSAGE, eventText);
                        values.put(EventStream.EventColumns.PUBLISHED_TIME, published);

                        // Add the values to the bulk list
                        bulkValues.add(values);
                    }
                } while (smsCursor.moveToNext());

                // Perform bulk insert
                List<ContentValues> bulkInsertValues = new ArrayList<ContentValues>();
                for (int i = 0; i < bulkValues.size(); i++) {
                    bulkInsertValues.add(bulkValues.get(i));

                    if (bulkInsertValues.size() >= BULK_INSERT_MAX_COUNT) {
                          contentResolver.bulkInsert(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                                (ContentValues[])bulkInsertValues.toArray(new ContentValues[bulkInsertValues
                            .size()]));
                        bulkInsertValues.clear();
                        // Give eventstream some time to run queries
                         try {
                           Thread.sleep(BULK_INSERT_DELAY);
                        } catch (InterruptedException e) {
                            // Do nothing
                         }
                     }
                }

                if (bulkInsertValues.size() > 0) {
                 contentResolver.bulkInsert(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                        (ContentValues[])bulkInsertValues.toArray(new ContentValues[bulkInsertValues.size()]));
                } else {
                    if (Config.DEBUG) {
                        Log.d(Config.LOG_TAG, "syncTelephonyAndEventStream cursor invalid");
                    }
                }
             }
        } finally {
            if (smsCursor != null) {
                smsCursor.close();
            }
        }

        // Delete remaining events in EventStream event table and Friend table
        for (Event event : events.values()) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Delete event: " + event.id);
            }
            contentResolver.delete(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    EventStream.EventColumns._ID + " = " + event.id, null);

            // Clean up if no event is mapped to the friend key
            if (!isFriendKeyAvailableInEvents(event.friend_key)) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Delete friend: " + event.friend_key);
                }
                contentResolver.delete(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                        EventStream.FriendColumns.FRIEND_KEY + " = " + "'" + event.friend_key + "'", null);
            }
        }
    }

    /**
     * Will check if a friend id is available in the events
     *
     * @param friendId The id to check for
     * @return true if available else false
     */
    private boolean isFriendKeyAvailableInEvents(String friendKey) {
        Cursor cursor = null;
        boolean available = false;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    new String[] {
                        EventStream.EventColumns.FRIEND_KEY
                    }, EventStream.EventColumns.FRIEND_KEY + " = " + "'" + friendKey + "'", null, null);
            if (cursor != null && cursor.getCount() > 0) {
                available = true;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return available;
    }

    /**
     * Update the friends table with the latest data from the contacts database
     */
    private void updateFriends(int sourceId) {
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    new String[] {
                        EventStream.FriendColumns.FRIEND_KEY
                    }, null, null, null);

            if (cursor != null && cursor.moveToFirst() && cursor.getCount() > 0) {
                do {
                    String address = cursor.getString(cursor
                            .getColumnIndexOrThrow(EventStream.FriendColumns.FRIEND_KEY));
                    if (address != null) {
                        insertOrUpdateFriend(address, sourceId);
                    }
                } while (cursor.moveToNext());
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Inserts or updates the friend table with the mapped address
     *
     * @param phoneNumber The address of the new friend
     * @return the friend key else null in case of error
     */
    private String insertOrUpdateFriend(String address, int sourceId) {
        if (address == null || address.length() == 0) {
            return null;
        }
        String friendKey = address;
        Cursor c = null;
        boolean update = false;
        int friendId = -1;
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "insertOrUpdateFriend address: " + address);
        }

        // First check if there is a match in the friend table
        try {
            c = getContentResolver().query(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    new String[] {
                        EventStream.FriendColumns._ID,
                        EventStream.FriendColumns.FRIEND_KEY
                    }, "PHONE_NUMBERS_EQUAL(friend_key, '" + address + "')", null, null);
            if (c != null && c.moveToFirst()) {
                update = true;
                friendId = c.getInt(c.getColumnIndexOrThrow(EventStream.FriendColumns._ID));
                friendKey = c.getString(c.getColumnIndexOrThrow(EventStream.FriendColumns.FRIEND_KEY));
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }

        ContentValues friend = new ContentValues();
        try {
            c = mSourceHelper.contactLookupAddress(this, address);
            if (c != null && c.moveToFirst()) {
                String displayName = c.getString(c.getColumnIndexOrThrow(PhoneLookup.DISPLAY_NAME));
                Long contactId = c.getLong(c.getColumnIndexOrThrow(PhoneLookup._ID));
                Uri contactUri = Uri.withAppendedPath(ContactsContract.Contacts.CONTENT_URI,
                        String.valueOf(contactId));

                friend.put(EventStream.FriendColumns.DISPLAY_NAME, displayName);
                friend.put(EventStream.FriendColumns.CONTACTS_REFERENCE, contactUri.toString());

                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Found contact id = " + contactId);
                }
            } else {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "No contact found");
                }
                friend.putNull(EventStream.FriendColumns.CONTACTS_REFERENCE);
                friend.put(EventStream.FriendColumns.DISPLAY_NAME, address);
            }
            friend.putNull(EventStream.FriendColumns.PROFILE_IMAGE_URI);
        } finally {
            if (c != null) {
                c.close();
            }
        }
        friend.put(EventStream.FriendColumns.SOURCE_ID, sourceId);

        // Update or insert friend
        if (update) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Update friend address: " + address);
            }
            int rows = getContentResolver().update(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    friend, EventStream.FriendColumns._ID + " = " + friendId, null);
            if (Config.DEBUG && rows != 1) {
                Log.d(Config.LOG_TAG, "Failed to update friend " + friendId + ", rows=" + rows);
            }
        } else {
            friend.put(EventStream.FriendColumns.FRIEND_KEY, address);
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Insert friend address: " + address);
            }
            Uri newFriend = getContentResolver().insert(
                    EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI, friend);
            if (newFriend == null) {
                Log.e(Config.LOG_TAG, "Could not create a friend");
            }
        }

        return friendKey;
    }

    /**
     * Will launch an intent to show a message view
     */
    protected void launchMessageIntent(Context context, String address) {
        if (context == null) {
            return;
        }
        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse("sms:" + address));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }
    /**
     * Return all message events from the event stream
     *
     * @return All message events
     */
    protected Map<Integer, Event> getEventStreamMessagingEvents() {
        Cursor eventsCursor = null;
        Map<Integer, Event> events = new HashMap<Integer, Event>();
        try {
            eventsCursor = getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    new String[] {
                        EventStream.EventColumns._ID,
                        EventStream.EventColumns.EVENT_KEY,
                        EventStream.EventColumns.FRIEND_KEY
                        }, null, null, null);
            if (eventsCursor != null) {
                while (eventsCursor.moveToNext()) {
                    Event event = new Event();
                    event.id = eventsCursor.getInt(eventsCursor
                            .getColumnIndexOrThrow(EventStream.EventColumns._ID));
                    event.event_key = eventsCursor.getString(eventsCursor
                            .getColumnIndexOrThrow(EventStream.EventColumns.EVENT_KEY));
                    event.friend_key = eventsCursor.getString(eventsCursor
                            .getColumnIndexOrThrow(EventStream.EventColumns.FRIEND_KEY));
                    events.put(Integer.valueOf(event.event_key), event);
                }
            }
        } finally {
            if (eventsCursor != null) {
                eventsCursor.close();
            }
        }

        return events;
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    /*
     * Write user-defined values to columns affected by locale change
     */
     private void setLocaleColumns(String pluginName, String sourceName) {

         putStringValueInColumn(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                                         EventStream.PluginColumns.NAME,
                                         pluginName);


         putStringValueInColumn(EventStream.EVENTSTREAM_SOURCES_PROVIDER_URI,
                                   EventStream.SourceColumns.NAME,
                                         sourceName);
     }

     private void putStringValueInColumn (Uri tableUri, String columnName, String value) {

         ContentValues values = new ContentValues();
         int result;

         values.put(columnName, value);

         try {
             result = getContentResolver().update(tableUri, values, null, null);
                 if (result != 1) {
                          Log.e(Config.LOG_TAG, "Failed to update column: " + columnName + "in: " + tableUri +
                                  "result= " + result);
                 }
         } catch (SQLException exception) {
         Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
         throw new RuntimeException(exception);
         } catch (SecurityException exception) {
         Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
         throw new RuntimeException(exception);
         }
     }

    /**
     * Timer task to start a friend update job
     */
    class FriendUpdateTask extends TimerTask {

        @Override
        public void run() {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Friend update timer expired, updating...");
            }
            Intent serviceIntent = new Intent();
            serviceIntent.setComponent(new ComponentName(getBaseContext(),
                    TelephonyPluginService.class));
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY,
                    ServiceIntentCmd.UPDATE_FRIENDS);
            startService(serviceIntent);
        }
    }

    /**
     * Timer task to start a message update job
     */
    class MessageUpdateTask extends TimerTask {

        @Override
        public void run() {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Message update timer expired, updating...");
            }
            Intent serviceIntent = new Intent();
            serviceIntent.setComponent(new ComponentName(getBaseContext(),
                    TelephonyPluginService.class));
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY,
                    ServiceIntentCmd.REFRESH_REQUEST);
            startService(serviceIntent);
        }
    }

    /**
     * Observer for Contact content changes
     */
    class ContactsContentObserver extends ContentObserver {

        /** Timer to handle frequent on change requests */
        private Timer mTimer;

        /**
         * Constructor
         *
         * @param handler
         */
        public ContactsContentObserver(Handler handler) {
            super(handler);
            mTimer = new Timer();
        }

        /**
         * Will close a existing bulk timer
         */
        public void closeExistingBulkTimer() {
            if (mTimer != null) {
                mTimer.cancel();
                mTimer.purge();
                mTimer = null;
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "ContactsContentObserver timer is closed.");
                }
            }
        }

        @Override
        public void onChange(boolean selfChange) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Change occured in Contacts provider.");
            }
            closeExistingBulkTimer();
            if (mTimer == null) {
                mTimer = new Timer();
                mTimer.schedule(new FriendUpdateTask(), Config.ONCHANGE_DELAY);
            }
        }
    }

    /**
     * Observer for Messaging content changes
     */
    class MessagingContentObserver extends ContentObserver {

        /** Timer to handle frequent on change requests */
        private Timer mTimer;

        /**
         * Constructor
         *
         * @param handler
         */
        public MessagingContentObserver(Handler handler) {
            super(handler);
        }

        /**
         * Will close a existing bulk timer
         */
        public void closeExistingBulkTimer() {
            if (mTimer != null) {
                mTimer.cancel();
                mTimer.purge();
                mTimer = null;
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "MessagingContentObserver timer is closed.");
                }
            }
        }

        @Override
        public void onChange(boolean selfChange) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Change occured in MessagingContentObserver");
            }

            closeExistingBulkTimer();
            if (mTimer == null) {
                mTimer = new Timer();
                mTimer.schedule(new MessageUpdateTask(), Config.ONCHANGE_DELAY);
            }
        }
    }

    /**
     * Holds an event item
     */
    static class Event {

        /** The event id */
        int id;

        /** User data */
        String event_key;

        /** The mapped friend id */
        String friend_key;
    }
}
