/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin;

import android.test.InstrumentationTestCase;

/**
 * Tests involving the com.sonyericsson.eventstream.facebookplugin.Settings class.
 */
public class UFacebookPluginTestSettings extends InstrumentationTestCase {
    private Settings mSettings;
    private static final String AUTH_TOKEN = "MY_SECRET_TOKEN";
    private static final String SESSION_KEY = "SESSION_KEY";
    private static final String SESSION_SECRET = "SESSION_SECRET";

    @Override
    protected void setUp() throws Exception {
        mSettings = new Settings(getInstrumentation().getContext());
        mSettings.removeSettings();

        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        mSettings.removeSettings();
        super.tearDown();
    }

    /**
     * Test methods related to authentication token
     */
    public void testAuthenticationToken() {
        // Get token that isn't set
        String token = mSettings.getAuthenticationToken();
        assertNull("The authentication token isn't null", token);

        // Set token and retrieve it
        mSettings.setAuthenticationToken(AUTH_TOKEN);
        token = mSettings.getAuthenticationToken();
        assertEquals("Couldn't set token", AUTH_TOKEN, token);

        // Create a new instance of the class, retrieve same token
        mSettings = new Settings(getInstrumentation().getContext());
        token = mSettings.getAuthenticationToken();
        assertEquals("Wasn't same token", AUTH_TOKEN, token);

        // Reset the token, should be null when retrieving
        mSettings.setAuthenticationToken(null);
        token = mSettings.getAuthenticationToken();
        assertNull("Reset doesn't work", token);
    }

    /**
     * Test methods related to youngest timestamp
     */
    public void testYoungestTimestamp() {
        // Get timestampe that isn't set
        Long timestamp = mSettings.getYoungestTimestamp();
        assertNotNull("timestamp is null", timestamp);
        assertEquals("The youngest timestamp isn't null", 0, timestamp.longValue());

        // Set timestamp and retrieve it
        Long currentTime = System.currentTimeMillis();
        mSettings.setYoungestTimestamp(currentTime);
        timestamp = mSettings.getYoungestTimestamp();
        assertEquals("Set and retrieve failed", timestamp, currentTime);

        // Create a new instance of the class, retrieve the same timestamp
        mSettings = new Settings(getInstrumentation().getContext());
        timestamp = mSettings.getYoungestTimestamp();
        assertEquals("Wasn't the same timestamp", timestamp, currentTime);

        // Reset the timestamp, should be default value when retrieving
        mSettings.setYoungestTimestamp(null);
        timestamp = mSettings.getYoungestTimestamp();
        assertNotNull("Reset failed", timestamp);
    }

    /**
     * Test methods related to last communication time
     */
    public void testLastCommunicationTime() {
        // Get communication time that isn't set
        Long timestamp = mSettings.getLastCommunicationTime();
        assertNotNull("Communication time is null", timestamp);
        assertEquals("The timestamp is set", 0, timestamp.longValue());

        // Set communication time and retrieve it
        Long currentTime = System.currentTimeMillis();
        mSettings.setLastCommunicationTime(currentTime);
        timestamp = mSettings.getLastCommunicationTime();
        assertEquals("Set and retrieve failed", timestamp, currentTime);

        // Create a new instance of the class, retrieve the same communication time
        mSettings = new Settings(getInstrumentation().getContext());
        timestamp = mSettings.getLastCommunicationTime();
        assertEquals("Wasn't the same timestamp", timestamp, currentTime);

        // Reset the communication time, should be null when retrieving
        mSettings.setLastCommunicationTime(null);
        timestamp = mSettings.getLastCommunicationTime();
        assertEquals("Reset failed", timestamp, new Long(0));
    }

    /**
     * Test methods related to own id
     */
    public void testOwnId() {
        // Get source id that isn't set
        String ownId = mSettings.getOwnId();
        assertNotNull("The id is null", ownId);
        assertEquals("The source id isn't an empty string", "", ownId);

        // Set source id and retrieve it
        mSettings.setOwnId("12");
        ownId = mSettings.getOwnId();
        assertEquals("Set and retrieve failed", "12", ownId);

        // Create a new instance of the class, retrieve same source id
        mSettings = new Settings(getInstrumentation().getContext());
        ownId = mSettings.getOwnId();
        assertEquals("Wasn't same id", ownId, "12");

        // Reset the source id, should be default value when retrieving
        mSettings.setOwnId(null);
        ownId = mSettings.getOwnId();
        assertTrue("The source id isn't empty", ownId.length() == 0);
    }

    /**
     * Test methods related to session key
     */
    public void testSessionKey() {
        // Get key that isn't set
        String key = mSettings.getSessionKey();
        assertNull("The key token isn't null", key);

        // Set key and retrieve it
        mSettings.setSessionKey(SESSION_KEY);
        key = mSettings.getSessionKey();
        assertEquals("Couldn't set key", SESSION_KEY, key);

        // Create a new instance of the class, retrieve same key
        mSettings = new Settings(getInstrumentation().getContext());
        key = mSettings.getSessionKey();
        assertEquals("Wasn't same string", SESSION_KEY, key);

        // Reset the key, should be null when retrieving
        mSettings.setSessionKey(null);
        key = mSettings.getSessionKey();
        assertNull("Reset doesn't work", key);
    }

    /**
     * Test methods related to session secret
     */
    public void testSessionSecret() {
        // Get key that isn't set
        String secret = mSettings.getSessionSecret();
        assertNull("The secret isn't null", secret);

        // Set secret and retrieve it
        mSettings.setSessionSecret(SESSION_SECRET);
        secret = mSettings.getSessionSecret();
        assertEquals("Couldn't set secret", SESSION_SECRET, secret);

        // Create a new instance of the class, retrieve same secret
        mSettings = new Settings(getInstrumentation().getContext());
        secret = mSettings.getSessionSecret();
        assertEquals("Wasn't same string", SESSION_SECRET, secret);

        // Reset the secret, should be null when retrieving
        mSettings.setSessionSecret(null);
        secret = mSettings.getSessionSecret();
        assertNull("Reset doesn't work", secret);
    }

    public void testReset() {
        mSettings.removeSettings();

        assertEquals("Own id isn't empty string", "", mSettings.getOwnId());
        assertNull("Authentication token isn't null", mSettings.getAuthenticationToken());
        assertEquals("Youngest timestamp isn't 0", new Long(0), mSettings.getYoungestTimestamp());
        assertEquals("Last communication time isn't 0", new Long(0), mSettings.getLastCommunicationTime());
        assertNull("Session key isn't null", mSettings.getSessionKey());
        assertNull("Session secret isn't null", mSettings.getSessionSecret());

        mSettings.setOwnId("101010101");
        mSettings.setAuthenticationToken("Great");
        mSettings.setYoungestTimestamp(42L);
        mSettings.setLastCommunicationTime(42L);
        mSettings.setSessionKey(SESSION_KEY);
        mSettings.setSessionSecret(SESSION_SECRET);

        mSettings.removeSettings();
        assertEquals("Own id isn't empty string", "", mSettings.getOwnId());
        assertNull("Authentication token isn't null", mSettings.getAuthenticationToken());
        assertEquals("Youngest timestamp isn't 0", new Long(0), mSettings.getYoungestTimestamp());
        assertEquals("Last communication time isn't 0", new Long(0), mSettings.getLastCommunicationTime());
        assertNull("Session key isn't null", mSettings.getSessionKey());
        assertNull("Session secret isn't null", mSettings.getSessionSecret());
    }
}