/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.calllogplugin.utilities;

import com.sonyericsson.eventstream.calllogplugin.PluginConstants;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream.EventTable;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream.FriendTable;
import com.sonyericsson.eventstream.calllogplugin.R;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.accounts.OnAccountsUpdateListener;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.provider.ContactsContract;
import android.provider.CallLog.Calls;
import android.provider.ContactsContract.Contacts;
import android.provider.ContactsContract.Data;
import android.provider.ContactsContract.RawContacts;
import android.provider.ContactsContract.CommonDataKinds.Phone;
import android.provider.ContactsContract.CommonDataKinds.StructuredName;
import android.test.AndroidTestCase;

public abstract class CallLogPluginTestCase extends AndroidTestCase implements OnAccountsUpdateListener {

    Context mContext;

    ContentResolver mCr;

    String mSyncAccountName;

    String mSyncAccountType;

    @Override
    protected void setUp() throws Exception {
        super.setUp();
        mContext = this.getContext();
        mCr = mContext.getContentResolver();
        // Uninstall plugin
        unregisterPlugin();
        // Clear Call log
        clearCallLog();

        AccountManager.get(mContext).addOnAccountsUpdatedListener(this, null, true);
    }

    @Override
    protected void tearDown() throws Exception {
        unregisterPlugin();
    }

    public void sleep(int seconds) {
        try {
            Thread.sleep(seconds * 1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    public void registerPlugin() {
        System.out.println("Registering CallLogPlugin");
        Intent intent = new Intent();
        intent.setAction(EventStream.EVENTSTREAM_REGISTER_ACTION);
        mContext.sendBroadcast(intent);
    }

    public void unregisterPlugin() {
        System.out.println("Unregistering CallLogPlugin");
        mCr.delete(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
        mCr.delete(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
        mCr.delete(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI, null, null);
        mCr.delete(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, null, null);
        mCr.delete(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null);
    }

    public boolean isPluginRegistered() {
        Cursor cursor = null;
        boolean result = false;
        try {
            cursor = mContext.getContentResolver().query(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null,
                    null, null, null);
            if (cursor.getCount() > 0) {
                result = true;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }

    public boolean isServiceRegistered() {
        Cursor cursor = null;
        boolean result = false;
        try {
            cursor = mContext.getContentResolver().query(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, null,
                    null, null, null);
            if (cursor.getCount() > 0) {
                result = true;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }

    public void clearCallLog() {
        mContext.getContentResolver().delete(Calls.CONTENT_URI, null, null);
    }

    public int addCall(int type, String number, String name) {
        ContentValues values = new ContentValues();
        values.put(Calls.CACHED_NAME, name);
        values.put(Calls.CACHED_NUMBER_LABEL, "1");
        values.put(Calls.DATE, "0");
        values.put(Calls.NUMBER, number);
        values.put(Calls.TYPE, type);
        Uri result = mContext.getContentResolver().insert(Calls.CONTENT_URI, values);
        return (int)ContentUris.parseId(result);
    }

    public void deleteCalls(String number) {
        mContext.getContentResolver().delete(
                Calls.CONTENT_URI,
                Calls.NUMBER + "='" + number + "'",
                null);
    }

    public int removeCall(int callId) {
        return mContext.getContentResolver().delete(Calls.CONTENT_URI, Calls._ID + "==" + callId,
                null);
    }

    public int getNumberOfEvents() {
        Cursor c = null;
        int numEvents = 0;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null,
                    null, null);
            numEvents = c.getCount();
        } finally {
            if (c != null)
                c.close();
        }
        return numEvents;
    }

    public Event getLastEvent() {
        Event e = null;
        Cursor c = null;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null,
                    null, null);
            if (c.moveToLast()) {
                e = new Event();
                e.eventKey = c.getString(c.getColumnIndex(EventTable.EVENT_KEY));
                e.friendKey = c.getString(c.getColumnIndex(EventTable.FRIEND_KEY));
                e.title = c.getString(c.getColumnIndex(EventTable.TITLE));
                e.message = c.getString(c.getColumnIndex(EventTable.MESSAGE));
                e.sourceId = c.getInt(c.getColumnIndex(EventTable.SOURCE_ID));
            }
        } finally {
            if (c != null)
                c.close();
        }
        return e;
    }

    public Event getNextLastEvent() {
        Event e = null;
        Cursor c = null;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null,
                    null, null);
            if (c.moveToLast()) {
                c.moveToPrevious();
                e = new Event();
                e.eventKey = c.getString(c.getColumnIndex(EventTable.EVENT_KEY));
                e.friendKey = c.getString(c.getColumnIndex(EventTable.FRIEND_KEY));
                e.title = c.getString(c.getColumnIndex(EventTable.TITLE));
                e.message = c.getString(c.getColumnIndex(EventTable.MESSAGE));
                e.sourceId = c.getInt(c.getColumnIndex(EventTable.SOURCE_ID));
            }
        } finally {
            if (c != null)
                c.close();
        }
        return e;
    }

    public Event getEventFromCallId(int callId) {
        Event e = null;
        Cursor c = null;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null,
                    EventStream.EventTable.EVENT_KEY + "==" + callId, null, null);
            if (c != null && c.moveToFirst()) {
                if (c.getCount() == 1) {
                    e = new Event();
                    e.eventKey = c.getString(c.getColumnIndex(EventTable.EVENT_KEY));
                    e.friendKey = c.getString(c.getColumnIndex(EventTable.FRIEND_KEY));
                    e.title = c.getString(c.getColumnIndex(EventTable.TITLE));
                    e.message = c.getString(c.getColumnIndex(EventTable.MESSAGE));
                    e.sourceId = c.getInt(c.getColumnIndex(EventTable.SOURCE_ID));
                } else {
                    fail("Duplicate events, CallLog id = " + callId);
                }
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return e;
    }

    public Friend getFriend(String phoneNumber) {
        Cursor c = null;
        Friend f = null;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI, null,
                    FriendTable.FRIEND_KEY + "==\"" + phoneNumber + "\"", null, null);
            if (c.getCount() > 1) {
                fail("More than one friend matching eventKey==" + phoneNumber);
            }
            if (c.moveToFirst()) {
                f = new Friend();
                f.displayName = c.getString(c.getColumnIndex(FriendTable.DISPLAY_NAME));
                f.contactUri = c.getString(c.getColumnIndex(FriendTable.CONTACTS_REFERENCE));
                f.friendKey = c.getString(c.getColumnIndex(FriendTable.FRIEND_KEY));
            }
        } finally {
            if (c != null)
                c.close();
        }
        return f;
    }

    public Friend getFriendFromFriendKey(String friendkey) {
        Cursor c = null;
        Friend f = null;
        try {
            c = mContext.getContentResolver().query(EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI, null,
                    FriendTable.FRIEND_KEY + "='" + friendkey + "'", null, null);
            if (c != null && c.moveToFirst()) {
                f = new Friend();
                f.displayName = c.getString(c.getColumnIndex(FriendTable.DISPLAY_NAME));
                f.contactUri = c.getString(c.getColumnIndex(FriendTable.CONTACTS_REFERENCE));
                f.friendKey = c.getString(c.getColumnIndex(FriendTable.FRIEND_KEY));
            }
        } finally {
            if (c != null)
                c.close();
        }
        return f;
    }

    public String getCallerInfoString(String CallerInfo) {
        String result;
        if (CallerInfo != null) {
            if (PluginConstants.CallerInfo.UNKNOWN_NUMBER.equals(CallerInfo)) {
                result = mContext.getApplicationContext().getResources().getString(R.string.calllog_callerinfo_unknown);
            } else if (PluginConstants.CallerInfo.PRIVATE_NUMBER.equals(CallerInfo)) {
                result = mContext.getApplicationContext().getResources().getString(R.string.calllog_callerinfo_private_num);
            } else if (PluginConstants.CallerInfo.PAYPHONE_NUMBER.equals(CallerInfo)) {
                result = mContext.getApplicationContext().getResources().getString(R.string.calllog_callerinfo_payphone);
            } else if ("".equals(CallerInfo)) {
                result = mContext.getApplicationContext().getResources().getString(R.string.calllog_callerinfo_unknown);
            } else {
                result = CallerInfo;
            }
        } else {
            result = mContext.getApplicationContext().getResources().getString(R.string.calllog_callerinfo_unknown);
        }
        return result;
    }

    /**
     * Create a new contact and return the raw contact uri
     *
     * @param name Name of the new contact.
     * @param phoneNumber Phone number of the new contact.
     * @return the contact URI.
     */
    public String addContact(String name, String phoneNumber) {
        ContentValues values = new ContentValues();
        values.put(RawContacts.ACCOUNT_TYPE, mSyncAccountType);
        values.put(RawContacts.ACCOUNT_NAME, mSyncAccountName);
        Uri result = mCr.insert(RawContacts.CONTENT_URI, values);
        int rawId = (int)ContentUris.parseId(result);

        // Insert name
        values.clear();
        values.put(Data.RAW_CONTACT_ID, rawId);
        values.put(Data.MIMETYPE, StructuredName.CONTENT_ITEM_TYPE);
        values.put(StructuredName.DISPLAY_NAME, name);
        mCr.insert(Data.CONTENT_URI, values);

        // Insert phone number
        values.clear();
        values.put(Data.RAW_CONTACT_ID, rawId);
        values.put(Data.MIMETYPE, Phone.CONTENT_ITEM_TYPE);
        values.put(Phone.NUMBER, phoneNumber);
        values.put(Phone.TYPE, Phone.TYPE_HOME);
        mCr.insert(Data.CONTENT_URI, values);

        Uri lookupUri = ContactsContract.RawContacts.getContactLookupUri(mCr, result);

        return ContactsContract.Contacts.lookupContact(mCr, lookupUri).toString();
    }

    /**
     * Find the raw contact uri given a contacts name
     *
     * @param name
     * @return
     */
    public String findRawContactFromName(String name) {
        String result = "";
        Cursor c = null;
        try {
            long rawId = -1;
            c = mCr.query(Data.CONTENT_URI, null,
                    StructuredName.DISPLAY_NAME + "==\"" + name
                            + "\"", null, null);
            if (c != null && c.moveToLast()) {
                rawId = c.getLong(c.getColumnIndex(Data.RAW_CONTACT_ID));
                result = Uri.withAppendedPath(ContactsContract.RawContacts.CONTENT_URI, String.valueOf(rawId)).toString();
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return result;
    }

    public void changeContactName(String rawContactUri, String newName) {
        long rawId = Long.valueOf(Uri.parse(rawContactUri).getLastPathSegment());
        ContentValues values = new ContentValues();
        values.put(StructuredName.DISPLAY_NAME, newName);
        mCr.update(Data.CONTENT_URI, values,
                Data.RAW_CONTACT_ID + "==? AND " +
                Data.MIMETYPE + "==\"" + StructuredName.CONTENT_ITEM_TYPE+ "\"",
                new String[] {Long.toString(rawId)});
    }

    public void changePhoneNumber(String rawContactUri, String phoneNumber) {
        long rawId = Long.valueOf(Uri.parse(rawContactUri).getLastPathSegment());
        ContentValues values = new ContentValues();
        values.put(Phone.NUMBER, phoneNumber);
        mCr.update(Data.CONTENT_URI, values,
                Data.RAW_CONTACT_ID + "==? AND " +
                Data.MIMETYPE + "==\"" + Phone.CONTENT_ITEM_TYPE + "\"",
                new String[] {Long.toString(rawId)});
    }

    private long lookupRawId(long contactId) {
        long rawId = -1;
        Cursor c = null;
        try {
            c = mCr.query(RawContacts.CONTENT_URI, new String[] {RawContacts._ID}, RawContacts.CONTACT_ID + "=" + contactId, null, null);
            if (c != null && c.moveToFirst()) {
                rawId = c.getLong(c.getColumnIndex(RawContacts._ID));
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return rawId;
    }

    public long lookupContactId(String lookupKey) {
        long contactId = -1;
        Uri lookupUri = Uri.withAppendedPath(Contacts.CONTENT_LOOKUP_URI, lookupKey);
        Uri contactUri = Contacts.lookupContact(mCr, lookupUri);
        if (contactUri != null) {
            contactId = Long.parseLong(contactUri.getLastPathSegment());
        }
        return contactId;
    }

    public boolean lookupKeysEquals(String lookup1, String lookup2) {
        return lookupContactId(lookup1) == lookupContactId(lookup2);
    }

    public void removeContact(String rawContactUri) {
        mCr.delete(Uri.parse(rawContactUri), null, null);
    }

    public void onAccountsUpdated(Account[] accounts) {
        for(Account acc : accounts) {
            mSyncAccountName = acc.name;
            mSyncAccountType = acc.type;
        }
    }

    public class Event {
        public int sourceId;

        public String eventKey;

        public String title;

        public String message;

        public String friendKey;

        public String toString() {
            return "source_id = " + sourceId + "\n" +
                   "eventKey = " + eventKey + "\n"  +
                   "title = " + title + "\n" +
                   "message = " + message + "\n" +
                   "friend_id = " + friendKey;
        }
    }

    public class Friend {
        public String displayName;
        public String contactUri;
        public String friendKey;
    }
}
