	.list ON, EXP
	
; Primitive (kernel) definitions for fig-FORTH for SH-3
; Joel Matthew Rees, Hyougo Polytec Center
; 2014.02.28


; Monolithic, not separate assembly:
; context.inc must be included before this file.
;	.include	"context.inc"
;
;	.section	primitives, code, align=4


; ***** Need to load the return register with something safe.
; Probably the call to next from warm?
;
; Anyway, this is the inner interpreter.
;
next:
	mov.l	@fIP+, fW	; get the pointer to the next definition to execute
	mov.l	@fW, r0		; get the defitinition characteristic
	jsr		@r0
; 3 cycles to get back to the top of the loop.
	nop	
	bra		next
	nop
; Note that, since jumps to absolute addresses have limits on constant-width instruction sets,
; using the subroutine call mode for the virtual machine is not as much a penalty as it might seem.
; It also has the advantage of being more compatible with more conventional code.
; Ways to make an absolute jump work might include 
; * the address of next in a table of constants (and reserving a register for the table base), or
; * reserving a register for the address of next.


; LIT     ( --- n )                                               C
;         Push the following word from the instruction stream as a
;         literal, or immediate value.
;
	HEADER	LIT, LIT
	mov.l	@fIP+, r0
	mov.l	r0, @-fSP
	rts
	nop


; "character" (byte or word) literal doesn't work on SH3
; It'll cause alignment problems.


; EXECUTE ( adr --- )                                             C
;         Jump to address on stack.  Used by the "outer" interpreter to
;         interactively invoke routines.  (Not compile-only in fig.)
;
	HEADER	EXECUTE, EXECUTE
	mov.l	@fSP+, fW
	mov.l	@fW, r0
	jmp 	@r0		; borrow the return there
	nop


; BRANCH  ( --- )                                                 C
;         Add the following word from the instruction stream to the
;         instruction pointer (Y++).  Causes a program branch.
;
	HEADER	BRANCH, BRANCH
	mov.l	@fIP+, r0
BRANCHgo:
	add.l	r0, fIP
	rts
	nop


; 0BRANCH ( f --- )                                               C
;         BRANCH if flag is zero.
;
	HEADER	0BRANCH, ZBRANCH
	mov.l	@fSP+, r0
	cmp/eq	#0, r0	
	bt/s	BRANCHgo
	mov.l	@fIP+, r0
	rts
	nop
	

; fig-FORTH puts temporaries on the control stack. I prefer a third stack.
; But if we put I in registers, (DO) is affected.
; One might put I and the loop limit in, say, r8 and r9, 
; but then they must be saved and restored,
; and interrupts have to avoid r8 and r9 or save them.
;
; Note: fig-FORTH +LOOP has an un-signed loop counter, but a signed increment.
; (JMR: but the increment is signed!)


; (LOOP)  ( --- )         ( limit index *** limit index+1)        C
;                         ( limit index *** )
;         Counting loop primitive.  The counter and limit are the top two
;         words on the return stack.  If the updated index/counter does
;         not exceed the limit, a branch occurs.  If it does, the branch
;         does not occur, and the index and limit are dropped from the
;         return stack.
;
	HEADER	(LOOP), xLOOP
	mov.l	@fRP, r0	; I (loop counter)
	add.l	#1, r0
	mov.l	r0, @fRP	; update I
	mov.l	@(NATURAL_SIZE,fRP), r1	; limit
	cmp/ge	r1, r0		; r0 >= r1 ?
	bf/s	BRANCHgo	; not yet
	mov.l	@fIP+, r0
	rts
	add.l	#2*NATURAL_SIZE, fRP
	

; (+LOOP) ( n --- )       ( limit index *** limit index+n )       C
;                         ( limit index *** )
;         Loop with a variable increment.  Terminates when the index
;         crosses the boundary from one below the limit to the limit.  A
;         positive n will cause termination if the result index equals the
;         limit.  A negative n must cause the index to become less than
;         the limit to cause loop termination.
;
	HEADER	(+LOOP), xPLOOP
	mov.l	@fSP+, r1	; increment
	mov.l	@fRP, r0	; I (loop counter)
	add.l	r1, r0
	mov.l	r0, @fRP	; update I
	shal	r1		 	; increment negative or positive?
	bt/s	xPLOOPminus
	mov.l	@(NATURAL_SIZE,fRP), r1	; limit
;
; Stealing too much code would cost more than it would save.
xPLOOPplus:
	cmp/ge	r0, r1		; limit (r1) >= counter (I=r0) ?
	bf/s	BRANCHgo	; not yet
	mov.l	@fIP+, r0	; grab offset and bump fIP before we go
	rts
	add.l	#2*NATURAL_SIZE, fRP	; drop I and limit before we return
;
xPLOOPminus:
	cmp/ge	r0, r1		; limit (r1) >= counter (I=r0) ?
	bt/s	BRANCHgo	; not yet
	mov.l	@fIP+, r0	; grab offset and bump fIP before we go
	rts
	add.l	#2*NATURAL_SIZE, fRP	; drop I and limit before we return


; Putting I and limit in registers would require (DO) to save the registers first 
; and it would require LOOP and +LOOP to restore the registers on exit.
; That would cost more than it would save.
; 
; (DO)    ( limit index --- )     ( *** limit index )
;         Move the loop parameters to the return stack.  Synonym for D>R, here.
; 
	HEADER (DO), xPDO
	mov.l	@fSP+, r0
	mov.l	@fSP+, r1
	add.l	#-2*NATURAL_SIZE, fRP
	mov.l	r1, @(NATURAL_SIZE,fRP)
	mov.l	r0, @fRP
	rts






