//
//  KMThreadList.m
//  BathyScaphe
//
//  Created by Hori,Masaki on 11/07/18.
//  Copyright 2011 masakih. All rights reserved.
//

#import "KMThreadList.h"
#import "ThreadsListTable.h"

#import "BoardListItem.h"
#import "BSDBThreadList.h"
#import "CMRMainMenuManager.h"

#import "BSQuickLookPanelController.h"
#import "CMRTextColumnCell.h"
//#import <CocoMonar/CMRPropertyKeys.h>
#import "AppDefaults.h"
#import "CMXMenuHolder.h"

#import "KMWorkerEmulator.h"

#import "DatabaseManager.h"

#import "CMRBrowser_p.h"
#import "CMRAppDelegate.h"


@interface KMThreadList()
@property CMRAutoscrollCondition keepConditionForScrolling;
@property (retain) NSArray *keepSelectedThreadPathForScrolling;

- (void)registerKeepCondition:(CMRAutoscrollCondition)type;
- (void)clearKeepCondition;



- (NSUInteger)selectRowIndexesWithThreadPaths:(NSArray *)paths byExtendingSelection:(BOOL)extend scrollToVisible:(BOOL)scroll;
- (NSUInteger)selectRowWithThreadPath:(NSString *)filepath byExtendingSelection:(BOOL)flag scrollToVisible:(BOOL)scroll;
- (void)reselectThreadIfNeeded;

@end

@interface KMThreadList(Actions)
- (IBAction)reloadThreadsList:(id)sender;
- (IBAction)showSelectedThread:(id)sender;
- (IBAction)selectThread:(id)sender;
- (IBAction)listViewAction:(id)sender;
- (IBAction)listViewDoubleAction:(id)sender;

- (IBAction)tableViewActionDispatch:(id)sender actionKey:(NSString *)aKey defaultAction:(SEL)defaultAction;

@end

@interface KMThreadList(VIewBuilding)
- (void)setupThreadsListTable;
@end

@interface KMThreadList(KM_NSTableViewDelegate) <NSTableViewDelegate>
@end


@implementation KMThreadList
@synthesize menu = _menu;
@synthesize threadListView = _threadListView;
@synthesize selection = _selection;

@synthesize keepConditionForScrolling = _keepConditionForScrolling;
@synthesize keepSelectedThreadPathForScrolling = _keepSelectedThreadPathForScrolling;

- (id)init
{
	self = [super initWithNibName:@"KMThreadList" bundle:nil];
	if(self) {
		worker = [[KMWorkerEmulator alloc] init];
		
		
		NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
        DatabaseManager *dbm =[DatabaseManager defaultManager];
        [nc addObserver:self
               selector:@selector(updateThreadsListNow:)
                   name:DatabaseDidFinishUpdateDownloadedOrDeletedThreadInfoNotification
                 object:dbm];
        [nc addObserver:self
               selector:@selector(updateThreadsListPartially:)
                   name:DatabaseWantsThreadItemsUpdateNotification
                 object:dbm];
	}
	return self;
}
- (void)dealloc
{
	[worker release];
	[_selection release];
	[datasource release];
	
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	[super dealloc];
}

- (void)awakeFromNib
{
	[self setupThreadsListTable];
}

- (void)setRepresentedObject:(id)representedObject
{
	if([self representedObject] == representedObject) return;
	if(![representedObject isKindOfClass:[BoardListItem class]]) return;
	if([BoardListItem isFolderItem:representedObject]) return;
	
	[super setRepresentedObject:representedObject];
	
	[datasource release];
	datasource = [[BSDBThreadList alloc] initWithBoardListItem:representedObject];
	NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
	[nc addObserver:self
		   selector:@selector(listDidChangeNotification:)
			   name:CMRThreadsListDidChangeNotification
			 object:datasource];
	[nc addObserver:self
		   selector:@selector(threadsListWantsPartialReload:)
			   name:BSDBThreadListWantsPartialReloadNotification
			 object:datasource];
	
	[self.threadListView setDataSource:datasource];
	
	// sort column change
	NSString *boardName = [datasource boardName];
	NSArray *foo = [self.threadListView sortDescriptors];
	NSArray *bar = [[BoardManager defaultManager] sortDescriptorsForBoard:boardName];
	[self.threadListView setSortDescriptors:[[BoardManager defaultManager] sortDescriptorsForBoard:boardName]];
	// SortDescriptors が変わらない時は tableView:sortDescriptorsDidChange: が呼ばれない。
	// これは困るので、無理矢理呼ぶ。
	if ([foo isEqualToArray:bar]) {
		[[self.threadListView dataSource] tableView:self.threadListView
						   sortDescriptorsDidChange:foo];
	}
	
	[self reloadThreadsList:nil];
}

- (NSString *)filterString
{
	return [datasource searchString];
}
- (void)setFilterString:(NSString *)filterString
{
	[datasource filterByString:filterString];
	[self.threadListView reloadData];
}

- (void)listDidChangeNotification:(id)notification
{
	[self.threadListView reloadData];
//	[self synchronizeWindowTitleWithDocumentName];
    [self reselectThreadIfNeeded];
    UTILNotifyName(CMRBrowserThListUpdateDelegateTaskDidFinishNotification);
}
- (void)tableViewSelectionDidChange:(NSNotification *)notification
{
	NSMutableArray *array = [NSMutableArray array];
	
	NSIndexSet *selectedIndexes = [self.threadListView selectedRowIndexes];
	NSUInteger row = [selectedIndexes firstIndex];
	while(row != NSNotFound) {
		[array addObject:[datasource threadAttributesAtRowIndex:row inTableView:self.threadListView]];
		row = [selectedIndexes indexGreaterThanIndex:row];
	}
	
	[self registerKeepCondition:CMRAutoscrollWhenThreadUpdate];
	self.selection = array;
	
	BSQuickLookPanelController *qlc = [BSQuickLookPanelController sharedInstance];
    NSTableView *tableView = [notification object];
    if ([qlc isLooking]) {
        [datasource tableView:tableView quickLookAtRowIndexes:[tableView selectedRowIndexes] keepLook:YES];
    }
}


- (void)updateThreadsListNow:(NSNotification *)notification
{
    [datasource updateCursor];
}
- (void)updateThreadsListPartially:(NSNotification *)notification
{
    // スマート掲示板の場合は、自動ソートが無効になっていても「自動ソート」せざるを得ない。
    if ([datasource viewMode] == BSThreadsListShowsSmartList) {
        [datasource updateCursor];
        return;
    }
	
    NSDictionary *userInfo = [notification userInfo];
    NSArray *files = [userInfo objectForKey:UserInfoThreadPathsArrayKey];
    if (files) {
        [datasource cleanUpThreadItem:files];
    } else {
        BOOL isInsertion = [[userInfo objectForKey:UserInfoIsDBInsertedKey] boolValue];
        if (isInsertion && ([datasource viewMode] == BSThreadsListShowsStoredLogFiles)) {
            [datasource updateCursor];
        } else {
            [datasource updateThreadItem:userInfo];
        }
    }
}


//- (NSUInteger)selectRowWithCurrentThread:(BOOL)scroll
//{
//    NSString *boardName = [self boardName];
//    NSString *threadsListBoardName = [datasource boardName];
//    if ([boardName isEqualToString:threadsListBoardName]) {
//        return [self selectRowWithThreadPath:[self path]
//						byExtendingSelection:NO
//							 scrollToVisible:scroll];
//    } else {
//        return NSNotFound;
//    }
//}

- (NSUInteger)selectRowWithThreadPath:(NSString *)filepath byExtendingSelection:(BOOL)flag scrollToVisible:(BOOL)scroll
{
    if (!filepath) {
        return NSNotFound;
    }
    return [self selectRowIndexesWithThreadPaths:[NSArray arrayWithObject:filepath] byExtendingSelection:flag scrollToVisible:scroll];
}

- (NSUInteger)selectRowIndexesWithThreadPaths:(NSArray *)paths byExtendingSelection:(BOOL)extend scrollToVisible:(BOOL)scroll
{
    if (!paths || [paths count] == 0) {
        return NSNotFound;
    }
	
    NSIndexSet *indexes = nil;
    indexes = [datasource indexesOfFilePathsArray:paths ignoreFilter:NO];
	
    if (!indexes || [indexes count] == 0) {
        if (!extend) {
            [self.threadListView deselectAll:nil];
            [self.threadListView scrollRowToVisible:0];
        }
        return NSNotFound;
    }
	
    NSUInteger index = [indexes lastIndex];
    if (![[self.threadListView selectedRowIndexes] containsIndexes:indexes]) { // すでに選択されているか確認
        [self.threadListView selectRowIndexes:indexes byExtendingSelection:extend];
    }
    if (scroll) {
        [self.threadListView scrollRowToVisible:index];
    } else {
        CMRAutoscrollCondition type = self.keepConditionForScrolling;
        if (type != CMRAutoscrollWhenThreadUpdate && type != CMRAutoscrollWhenThreadDelete) {
            [self.threadListView scrollRowToVisible:0];
        }
    }
    return index;
}
- (void)reselectThreadIfNeeded
{
    CMRAutoscrollCondition type = self.keepConditionForScrolling;
    if (self.keepSelectedThreadPathForScrolling) {
        CMRAutoscrollCondition mask = [CMRPref threadsListAutoscrollMask];
        [self selectRowIndexesWithThreadPaths:self.keepSelectedThreadPathForScrolling
						 byExtendingSelection:NO
							  scrollToVisible:(mask & type)];
        [self clearKeepCondition];
    } else {
        // 3ペインで、掲示板を切り替えた場合に、表示中のスレッドを再選択するために
//        if (type == CMRAutoscrollNone && [self shouldShowContents] && [self path]) {
//            [self selectRowWithCurrentThread:NO];
//            return;
//        }
//        if (type != CMRAutoscrollWhenThreadUpdate && type != CMRAutoscrollWhenThreadDelete) {
//            [self.threadListView scrollRowToVisible:0];
//        }
    }
}

- (void)threadsListWantsPartialReload:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, BSDBThreadListWantsPartialReloadNotification);
    id indexes = [[notification userInfo] objectForKey:@"Indexes"];
	
    if (indexes == [NSNull null]) {
        [self.threadListView reloadData];
    } else {
        NSIndexSet *allColumnIndexes = [NSIndexSet indexSetWithIndexesInRange:NSMakeRange(0, [self.threadListView numberOfColumns])];
        [self.threadListView reloadDataForRowIndexes:indexes columnIndexes:allColumnIndexes];
    }
		
    if (self.keepSelectedThreadPathForScrolling) {
        [self clearKeepCondition];
    }
}

- (void)registerKeepCondition:(CMRAutoscrollCondition)type
{
	NSIndexSet *rows = [self.threadListView selectedRowIndexes];
    NSArray *tmp = nil;
    if (rows && ([rows count] > 0)) {
        tmp = [datasource tableView:self.threadListView
					threadFilePathsArrayAtRowIndexes:rows];
    }
	self.keepSelectedThreadPathForScrolling = tmp;
	self.keepConditionForScrolling = type;
}
- (void)clearKeepCondition
{
	self.keepConditionForScrolling = CMRAutoscrollNone;
	self.keepSelectedThreadPathForScrolling = nil;
}

@end

@implementation KMThreadList(Actions)
- (IBAction)reloadThreadsList:(id)sender
{
	[self registerKeepCondition:CMRAutoscrollWhenTLUpdate];
	[datasource startLoadingThreadsList:worker];
}
- (IBAction)showSelectedThread:(id)sender
{
	NSInteger rowIndex = [self.threadListView selectedRow];
	
	if (-1 == rowIndex) return;
	if ([self.threadListView numberOfSelectedRows] != 1) return;
	// #warning 64BIT: Check formatting arguments
	// 2010-03-28 tsawada2 修正済
	NSAssert2(
			  (rowIndex >= 0 && rowIndex < [self.threadListView numberOfRows]),
			  @"  rowIndex was over. size = %ld but was %ld",
			  (long)[self.threadListView numberOfRows],
			  (long)rowIndex);
	
}
- (IBAction)selectThread:(id)sender
{
	// 特定のモディファイア・キーが押されているときは
	// クリックで項目を選択してもスレッドを読み込まない
	if ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask) return;
	
	[self showSelectedThread:self];
}
/*
 NSTableView action, doubleAction はカラムのクリックでも
 発生するので、以下のメソッドでフックする。
 */
- (IBAction)tableViewActionDispatch:(id)sender actionKey:(NSString *)aKey defaultAction:(SEL)defaultAction
{
	SEL action_;
	
	// カラムのクリック
	if (-1 == [self.threadListView clickedRow]) return;
	
	// 設定されたアクションにディスパッチ
	action_ = SGTemplateSelector(aKey);
	if (NULL == action_ || _cmd == action_) {
		action_ = defaultAction;
	}
	[NSApp sendAction:action_ to:nil /*self*/ from:sender];
}

- (IBAction)listViewAction:(id)sender
{
    // 時間内に二度目のクリックが発生した場合は、ダブルクリックと判断し、クリックの action を実行しない
    NSTimeInterval interval = [NSEvent bs_doubleClickInterval];
    NSEvent *nextEvent = [[self.threadListView window] nextEventMatchingMask:NSLeftMouseUpMask
															  untilDate:[NSDate dateWithTimeIntervalSinceNow:interval]
																 inMode:NSEventTrackingRunLoopMode
																dequeue:NO];
    NSEventType type = [nextEvent type];
    if (NSLeftMouseUp == type) {
        return;
    }
	[self tableViewActionDispatch:sender
						actionKey:kThreadsListTableActionKey
					defaultAction:@selector(selectThread:)];
}

- (IBAction)listViewDoubleAction:(id)sender
{
	[self tableViewActionDispatch:sender
						actionKey:kThreadsListTableDoubleActionKey
					defaultAction:@selector(openSelectedThreads:)];
}
@end


@implementation KMThreadList(ViewBuilding)
- (void)setupStatusColumnWithTableColumn:(NSTableColumn *)column
{
    NSImage            *statusImage_;
    NSImageCell        *imageCell_;
    
    statusImage_ = [NSImage imageAppNamed:STATUS_HEADER_IMAGE_NAME];
    imageCell_  = [[NSImageCell alloc] initImageCell:nil];
	
    [[column headerCell] setAlignment:NSCenterTextAlignment];
    [[column headerCell] setImage:statusImage_];
	
    [imageCell_ setImageAlignment:NSImageAlignCenter];
    [imageCell_ setImageScaling:NSScaleNone];
    [imageCell_ setImageFrameStyle:NSImageFrameNone];
    
    [column setDataCell:imageCell_];
    [imageCell_ release];
}

- (void)updateThreadEnergyColumn
{
    NSTableColumn *column = [self.threadListView initialColumnWithIdentifier:BSThreadEnergyKey];
    if (!column) {
        return;
    }
    if ([CMRPref energyUsesLevelIndicator]) {
		BSIkioiCell *cell = [[BSIkioiCell alloc] initWithLevelIndicatorStyle:NSRelevancyLevelIndicatorStyle];
		[cell setMaxValue:10.0];
		[cell setMinValue:0.0];
		[cell setEditable:NO];
		[column setDataCell:cell];
		[cell release];
    } else {
        CMRRightAlignedTextColumnCell *cell = [[CMRRightAlignedTextColumnCell alloc] initTextCell:@""];
		
        [cell setAlignment:NSRightTextAlignment];
        [cell setAllowsEditingTextAttributes:NO];
        [cell setBezeled:NO];
        [cell setBordered:NO];
        [cell setEditable:NO];
        [cell setScrollable:NO];
		
        [cell setWraps:YES];
        [cell setDrawsBackground:NO];
        [column setDataCell:cell];
        [cell release];
    }
    NSTableColumn *currentColumn = [self.threadListView tableColumnWithIdentifier:BSThreadEnergyKey];
    if (currentColumn) {
        NSRect rect = NSZeroRect;
        NSUInteger idx = [[self.threadListView tableColumns] indexOfObject:currentColumn];
        if (idx != NSNotFound) {
            rect = [self.threadListView rectOfColumn:idx];
        }
        if (!NSEqualRects(rect, NSZeroRect)) {
            [self.threadListView setNeedsDisplayInRect:rect];
        }
    }
}

- (void)setupTableColumn:(NSTableColumn *)column
{
    if ([CMRThreadStatusKey isEqualToString:[column identifier]]) {
        [self setupStatusColumnWithTableColumn:column];
        return;
    } else if ([BSThreadEnergyKey isEqualToString:[column identifier]] && [CMRPref energyUsesLevelIndicator]) {
		BSIkioiCell *cell = [[BSIkioiCell alloc] initWithLevelIndicatorStyle:NSRelevancyLevelIndicatorStyle];
		[cell setMaxValue:10.0];
		[cell setMinValue:0.0];
		[cell setEditable:NO];
		[column setDataCell:cell];
		[cell release];
        return;
    }
	
	Class	cellClass;
	id		newCell;
	id		dataCell;
	
	dataCell = [column dataCell];
	if ([dataCell alignment] == NSRightTextAlignment) {
		cellClass = [CMRRightAlignedTextColumnCell class];
	} else {
		cellClass = [CMRTextColumnCell class];
	}
	
	newCell = [[cellClass alloc] initTextCell:@""];
	[newCell setAttributesFromCell:dataCell];
	[newCell setWraps:YES];
	[newCell setDrawsBackground:NO];
	[column setDataCell:newCell];
	[newCell release];
}
- (NSTableColumn *)tableColumnWithPropertyListRep:(id)plistRep
{
    id column_ = [(id <CMRPropertyListCoding>)[NSTableColumn alloc] initWithPropertyListRepresentation:plistRep];
    [self setupTableColumn:column_];
    return [column_ autorelease];
}
- (void)updateMenuItemStatusForColumnsMenu:(NSMenu *)menu_
{
    NSEnumerator        *iter_;
    NSMenuItem          *rep_;
    
    iter_ = [[menu_ itemArray] objectEnumerator];
    while (rep_ = [iter_ nextObject]) {
        NSInteger state_;
		
        state_ = (-1 == [self.threadListView columnWithIdentifier:[rep_ representedObject]])
		? NSOffState
		: NSOnState;
		
        [rep_ setState:state_];
    }
}
- (void)createDefaultTableColumnsWithTableView:(NSTableView *)tableView
{
    NSEnumerator        *iter_;
    id                  rep_;
    
    iter_ = [[CMRAppDelegate defaultColumnsArray] objectEnumerator];
	
    while (rep_ = [iter_ nextObject]) {
        NSTableColumn        *column_;
        
        column_ = [self tableColumnWithPropertyListRep:rep_];
        if (!column_) continue;
		
        [tableView addTableColumn:column_];
    }
	
	[(ThreadsListTable *)tableView setInitialState];
}

//
- (void)updateThreadsListTableWithNeedingDisplay:(BOOL)display
{
//	NSTableView *tv = [self threadsListTable];
	AppDefaults *pref = CMRPref;
	BOOL	dontDrawBgColor = [pref threadsListTableUsesAlternatingRowBgColors];
	
    [self.threadListView setRowHeight:[pref threadsListRowHeight]];
    [self.threadListView setFont:[pref threadsListFont]];
    
    [self.threadListView setUsesAlternatingRowBackgroundColors:dontDrawBgColor];
	
	if (!dontDrawBgColor) { // do draw bg color
		[self.threadListView setBackgroundColor:[pref threadsListBackgroundColor]];
	}
	
	[self.threadListView setGridStyleMask:([pref threadsListDrawsGrid] ? NSTableViewSolidVerticalGridLineMask : NSTableViewGridNone)];
	
	[self.threadListView setNeedsDisplay:display];
}
///
- (void)updateTableColumnsMenu
{
	[self updateMenuItemStatusForColumnsMenu:[[[CMRMainMenuManager defaultManager] browserListColumnsMenuItem] submenu]];
	[self updateMenuItemStatusForColumnsMenu:[[self.threadListView headerView] menu]];
}
- (void)setupBrowserContextualMenuLabelNames
{
    NSMenuItem *tmp = [[self menu] itemWithTag:kTLContMenuLabelMenuTag];
    [tmp setView:[BSLabelMenuItemHolder labelMenuItemView]];
    [(BSLabelMenuItemView *)[tmp view] setTarget:self.threadListView];
}    

///
- (void)setupThreadsListTable
{
//    ThreadsListTable    *tbView_ = [self threadsListTable];
	id	tmp2;
	id	tmp;
    
    [self createDefaultTableColumnsWithTableView:self.threadListView];
	
    tmp = SGTemplateResource(kThreadsListTableICSKey);
    UTILAssertRespondsTo(tmp, @selector(stringValue));
    [self.threadListView setIntercellSpacing:NSSizeFromString([tmp stringValue])];
	
	[self updateThreadsListTableWithNeedingDisplay:NO];
	
	tmp2 = [CMRPref threadsListTableColumnState];
	if (tmp2) {
		[self.threadListView restoreColumnState:tmp2];
	}
	
    [self.threadListView setTarget:self];
    [self.threadListView setDelegate:self];
	
    // dispatch in listViewAction:
    [self.threadListView setAction:@selector(listViewAction:)];
    [self.threadListView setDoubleAction:@selector(listViewDoubleAction:)];
	
	// Favorites Item's Drag & Drop operation support:
	[self.threadListView registerForDraggedTypes:[NSArray arrayWithObjects:BSFavoritesIndexSetPboardType, nil]];
	
	[self.threadListView setAutosaveTableColumns:NO];
    [self.threadListView setVerticalMotionCanBeginDrag:NO];
	
    // Menu and Contextual Menus
	[self.threadListView setMenu:[self menu]];
	[[self.threadListView headerView] setMenu:[(CMRAppDelegate *)[NSApp delegate] browserListColumnsMenuTemplate]];
	
	// Leopard
    [self.threadListView setAllowsTypeSelect:NO];
	
	[self updateTableColumnsMenu];
	[self setupBrowserContextualMenuLabelNames];
}

@end

@implementation KMThreadList(KM_NSTableViewDelegate)
static BOOL isOptionKeyDown(void)
{
    NSUInteger flag_ = [NSEvent modifierFlags];
    if (flag_ & NSAlternateKeyMask) {
        return YES;
    } else {
        return NO;
    }
}
- (void)tableView:(NSTableView *)tableView didClickTableColumn:(NSTableColumn *)tableColumn
{
    static BOOL hasOptionClicked = NO;
    BoardManager *bm = [BoardManager defaultManager];
    NSString *boardName = [datasource boardName];
	
    // Sort:
    // カラムヘッダをクリックしたとき、まず
    // -[NSObject(NSTableDataSource) tableView:sortDescriptorsDidChange:] が送られ、
    // その後で -[NSObject(NSTableViewDelegate) tableView:didClickTableColumn:] が送られる。
	
    // Sort:
    // Mac OS標準的ソート変更 (Finderのリスト表示参照)
    // ソートの向きは各カラムごとに保存されており、
    // ハイライトされているカラムヘッダがクリックされた時以外は、
    // 保存されている向きでソートされる。
    // 既にハイライトされているヘッダをクリックした場合は
    // 昇順／降順の切り替えと見なす。
	
    // Sort:
    // option キーを押しながらヘッダをクリックした場合は、変更後の設定を CMRPref に保存する（グローバルな設定の変更）。
    // ただし、option キーを押しながらクリックした場合、sortDescriptorDidChange: は呼ばれない。
    // それどころか、カラムのハイライトも更新されない。
    // 仕方がないので、option キーを押しながらクリックされた場合は、
    // ここでダミーのクリックイベントをもう一度発生させ、通常のカラムヘッダクリックをシミュレートする。
    // ダミーイベントによってもう一度 -tableView:didClickTableColumn: が発生するので、
    // そこで必要な処理を行なう。
    if (isOptionKeyDown()) {
        NSEvent *dummyEvent = [NSApp currentEvent];
        hasOptionClicked = YES;
        // このへん、Thousand のコード（THTableHeaderView.m）を参考にした
        NSEvent *downEvent = [NSEvent mouseEventWithType:NSLeftMouseDown
                                                location:[dummyEvent locationInWindow]
                                           modifierFlags:0
                                               timestamp:[dummyEvent timestamp]
                                            windowNumber:[dummyEvent windowNumber]
                                                 context:[dummyEvent context]
                                             eventNumber:[dummyEvent eventNumber]+1
                                              clickCount:1
                                                pressure:1.0];
        NSEvent *upEvent = [NSEvent mouseEventWithType:NSLeftMouseUp
                                              location:[dummyEvent locationInWindow]
                                         modifierFlags:0
                                             timestamp:[dummyEvent timestamp]
                                          windowNumber:[dummyEvent windowNumber]
                                               context:[dummyEvent context]
                                           eventNumber:[dummyEvent eventNumber]+2
                                            clickCount:1
                                              pressure:1.0];
        [NSApp postEvent:upEvent atStart:NO];
        [NSApp postEvent:downEvent atStart:YES];
		
        return;
    }
	
    // 設定の保存
    [bm setSortDescriptors:[tableView sortDescriptors] forBoard:boardName];
	
    if (hasOptionClicked) {
        [CMRPref setThreadsListSortDescriptors:[tableView sortDescriptors]];
        hasOptionClicked = NO;
    }
	
    NSInteger selected = [tableView selectedRow];
    if (selected != -1) {
        CMRAutoscrollCondition prefMask = [CMRPref threadsListAutoscrollMask];
        if (prefMask & CMRAutoscrollWhenTLSort) {
            [tableView scrollRowToVisible:selected];
        } else {
            [tableView scrollRowToVisible:0];
        }
    } else {
        [tableView scrollRowToVisible:0];
    }
	
    UTILDebugWrite(@"Catch tableView:didClickTableColumn:");
}

- (void)saveBrowserListColumnState:(NSTableView *)targetTableView
{
	if(self.threadListView != targetTableView) return;
    [CMRPref setThreadsListTableColumnState:[self.threadListView columnState]];
	/*    if ([[self splitView] isVertical]) {
	 NSScrollView *scrollView = [targetTableView enclosingScrollView];
	 if (!scrollView) {
	 return;
	 }
	 //        CGFloat tableViewWidth = [targetTableView frame].size.width;
	 //        CGFloat scrollViewWidth = [[scrollView contentView] frame].size.width;
	 //        [scrollView setHasHorizontalScroller:(tableViewWidth > scrollViewWidth)];        
	 }*/
}

- (void)tableViewColumnDidMove:(NSNotification *)aNotification
{
    [self saveBrowserListColumnState:[aNotification object]];
}

- (void)tableViewColumnDidResize:(NSNotification *)aNotification
{
    [self saveBrowserListColumnState:[aNotification object]];
}

@end
