using System;
using System.Drawing;
using System.Windows.Forms;
using nft.framework.plugin;
using System.IO;

namespace nft.framework.drawing
{
    public interface IImageSrc
    {
    }

    public interface ITexture
    {
        UInt32 GetPixel(int x, int y);
        Size Size { get; }
    }

    /// <summary>
	/// IGraphicManager ̊Tv̐łB
	/// </summary>
	public interface IGraphicManager : IGlobalModule{

        /// <summary>
        /// Create a IImageSrc based on specified image file(BMP/GIF/PNG).
        /// May create optimized data in the directory 'data_subdir'.
        /// 'data_subdir' is recommended to contains plugin-ID so that it should not be affected by other plugins.
        /// </summary>
        /// <param name="data_subdir">may used to store some cached data.</param>
        /// <param name="file">file source of an image.</param>
        /// <param name="colorkey">specifies transparent color. (use 'Color.Transparent' for transparent GIF/PNG)</param>
        /// <returns></returns>
        IImageSrc LoadImageFromFile(ResourceKey rkey, IFileSource file, Color colorkey);

        /// <summary>
        /// Create a IImageSrc based on specified image file(BMP/GIF/PNG).
        /// May create optimized data in the directory 'data_subdir'.
        /// 'data_subdir' is recommended to contains plugin-ID so that it should not be affected by other plugins.
        /// for BMP the color at left bottom pixel is used for transpalent color.
        /// for GIF/PNG original transpalency is accepted.
        /// </summary>
        /// <param name="data_subdir">may used to store some cached data.</param>
        /// <param name="file">file source of an image.</param>
        /// <returns></returns>
        IImageSrc LoadImageFromFileWithDefaultColorKey(ResourceKey rkey, IFileSource file);

        /// <summary>
        /// Creates a primitive ITexture instance from specified region of image.
        /// </summary>
        /// <param name="src"></param>
        /// <param name="region"></param>
        /// <returns></returns>
        ITexture CreateStaticTexture(ResourceKey rkey, IImageSrc src, Rectangle region);
        
        /// <summary>
        /// Creates a primitive ITexture instance which based on an image.
        /// Use whole image as Texture.
        /// </summary>
        /// <param name="src"></param>
        /// <returns></returns>
        ITexture CreateStaticTexture(ResourceKey rkey, IImageSrc src);

        /// <summary>
        /// Create terrain piece template with specified texture.
        /// </summary>
        /// <param name="texture"></param>
        /// <param name="vertices">(if vertices.Length>3, these are divided into multi triangles)</param>
        /// <param name="texture_pos">texture pixel position corresponds to each vertex.</param>
        /// <param name="brightness"></param>
        /// <returns></returns>
        ITerrainPlateTemplate CreateTerrainTemplate(ResourceKey rkey, ITexture texture, Point3D[] vertices, PointF[] texture_pos, float brightness);

        /// <summary>
        /// Create terrain piece template with specified color.
        /// </summary>
        /// <param name="basecol"></param>
        /// <param name="vertices">vertices (if vertices.Length>3, these are divided into  multi triangles)</param>
        /// <param name="brightness"></param>
        /// <returns></returns>
        ITerrainPlateTemplate CreateTerrainTemplate(ResourceKey rkey, Color basecol, Point3D[] vertices, float brightness);

        /// <summary>
        /// Create terrain piece from specified template.
        /// </summary>
        /// <param name="template"></param>
        /// <returns></returns>
        ITerrainPlate CreateTerrainPlate(UInt32 id, ITerrainPlateTemplate template);

		/// <summary>
		/// Creates an instance of Structure, as 3D model object.
		/// </summary>
		/// <param name="texture"></param>
		/// <returns></returns>
        ICubicStructure CreateStructure(UInt32 id, ResourceKey rkey, ITexture texture, Point3D size3d);

        /// <summary>
        /// Creates an instance of Billboard object.
        /// </summary>
        /// <param name="texture"></param>
        /// <returns></returns>
        IBillboard CreateBillboard(ITexture texture);

        /// <summary>
        /// Creates an instnace of Overlay (or sprite).
        /// </summary>
        /// <param name="texture"></param>
        /// <returns></returns>
        IOverlay CreateOverlay(ITexture texture);

		/// <summary>
		/// Creates a DrawableControl which contains primary surface
		/// </summary>
        /// <param name="parent"></param>
        /// <param name="eventtarget">The Control to which attach event handlers, May or may not be same as 'parent'.</param>
        /// <returns></returns>
        ISurface CreateSurfaceForControl(Control parent, out Control eventtarget, SurfaceUsage usage);

        /// <summary>
        /// Creates an off-screen surface with specified size.
        /// </summary>
        /// <param name="size"></param>
        /// <returns></returns>
        ISurface CreateOffscreenSurface(Size size, SurfaceUsage usage);

        /// <summary>
        /// Get Effect filter for specific purpose. 
        /// Representation is GraphicsManager module dependent.
        /// </summary>
        /// <returns></returns>
        IEffectFilter GetFilterByUsage();
	}
}
