﻿using System;
using System.Collections.Generic;
using System.Text;
using nft.core.view;

namespace nft.core.schedule {
    public interface IConditionValueExtracter {
        Type ComparableType { get; }
        IComparable GetTestValue(ISceneParams p);
        ConditionTypes ConditionType { get; }
    }

    public abstract class LongConditionValueExtracter : IConditionValueExtracter {
        protected ConditionTypes condtype;
        public LongConditionValueExtracter(ConditionTypes ctype) {
            this.condtype = ctype;
        }

        public ConditionTypes ConditionType {
            get {
                return condtype;
            }
        }

        public Type ComparableType {
            get {
                return typeof(long);
            }
        }

        public abstract IComparable GetTestValue(ISceneParams p);
    }

    public class TimeConditionValueExtracter : LongConditionValueExtracter {
        TimeUnit unit;
        public TimeConditionValueExtracter(ConditionTypes ctype, TimeUnit unit)
            : base(ctype) {
                this.unit = unit;
        }

        public override IComparable GetTestValue(ISceneParams p) {
            return p.Time.GetUnitValue((TimeUnit)condtype, unit);
        }
    }

    public class MonthConditionValueExtracter : LongConditionValueExtracter {
        public MonthConditionValueExtracter()
            : base(ConditionTypes.Yearly) {
        }

        public override IComparable GetTestValue(ISceneParams p) {
            return (long)p.Time.Month;
        }
    }

    public class FormattedTimeNumsConditionExtracter : LongConditionValueExtracter {
        string format;
        public FormattedTimeNumsConditionExtracter(ConditionTypes ctype, string date_format)
            : base(ctype) {
            this.format = date_format;
        }

        public override IComparable GetTestValue(ISceneParams p) {
            return long.Parse(p.Time.ToString(format));
        }
    }

    public class FormattedTimeStrConditionExtracter : IConditionValueExtracter {
        string format;
        ConditionTypes condtype;
        public FormattedTimeStrConditionExtracter(ConditionTypes ctype, string date_format) {
            this.condtype = ctype;
            this.format = date_format;
        }

        public IComparable GetTestValue(ISceneParams p) {
            return p.Time.ToString(format);
        }

        public Type ComparableType {
            get {
                return typeof(string);
            }
        }

        public ConditionTypes ConditionType {
            get {
                return condtype;
            }
        }
    }

    public class ScaleConditionValueExtracter : IConditionValueExtracter {
        public ScaleConditionValueExtracter(){            
        }
        public IComparable GetTestValue(ISceneParams p) {
            return p.Scaler.Value;
        }

        public ConditionTypes ConditionType {
            get {
                return ConditionTypes.Scale;
            }
        }

        public Type ComparableType {
            get {
                return typeof(double);
            }
        }
    }
}
