// Programming 2D Games
// Copyright (c) 2011 by: 
// Charles Kelly
// Draw animated spaceships with collision and shield
// Chapter 6 spacewar.cpp v1.0
// This class is the core of the game

#include "spaceWar.h"

//=============================================================================
// Constructor
//=============================================================================
Spacewar::Spacewar()
{}

//=============================================================================
// Destructor
//=============================================================================
Spacewar::~Spacewar()
{
    releaseAll();           // call onLostDevice() for every graphics item
}

//=============================================================================
// Initializes the game
// Throws GameError on error
//=============================================================================
void Spacewar::initialize(HWND hwnd)
{
    Game::initialize(hwnd); // throws GameError

    // nebula texture
    if (!nebulaTexture.initialize(graphics,NEBULA_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing nebula texture"));

    // main game textures
    if (!gameTextures.initialize(graphics,TEXTURES_IMAGE))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing game textures"));

    // nebula image
    if (!nebula.initialize(graphics,0,0,0,&nebulaTexture))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing nebula"));

    // planet
    if (!planet.initialize(this, planetNS::WIDTH, planetNS::HEIGHT, 2, &gameTextures))
        throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing planet"));

    // ship
    for (int i = 0;i < MAX_SHIPS;++i) {
        if (!ship[i].initialize(this, shipNS::WIDTH, shipNS::HEIGHT, shipNS::TEXTURE_COLS, &gameTextures))
            throw(GameError(gameErrorNS::FATAL_ERROR, "Error initializing ship[i]"));
        if ( i & 1 ) {
            ship[i].setFrames(shipNS::SHIP1_START_FRAME, shipNS::SHIP1_END_FRAME);
            ship[i].setCurrentFrame(shipNS::SHIP1_START_FRAME);
        } else {
            ship[i].setFrames(shipNS::SHIP2_START_FRAME, shipNS::SHIP2_END_FRAME);
            ship[i].setCurrentFrame(shipNS::SHIP2_START_FRAME);
        }
        ship[i].setX(GAME_WIDTH/4 + ( (GAME_WIDTH/2) / (MAX_SHIPS/2) * ( i % (MAX_SHIPS/2) ) ));
        ship[i].setY(GAME_HEIGHT/4 + ( (GAME_HEIGHT/2) * ( i / (MAX_SHIPS/2) ) ));

        float   vx, vy;
        if ( i & 1 ) {
            vx = shipNS::SPEED;
        } else {
            vx = -shipNS::SPEED;
        }
        if ( i / (MAX_SHIPS/2) ) {
            vy = shipNS::SPEED;
        } else {
            vy = -shipNS::SPEED;
        }
        ship[i].setVelocity(VECTOR2( vx, vy )); // VECTOR2(X, Y)
    }

    return;
}

//=============================================================================
// Update all game items
//=============================================================================
void Spacewar::update()
{
    planet.update(frameTime);
    for (int i = 0;i < MAX_SHIPS;++i) {
        ship[i].update(frameTime);
    }
}

//=============================================================================
// Artificial Intelligence
//=============================================================================
void Spacewar::ai()
{}

//=============================================================================
// Handle collisions
//=============================================================================
void Spacewar::collisions()
{
    VECTOR2 collisionVector;

    for (int i = 0;i < MAX_SHIPS;++i) {
        // if collision between ship and planet
        if(ship[i].collidesWith(planet, collisionVector))
        {
            // bounce off planet
            ship[i].bounce(collisionVector, planet);
            ship[i].damage(PLANET);
        }

        for (int j = i + 1;j < MAX_SHIPS;++j) {
            // if collision between ships
            if(ship[i].collidesWith(ship[j], collisionVector))
            {
                // bounce off ship
                ship[i].bounce(collisionVector, ship[j]);
                ship[i].damage(SHIP);
                // change the direction of the collisionVector for ship[j]
                ship[j].bounce(collisionVector*-1, ship[i]);
                ship[j].damage(SHIP);
            }
        }
    }
}

//=============================================================================
// Render game items
//=============================================================================
void Spacewar::render()
{
    graphics->spriteBegin();                // begin drawing sprites

    nebula.draw();                          // add the orion nebula to the scene
    planet.draw();                          // add the planet to the scene
    for (int i = 0;i < MAX_SHIPS;++i) {
        ship[i].draw();                     // add the spaceship to the scene
    }

    graphics->spriteEnd();                  // end drawing sprites
}

//=============================================================================
// The graphics device was lost.
// Release all reserved video memory so graphics device may be reset.
//=============================================================================
void Spacewar::releaseAll()
{
    nebulaTexture.onLostDevice();
    gameTextures.onLostDevice();
    Game::releaseAll();
    return;
}

//=============================================================================
// The grahics device has been reset.
// Recreate all surfaces.
//=============================================================================
void Spacewar::resetAll()
{
    gameTextures.onResetDevice();
    nebulaTexture.onResetDevice();
    Game::resetAll();
    return;
}
