﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Net;
using System.Threading;
using System.Xml.Linq;
using Codeplex.Data;
using LinqToTwitter;

namespace Azyobuzi.UserStreamEx
{
    public class UserStream
    {
        public UserStream(ITwitterAuthorizer auth)
        {
            this.auth = auth;
        }

        private ITwitterAuthorizer auth;
        private HttpWebRequest req;

        public void Start(string track, bool allReplies)
        {
            var reqUri = new UriBuilder("https://userstream.twitter.com/2/user.json");
            var param = new Dictionary<string, string>();
            if (!string.IsNullOrEmpty(track)) param.Add("track", track);
            if (allReplies) param.Add("replies", "all");
            reqUri.Query = string.Join("&", param.Select(_ => string.Format("{0}={1}", _.Key, _.Value)));

            req = auth.Get(reqUri.ToString()) as HttpWebRequest;
            req.BeginGetResponse(result =>
            {
                try
                {
                    if (Started != null) Started(this, EventArgs.Empty);
                    var res = req.EndGetResponse(result);
                    using (var sr = new StreamReader(res.GetResponseStream()))
                    {
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();

                            if (StreamingWriter != null)
                            {
                                try
                                {
                                    StreamingWriter.WriteLine(line);
                                    StreamingWriter.Flush();
                                }
                                catch { }
                            }

                            if (string.IsNullOrWhiteSpace(line)) continue;

                            var t = new Thread(_ =>
                            {
                                var _line = (string)_;
                                var json = DynamicJson.Parse(_line);
                                if (json.friends())
                                {
                                    if (ReceiveFriends != null) ReceiveFriends(this, new ReceiveFriendsEventArgs(_line));
                                }
                                else if (json.delete())
                                {
                                    if (DeleteStatus != null) DeleteStatus(this, new DeleteStatusEventArgs(_line));
                                }
                                else if (json.direct_message())
                                {
                                    if (NewDirectMessage != null) NewDirectMessage(this, new NewDirectMessageEventArgs(_line));
                                }
                                else if (json.@event())
                                {
                                    if (ReceiveEvent != null) ReceiveEvent(this, new ReceiveEventEventArgs(_line));
                                }
                                else if (json.text())
                                {
                                    if (NewTweet != null) NewTweet(this, new NewTweetEventArgs(_line));
                                }
                                else
                                {
                                    if (ReceiveUnsupportedData != null) ReceiveUnsupportedData(this, new ReceiveJsonEventArgs(_line));
                                }
                            });
                            t.Start(line);
                        }
                    }

                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(userStop ? StopReason.UserStop : StopReason.CloseResponse, null));
                    userStop = false;
                }
                catch (Exception ex)
                {
                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(StopReason.Error, ex));
                }
                req = null;
            }, null);
        }

        private bool userStop;

        public void Stop()
        {
            userStop = true;
            if (req != null) req.Abort();
            req = null;
        }

        public TextWriter StreamingWriter { set; get; }

        public event EventHandler Started;
        public event EventHandler<StoppedEventArgs> Stopped;
        public event EventHandler<ReceiveFriendsEventArgs> ReceiveFriends;
        public event EventHandler<NewTweetEventArgs> NewTweet;
        public event EventHandler<NewDirectMessageEventArgs> NewDirectMessage;
        public event EventHandler<DeleteStatusEventArgs> DeleteStatus;
        public event EventHandler<ReceiveEventEventArgs> ReceiveEvent;
        public event EventHandler<ReceiveJsonEventArgs> ReceiveUnsupportedData;
    }

    public class StoppedEventArgs : EventArgs
    {
        public StoppedEventArgs(StopReason reason, Exception ex)
        {
            this.Reason = reason;
            this.Exception = ex;
        }

        public StopReason Reason { private set; get; }
        public Exception Exception { private set; get; }
    }

    public enum StopReason
    {
        /// <summary>
        /// Twitterが切断
        /// </summary>
        CloseResponse,
        /// <summary>
        /// Stopメソッドが呼ばれた
        /// </summary>
        UserStop,
        /// <summary>
        /// その他の例外
        /// </summary>
        Error
    }

    public class ReceiveJsonEventArgs : EventArgs
    {
        public ReceiveJsonEventArgs(string line)
        {
            this.Line = line;
        }

        public string Line { private set; get; }
    }

    public class ReceiveFriendsEventArgs : ReceiveJsonEventArgs
    {
        public ReceiveFriendsEventArgs(string line)
            : base(line)
        {
            FriendIds = DynamicJson.Parse(line).friends;
        }

        public string[] FriendIds { private set; get; }
    }

    public class NewTweetEventArgs : ReceiveJsonEventArgs
    {
        public NewTweetEventArgs(string line)
            : base(line)
        {
            Status = Status.CreateStatus(line.JsonToXml());
        }

        public Status Status { private set; get; }
    }

    public class NewDirectMessageEventArgs : ReceiveJsonEventArgs
    {
        public NewDirectMessageEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line).direct_message;
            DirectMessage = new DirectMessage()
            {
                ID = (ulong)json.id,
                SenderID = (ulong)json.sender_id,
                Text = json.text,
                RecipientID = json.recipient_id,
                CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture),
                SenderScreenName = json.sender_screen_name,
                RecipientScreenName = json.recipient_screen_name,
                Sender = User.CreateUser(((string)json.sender).JsonToXml()),
                Recipient = User.CreateUser(((string)json.recipient).JsonToXml())
            };
        }

        public DirectMessage DirectMessage { private set; get; }
    }

    public class DeleteStatusEventArgs : ReceiveJsonEventArgs
    {
        public DeleteStatusEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line);
            if (!json.direct_message())
            {
                //ツイート
                StatusId = json.status.id_str;
                UserId = json.status.user_id_str;
                IsDirectMessage = false;
            }
            else
            {
                //DM
                StatusId = json.direct_message.id.ToString();
                UserId = json.direct_message.user_id.ToString();
                IsDirectMessage = true;
            }
        }

        public string StatusId { private set; get; }
        public string UserId { private set; get; }
        public bool IsDirectMessage { private set; get; }
    }

    public class ReceiveEventEventArgs : ReceiveJsonEventArgs
    {
        public ReceiveEventEventArgs(string line)
            : base(line)
        {
            var json = DynamicJson.Parse(line);
            EventTypes eventType;
            if (!Enum.TryParse(((string)json.@event).Replace("_", ""), true, out eventType))
                eventType = EventTypes.Unknown;
            EventType = eventType;
            CreatedAt = DateTime.ParseExact(json.created_at, "ddd MMM dd HH:mm:ss %zzzz yyyy", CultureInfo.InvariantCulture).ToLocalTime();
            Source = User.CreateUser(((string)json.source).JsonToXml());
            Target = User.CreateUser(((string)json.target).JsonToXml());
            if (json.target_object())
            {
                if (json.target_object.mode())
                {
                    //リスト
                    TargetList = List.CreateList(((string)json.target_onject.ToString()).JsonToXml(), new XElement("lists_list"));
                }
                else
                {
                    //ツイート
                    TargetStatus = Status.CreateStatus(((string)json.target_object.ToString()).JsonToXml());
                }
            }
        }

        public EventTypes EventType { private set; get; }
        public DateTime CreatedAt { private set; get; }
        public User Source { private set; get; }
        public User Target { private set; get; }
        public Status TargetStatus { private set; get; }
        public List TargetList { private set; get; }
    }

    public enum EventTypes
    {
        Unknown,
        Favorite,
        Unfavorite,
        Follow,
        ListMemberAdded,
        ListMemberRemoved,
        Block,
        Unblock,
        UserUpdate,
        ListCreated
    }
}
