package portablesimulator.decoration;

import java.util.List;
import portablesimulator.PSArmorSet;
import portablesimulator.PSItem;
import portablesimulator.PSItemType;
import portablesimulator.PSWrap;
import portablesimulator.gui.PSFrame;
import portablesimulator.skillset.SkillSet;

public class DecorationSlot implements Comparable {

    public final int[] slotCount = new int[7];
    public final int[] slotAvail = new int[7];
    public final DecorationCount mapDecoration;
    public final DecorationCount mapBodyDecoration;
    public final SkillSet diffSkills = new SkillSet();

    public PSArmorSet armorSet;

    private DecorationSlot(DecorationSlot from) {
        mapDecoration = from.mapDecoration.makeCopy();
        mapBodyDecoration = from.mapBodyDecoration.makeCopy();

        for(int i = 0; i < slotCount.length; ++ i) {
            slotCount[i] = from.slotCount[i];
            slotAvail[i] = from.slotAvail[i];
        }
        armorSet = from.armorSet;
        diffSkills.clear();
        diffSkills.set_all(from.diffSkills);
    }

    public DecorationSlot(List<PSItem> decorations) {
        mapDecoration = new DecorationCount(decorations);
        mapBodyDecoration = new DecorationCount(decorations);
    }

    public DecorationSlot(List<PSItem> decorations, SkillSet diffSkills, PSArmorSet set, int weaponSlotCount) {
        mapDecoration = new DecorationCount(decorations);
        mapBodyDecoration = new DecorationCount(decorations);
        construct(diffSkills, set, weaponSlotCount);
    }

    public void construct(SkillSet diffSkills, PSArmorSet set, int weaponSlotCount) {
        this.mapDecoration.clear();
        this.mapBodyDecoration.clear();
        this.armorSet = set;
        this.diffSkills.clear();
        this.diffSkills.set_all(diffSkills);

        int x = 1;
        for (int i = 0; i < set.listItems.size(); ++i) {
            PSWrap wrap = set.listItems.get(i);
            PSItem item = wrap.item;
            if (item.itemType == PSItemType.TYPE_BODY) {
                slotCount[0] = item.slotCount;
            } else {
                if (item.isCopieSkill) {
                    slotCount[x] = -1;
                    x ++;
                } else {
                    slotCount[x] = item.slotCount;
                    x ++;
                }
            }
        }
        slotCount[x] = weaponSlotCount;
        for (int i = 0; i < slotAvail.length; ++ i) {
            slotAvail[i] = slotCount[i] < 0 ? 0 : slotCount[i];
        }
    }

    public int totalAvailCount() {
        int total = 0;
        for (int i = 0; i < slotCount.length; ++i) {
            if (slotCount[i] == -1) {
                total += slotAvail[0];
            } else {
                total += slotAvail[i];
            }
        }
        return total;
    }

    public int getBodyAvailable() {
        if (armorSet.bodyCopieCount >= 1) {
            return slotAvail[0];
        }
        return 0;
    }

    public boolean haveCopie() {
        return armorSet.bodyCopieCount > 0;
    }

    public DecorationSlot applyToBody(PSItem deco) {
        if (armorSet.bodyCopieCount >= 1 && slotAvail[0] >= deco.slotCount) {
            DecorationSlot cache = new DecorationSlot(this);
            cache.slotAvail[0] -= deco.slotCount;
            cache.mapBodyDecoration.increment(deco);
            for (int x = 0; x <= armorSet.bodyCopieCount; ++x) {
                cache.diffSkills.minus_only(deco.skills);
            }
            return cache;
        }
        return null;
    }

    public DecorationSlot applyAuto(PSItem deco) {
        int foundX = -1;
        int foundWidth = -1;
        for (int i = armorSet.bodyCopieCount > 0 ? 1 : 0; i < slotCount.length; ++i) {
            if (slotAvail[i] >= deco.slotCount) {
                //if (foundX < 0) {
                    foundX = i;
                    foundWidth = slotAvail[i];
                    break;
                /*}else if (foundWidth > slotAvail[i]) {
                    foundX = i;
                    foundWidth = slotAvail[i];*/
                //}
            }
        }

        if (foundX < 0) {
            return null;
        }
                
        if (foundX >= 0) {
            DecorationSlot cache = new DecorationSlot(this);
            cache.diffSkills.minus_only(deco.skills);
            cache.slotAvail[foundX] -= deco.slotCount;
            cache.mapDecoration.increment(deco);
            return cache;
        }
        return null;
    }

    public int getSlotMaxCount() {
        int width = 0;
        for (int i = armorSet.bodyCopieCount > 0 ? 1 : 0; i < slotCount.length; ++i) {
            if (width < slotAvail[i]) {
                width = slotAvail[i];
            }
        }
        if (armorSet.bodyCopieCount > 0 && width < slotAvail[0]) {
            width = slotAvail[0];
        }
        return width;
    }

    public int getWeaponSlotAvailable() {
        int weaponSlotCount = armorSet.weaponSlotCount;
        int width = 0;
        for (int i = armorSet.bodyCopieCount > 0 ? 1 : 0; i < slotCount.length; ++i) {
            if (slotCount[i] >= weaponSlotCount) {
                if (width < slotAvail[i]) {
                    width = slotAvail[i];
                }
            }
        }
        return width;
    }

    public int getCharmSlotAvailable() {
        int charmSlotCount = slotCount[5];
        int width = 0;
        for (int i = armorSet.bodyCopieCount > 0 ? 1 : 0; i < slotCount.length; ++i) {
            if (slotCount[i] >= charmSlotCount) {
                if (width < slotAvail[i]) {
                    width = slotAvail[i];
                }
            }
        }
        return width;
    }

    public SkillSet getFullDecorationSkills() {
        SkillSet skills = new SkillSet();

        for (int x = 0; x < mapDecoration.size(); ++ x) {
            PSItem deco = mapDecoration.deco(x);
            int count = mapDecoration.count(x);
            for(int i = 0; i < count; ++ i) {
                skills.sum_all(deco.skills);
            }
        }
        for (int x = 0; x < mapBodyDecoration.size(); ++ x) {
            PSItem deco = mapBodyDecoration.deco(x);
            int count = mapBodyDecoration.count(x);
            for (int i = 0; i < count; ++ i) {
                for (int j = 0; j <= armorSet.bodyCopieCount; ++ j) {
                    skills.sum_all(deco.skills);
                }
            }
        }
        return skills;
    }

    private int[] copyIntArray(int[] from) {
        int[] to = new int[from.length];
        for (int i = 0; i < from.length; ++i) {
            to[i] = from[i];
        }
        return to;
    }

    public String simpleToString(DecorationCount map) {
        StringBuilder str = new StringBuilder();
        for (int i = 0; i < map.size(); ++ i) {
            PSItem deco = map.deco(i);
            int count = map.count(i);

            if (count == 0) {
                continue;
            }

            if (str.length() > 0) {
                str.append("A");
            }

            str.append(deco.name);
            str.append("~");
            str.append(PSFrame.Zenkaku(String.valueOf(count)));
        }
        return str.toString();
    }

    public String toDecorationString() {
        StringBuilder str = new StringBuilder();
        if (mapBodyDecoration.isEmpty() == false) {
            str.append("yF" + simpleToString(mapBodyDecoration) + "z");
        }
        str.append(simpleToString(mapDecoration));
        return str.toString();
    }

    public int countEmptyForDisp() {
        int count = 0;
        for (int i = 0; i < slotAvail.length; ++i) {
            if (slotAvail[i] >= 1) {
                count += slotAvail[i];
            }
        }
        return count;
    }

    public int compareTo(Object t) {
        DecorationSlot target = (DecorationSlot)t;

        int x = 0;

        if (x == 0) {
            x = totalAvailCount() - target.totalAvailCount();
        }
        if (x == 0) {
            x = mapDecoration.compareTo(target.mapDecoration);
        }
        if (x == 0) {
            x = mapBodyDecoration.compareTo(target.mapBodyDecoration);
        }
        if (x == 0) {
            for (int i = 0; i < slotAvail.length; ++ i) {
                x = slotAvail[i] - target.slotAvail[i];
                if (x != 0) {
                    break;
                }
            }
        }
        if (x < 0) {
            return -1;
        }
        if (x > 0) {
            return 1;
        }
        return 0;
    }

    public String toString() {
        return "[" + totalAvailCount() + " avail, " + getFullDecorationSkills() + "\n" + toDecorationString() + "]";
    }
}
