# coding: UTF-8

require 'hikidoc/hikidoc'

require 'cgi'

require 'plugin/format/lib/format_plugin_error'
require 'plugin/format/lib/format_plugin_warning'

M2W_FORMATTER_MARK_VERSION = "m2w_version"
M2W_FORMATTER_MARK_WIKICONTENT = "wikicontent"
M2W_FORMATTER_MARK_ATTACHMENT_URLS = "attachment_urls"
M2W_FORMATTER_COMMENTOUT_REGEX = /^(<!--#{M2W_FORMATTER_MARK_VERSION}:[0-9]{2}\.[0-9]{2}\.[0-9]{2}-->)?(<!--\[#{M2W_FORMATTER_MARK_WIKICONTENT}\] .+? -->)?(<!--\[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}\] .+? -->)?(<!-- .+? -->)?/m
M2W_PLUGIN_NAME_REGEX = "[a-zA-Z_][0-9a-zA-Z_]*"

#=本文のフォーマット変換を行います。
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: formatter.rb 501 2012-02-01 02:35:01Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class Formatter

	attr_reader :has_plugin_content

	def initialize(plugin_manager)
		@plugin_manager = plugin_manager
	end

	def preset(original_content, content, contents, vars, errors, warnings)
		M2W_LOGGER.debug("Start  #{self.class}#preset ... content = #{content}, vars = #{vars}")

		formatted_content = HikiDoc.to_xhtml(content, M2W_HIKI_OPTIONS).encode(M2W_SYSTEM_INNER_ENCODING, M2W_SYSTEM_HIKIDOC_ENCODING)

		__calls_plugin_preset(formatted_content, vars, errors, warnings)

		M2W_LOGGER.debug("Finish #{self.class}#preset ... vars = #{vars}")
	end

	def format(original_content, content, contents, vars, errors, warnings)
		M2W_LOGGER.debug("Start  #{self.class}#format ... original_content, contents = #{contents}, vars = #{vars}")

		# 全てのコンテンツに書式プラグインを適用する
		result = {'description' => ''}
		contents.each do |name, each_content|
			formatted_content = HikiDoc.to_xhtml(each_content, M2W_HIKI_OPTIONS).encode(M2W_SYSTEM_INNER_ENCODING, M2W_SYSTEM_HIKIDOC_ENCODING)
			result[name] = __calls_plugin_content(formatted_content, vars, errors, warnings)
		end

		description = __get_commentout_content(original_content)
		description << __get_commentout_attachment_urls_content(vars != nil ? vars[:attachment_urls] : {})
		description << result['description']
		result['description'] = description

		M2W_LOGGER.debug("Finish #{self.class}#format ... result = #{result}")

		return result
	end

	def escape(content)
		return CGI.escapeHTML(content)
	end

	def unescape(content)
		return CGI.unescapeHTML(content)
	end

	def get_uncommentout_content(content)
		M2W_LOGGER.debug("Start  #{self.class.name}#get_uncommentout_content")

		result_version = nil
		result_wikicontent = nil
		if M2W_FORMATTER_COMMENTOUT_REGEX =~ content then
			version = $1
			wikicontent = $2
			old_wikicontent = $4
			M2W_LOGGER.debug("match ... version = #{version}, wikicontent = #{wikicontent}, old_wikicontent = #{old_wikicontent}")

			if version != nil && wikicontent != nil then
				if /^<!--#{M2W_FORMATTER_MARK_VERSION}:([0-9]{2}\.[0-9]{2}\.[0-9]{2})-->$/ =~ version then
					result_version = $1
				end

				if /^<!--\[#{M2W_FORMATTER_MARK_WIKICONTENT}\] (.+?) -->$/m =~ wikicontent then
					result_wikicontent = unescape($1)
					result = true
				end
			elsif old_wikicontent != nil then
				M2W_LOGGER.debug('old_wikicontent != ""')
				if /^<!-- *(.+?) *-->$/m =~ old_wikicontent then
					result_wikicontent = unescape($1)
					M2W_LOGGER.debug("match old_wikicontent ... #{$1}")
					result = true
				else
					M2W_LOGGER.debug('no match old_wikicontent')
				end
			end
		else
			M2W_LOGGER.debug("no match")
		end

		M2W_LOGGER.debug("Finish #{self.class.name}#get_uncommentout ... version = #{result_version}, wikicontent = #{result_wikicontent}")

		return [result_version, result_wikicontent]
	end

	def get_uncommentout_attachment_urls_content(content)
		M2W_LOGGER.debug("Start  #{self.class.name}#get_uncommentout_attachment_urls_content")

		result = {}
		if M2W_FORMATTER_COMMENTOUT_REGEX =~ content then
			attachment_urls = $3
			M2W_LOGGER.debug("match ... version = attachment_urls = #{attachment_urls}")

			if /^<!--\[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}\] (.+?) -->$/ =~ attachment_urls then
				urls_map = $1.split(',')
				urls_map.each do |url_map|
					if url_map =~ /^([0-9]+)=(.*)$/ then
						index = $1
						url = $2
						if url.length > 0 then
							result[index] = url
						else
							result[index] = nil
						end
					end
				end
			end
		else
			M2W_LOGGER.debug("no match")
		end

		M2W_LOGGER.debug("Finish #{self.class.name}#get_uncommentout_attachment_urls_content ... result = #{result.to_s}")
		return result
	end

	def __calls_plugin_preset(content, vars, errors, warnings)
		M2W_LOGGER.debug("Start  #{self.class}#__calls_plugin_preset ... content = #{content}, vars = #{vars}")

		# プラグインを事前処理する
		while %r!(.*?)<(div|span) class="plugin">\{\{(.+?)\}\}</\2>(.*)$!m =~ content
			plugin_type = $2
			plugin_content = $3
			content = $4

			begin
				case plugin_type
				when "div"
					vars.deep_merge!(__call_plugin_preset(plugin_content, vars))
				when "span"
					vars.deep_merge!(__call_plugin_preset_inline(plugin_content, vars))
				end
			rescue FormatPluginWarning => warning
				warnings.push(warning)
			rescue FormatPluginError => error
				errors.push(error)
			end
  	end

		M2W_LOGGER.debug("Finish #{self.class}#__calls_plugin_preset ... vars = #{vars}")
	end

	def __calls_plugin_content(content, vars, errors, warnings)
		M2W_LOGGER.debug("Start  #{self.class}#__calls_plugin_content ... content = #{content}, vars = #{vars}")

		# プラグインを処理する
		result = ""
		while %r!(.*?)<(div|span) class="plugin">\{\{(.+?)\}\}</\2>(.*)$!m =~ content
			left = $1
			plugin_type = $2
			plugin_content = $3
			content = $4

			result << left

			begin
				case plugin_type
				when "div"
					result << __call_plugin_contents(plugin_content, vars)
				when "span"
					result << __call_plugin_contents_inline(plugin_content, vars)
				end
			rescue FormatPluginWarning => warning
				result << "{{警告: #{warning.content}}}"
				warnings.push(warning)
			rescue FormatPluginError => error
				result << "{{エラー: #{error.content}}}"
				errors.push(error)
			end
  	end

		result << content

		M2W_LOGGER.debug("Finish #{self.class}#__calls_plugin_content ... result = #{result}")

		return result
	end

	def __call_plugin_preset(content, vars)
		M2W_LOGGER.debug("Start  #{self.class}#__call_plugin_preset ... content = #{content}, vars = #{vars}")

		(plugin_name, param) = __parse_plugin_content(content)

		result = nil
		begin
			result = @plugin_manager.call_format_plugin_preset(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			M2W_LOGGER.error(e)
			raise e
		rescue FormatPluginError => e
			M2W_LOGGER.error(e)
			raise e
		rescue Exception => e
			M2W_LOGGER.error("プラグイン事前準備中に例外が発生しました。")
			M2W_LOGGER.error(e)
			raise FormatPluginError.new(
				"プラグイン事前準備中例外(コンテンツ = #{content})",
				"プラグイン事前準備中に例外が発生しました（コンテンツ = #{content}）。")
		ensure
			M2W_LOGGER.debug("Finish #{self.class}#__call_plugin_preset ... result = #{result}")
		end

		return result
	end

	def __call_plugin_preset_inline(content, vars)
		M2W_LOGGER.debug("Start  #{self.class}#__call_plugin_preset_inline ... content = #{content}, vars = #{vars}")

		(plugin_name, param) = __parse_plugin_content(content)

		result = nil
		begin
			result = @plugin_manager.call_format_plugin_preset_inline(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			M2W_LOGGER.error(e)
			raise e
		rescue FormatPluginError => e
			M2W_LOGGER.error(e)
			raise e
		rescue Exception => e
			M2W_LOGGER.error("プラグイン事前準備中に例外が発生しました。")
			M2W_LOGGER.error(e)
			raise FormatPluginError.new(
				"プラグイン事前準備中例外(コンテンツ = #{content})",
				"プラグイン事前準備中に例外が発生しました（コンテンツ = #{content}）。")
		ensure
			M2W_LOGGER.debug("Finish #{self.class}#__call_plugin_preset_inline ... result = #{result}")
		end

		return result
	end

	def __call_plugin_contents(content, vars)
		M2W_LOGGER.debug("Start  #{self.class}#__call_plugin_contents ... content = #{content}, vars = #{vars}")

		(plugin_name, param) = __parse_plugin_content(content)

		result = ""
		begin
			result = @plugin_manager.call_format_plugin_contents(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			M2W_LOGGER.error(e)
			raise e
		rescue FormatPluginError => e
			M2W_LOGGER.error(e)
			raise e
		rescue Exception => e
			M2W_LOGGER.error("プラグイン動作時に例外が発生しました。")
			M2W_LOGGER.error(e)
			raise FormatPluginError.new(
				"プラグイン動作時例外(コンテンツ = #{content})",
				"プラグイン動作時例外が発生しました（コンテンツ = #{content}）。")
		ensure
			M2W_LOGGER.debug("Finish #{self.class}#__call_plugin_contents ... result = #{result}")
		end

		return result
	end

	def __call_plugin_contents_inline(content, vars)
		M2W_LOGGER.debug("Start  #{self.class}#call_plugin_contents_inline ... content = #{content}, vars = #{vars}")

		(plugin_name, param) = __parse_plugin_content(content)

		result = ""
		begin
			result = @plugin_manager.call_format_plugin_contents_inline(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			M2W_LOGGER.error(e)
			raise e
		rescue FormatPluginError => e
			M2W_LOGGER.error(e)
			raise e
		rescue => e
			M2W_LOGGER.error("プラグイン動作時に例外が発生しました。")
			M2W_LOGGER.error(e)
			raise FormatPluginError.new(
				"プラグイン動作時例外(コンテンツ = #{content})",
				"プラグイン動作時例外が発生しました（コンテンツ = #{content}）。")
		ensure
			M2W_LOGGER.debug("Finish #{self.class}#__call_plugin_contents_inline ... result = #{result}")
		end

		return result
	end

	def __get_commentout_content(original_content)
		return "<!--#{M2W_FORMATTER_MARK_VERSION}:#{M2W_VERSION}--><!--[#{M2W_FORMATTER_MARK_WIKICONTENT}] #{escape(original_content)} -->"
	end

	def __get_commentout_attachment_urls_content(attachment_urls)
		result = "<!--[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}] "
		if attachment_urls != nil then
			attachment_urls.each do |index, attachment_url|
				result << "#{index}=#{attachment_url}"
				if index != attachment_urls.size then
					result << ','
				end
			end
		end
		result << " -->"
	end

	def __parse_plugin_content(content)
		M2W_LOGGER.debug("Start  #{self.class}#__parse_plugin_content ... content = #{content}")

		plugin_name = nil
		param = nil
		if /^ *(#{M2W_PLUGIN_NAME_REGEX}) *(.*)$/m =~ content then
			plugin_name = $1
			param_content = $2
			if /^\((.+)\)$/m =~ param_content then
				param = $1
				return [plugin_name, param]
			else
				param = param_content
			end
		else
			raise FormatPluginError.new(
				"プラグイン書式不正(コンテンツ = #{content})",
				"プラグインの書式が不正です（コンテンツ = #{content}）。")
		end

		M2W_LOGGER.debug("Finish #{self.class}#__parse_plugin_content ... plugin_name = #{plugin_name}, param = #{param}")
		return [plugin_name, param]
	end

end
