# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

# mail2weblog のルートパスを設定する
M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

require 'm2w_configurator'
require 'm2w_mail_parser'
require 'mailutils/mail_processor'

require 'minitest/unit'

MiniTest::Unit.autorun

#=Mail2WeblogMailParser のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_m2w_mail_parser.rb 296 2011-09-24 06:32:32Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMail2WeblogMailParser < MiniTest::Unit::TestCase

	def test_parse_subject
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

		# .
		mail = Mail.new("m.abc.def", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal 'def', result[1]

		# :
		mail = Mail.new("m:abc:def", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal 'def', result[1]

		# /
		mail = Mail.new("m/abc/def", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal 'def', result[1]
	end

	def test_parse_subject_without_header
		mail_parser = Mail2WeblogMailParser.new(nil, M2W_SUBJECT_SEPARATOR_REGEX, {})

		# .
		mail = Mail.new("abc.def", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal 'def', result[1]
	end

	def test_parse_subject_without_param
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

		# .
		mail = Mail.new("m.abc", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal '', result[1]
	end

	def test_parse_subject_without_header_and_param
		mail_parser = Mail2WeblogMailParser.new(nil, M2W_SUBJECT_SEPARATOR_REGEX, {})

		# .
		mail = Mail.new("abc", nil, nil, nil, nil)
		result = mail_parser.parse_subject(mail.subject)
		assert_equal 2, result.size
		assert_equal 'abc', result[0]
		assert_equal '', result[1]
	end

	def test_parse_content
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

		# .
		mail = Mail.new(nil, nil, nil, "tit.abc\n\ndef", [])
		(header, content) = mail_parser.parse_content(mail.content, {'tit' => 'title'})
		assert_equal 1, header.size
		assert_equal 'abc', header['title']
		assert_equal 'def', content

		# :
		mail = Mail.new(nil, nil, nil, "tit:abc\n\ndef", [])
		(header, content) = mail_parser.parse_content(mail.content, {'tit' => 'title'})
		assert_equal 1, header.size
		assert_equal 'abc', header['title']
		assert_equal 'def', content

		# /
		mail = Mail.new(nil, nil, nil, "tit/abc\n\ndef", [])
		(header, content) = mail_parser.parse_content(mail.content, {'tit' => 'title'})
		assert_equal 1, header.size
		assert_equal 'abc', header['title']
		assert_equal 'def', content
	end

	def test_parse_content_header_alias_plus
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

		mail = Mail.new(nil, nil, nil, "a.value1\nb.value2\ndef", [])
		(header, content) = mail_parser.parse_content(mail.content, {"b" => "c"})
		assert_equal 2, header.size
		assert_equal 'value1', header['a']
		assert_equal 'value2', header['c']
	end

	def test_parse_content_no_header_value
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

		mail = Mail.new(nil, nil, nil, "a.\nb.\ndef", [])
		(header, content) = mail_parser.parse_content(mail.content, {})
		assert_equal 2, header.size
		assert_equal '', header['a']
		assert_equal '', header['b']
	end

	def test_escaped_prefix_content
		mail_parser = Mail2WeblogMailParser.new(M2W_SUBJECT_HEADER, M2W_SUBJECT_SEPARATOR_REGEX, {})

content = <<"END_CONTENT"
メーラーによって邪魔な引用ヘッダーが入っている場合
また、引用記号(>)が入っている場合

> start.ここから下に記入
> ここが本文
END_CONTENT

expected_content = <<"END_CONTENT"
start.ここから下に記入
ここが本文
END_CONTENT
		assert_equal expected_content.chomp, mail_parser.escaped_prefix_content(content, 'start\..*')
	end

end
