# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/")
$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

require 'base64'
require 'net/pop'
require 'net/smtp'

require 'conf/01_system/ruby_version_absorber'
require 'conf/setting'
require 'mailutils/mail_processor'
require 'testutils/mail_testutils'

require 'minitest/unit'

MiniTest::Unit.autorun

#=MailProcessor のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test.rb 192 2011-08-11 01:49:43Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMailProcessor < MiniTest::Unit::TestCase

	def test_escaped_prefix_subject
		mail = Mail.new("Re:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fw:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fwd:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re>title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re > title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject
	end

	def test_encode_header1
		assert_equal '=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', MailProcessor.encode_header('テスト')
	end

	def test_decode_header2
		assert_equal 'テスト', MailProcessor.decode_header('=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', 'UTF-8')
	end

	def test_decode_header3
		assert_equal 'm2w:post:1', MailProcessor.decode_header('=?iso-8859-1?q?m2w:post:1?=', 'UTF-8')
	end

	def test_parse_address_content1
		address = MailProcessor.parse_mail_address('name <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content2
		address = MailProcessor.parse_mail_address('"name" <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content3
		address = MailProcessor.parse_mail_address("'name' <abcdef@ghjik.lmnop>")
		assert_equal("'name'", address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_conten4
		address = MailProcessor.parse_mail_address("<w-w.w-w.0310@docomo.ne.jp>")
		assert_equal("", address.name)
		assert_equal('w-w.w-w.0310@docomo.ne.jp', address.address)
	end

	def test_make1
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << "From: test_sender@ownway.info\n"
		expected << "To: test_receiver1@ownway.info\n"
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_make2
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info', 'テスト送信者'),
			[MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << %Q!From: "=?ISO-2022-JP?B?GyRCJUYlOSVIQXc/LjxUGyhC?=" <test_sender@ownway.info>\n!
		expected << %Q!To: "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzEbKEI=?=" <test_receiver1@ownway.info>, "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzIbKEI=?=" <test_receiver2@ownway.info>\n!
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_parse
		from_address = MailAddress.new('test_sender@ownway.info', 'テスト送信者')
		to_addresses = [MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')]
		attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/information.png")]
		result = MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6))

		mail = MailProcessor.parse(result, "UTF-8")
		assert_equal 'テスト', mail.subject
		assert_equal from_address.address, mail.from_address.address
		assert_equal from_address.name, mail.from_address.name
		assert_equal to_addresses.size, mail.to_addresses.size
		assert_equal to_addresses[0].address, mail.to_addresses[0].address
		assert_equal to_addresses[0].name, mail.to_addresses[0].name
		assert_equal to_addresses[1].address, mail.to_addresses[1].address
		assert_equal to_addresses[1].name, mail.to_addresses[1].name
		assert_equal 'テスト', mail.content
		assert_equal attachments.size, mail.attachments.size
		assert_equal attachments[0].filename, mail.attachments[0].filename
		assert_equal attachments[0].content, mail.attachments[0].content
	end

	if REAL_MAIL_TEST then
		def setup
		end

		# リアルメール送受信テストです。
		# 本テストは、設定（conf/setting.rb）に依存します。
		# また、本テストは、SMTP サーバへの認証に SMTP Authentication を想定しています。
		def test_real
			# メールを送信する
			#from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS)
			#to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS), MailAddress.new(POP3_SERVER2_MAILADDRESS)]
			from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS, 'テスト送信者')
			to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS, 'テスト受信者１'), MailAddress.new(POP3_SERVER2_MAILADDRESS, 'テスト受信者２')]
			attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/information.png")]
			Net::SMTP.start(SMTP_SERVER_ADDRESS, SMTP_SERVER_PORT, SMTP_SERVER_ADDRESS, SMTP_SERVER_USER, SMTP_SERVER_PASSWORD, :login) do |smtp|
				smtp.send_message(
					MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6)),
					from_address.address, MailProcessor.make_address_list(to_addresses)
				)
			end

			# メールを受信する（サーバ１）
			mail_testutils1 = MailTestUtils.new(
				POP3_SERVER1_ADDRESS,
				POP3_SERVER1_PORT,
				POP3_SERVER1_USER,
				POP3_SERVER1_PASSWORD,
				'UTF-8')
			mail = mail_testutils1.get_mail()

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content

			# メールを受信する（サーバ２）
			mail_testutils2 = MailTestUtils.new(
				POP3_SERVER2_ADDRESS,
				POP3_SERVER2_PORT,
				POP3_SERVER2_USER,
				POP3_SERVER2_PASSWORD,
				'UTF-8')
			mail = mail_testutils2.get_mail()

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content
		end
	end

end
