# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/")
$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

require 'base64'
require 'net/pop'
require 'net/smtp'

require 'conf/01_system/ruby_version_absorber'
require 'conf/setting'
require 'mailutils/mail_processor'

require 'minitest/unit'

MiniTest::Unit.autorun

#=MailProcessor のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test.rb 99 2011-06-27 10:52:03Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMailProcessor < MiniTest::Unit::TestCase

	def test_encode_header
		assert_equal '=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', MailProcessor.encode_header('テスト')
	end

	def test_decode_header
		assert_equal 'テスト', MailProcessor.decode_header('=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', 'UTF-8')
	end

	def test_decode_header
		assert_equal 'm2w:post:1', MailProcessor.decode_header('=?iso-8859-1?q?m2w:post:1?=', 'UTF-8')
	end

	def test_make1
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << "From: test_sender@ownway.info\n"
		expected << "To: test_receiver1@ownway.info\n"
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_make2
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info', 'テスト送信者'),
			[MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << %Q!From: "=?ISO-2022-JP?B?GyRCJUYlOSVIQXc/LjxUGyhC?=" <test_sender@ownway.info>\n!
		expected << %Q!To: "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzEbKEI=?=" <test_receiver1@ownway.info>, "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzIbKEI=?=" <test_receiver2@ownway.info>\n!
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_parse
		from_address = MailAddress.new('test_sender@ownway.info', 'テスト送信者')
		to_addresses = [MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')]
		attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/information.png")]
		result = MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6))

		mail = MailProcessor.parse(result, "UTF-8")
		assert_equal 'テスト', mail.subject
		assert_equal from_address.address, mail.from_address.address
		assert_equal from_address.name, mail.from_address.name
		assert_equal to_addresses.size, mail.to_addresses.size
		assert_equal to_addresses[0].address, mail.to_addresses[0].address
		assert_equal to_addresses[0].name, mail.to_addresses[0].name
		assert_equal to_addresses[1].address, mail.to_addresses[1].address
		assert_equal to_addresses[1].name, mail.to_addresses[1].name
		assert_equal 'テスト', mail.content
		assert_equal attachments.size, mail.attachments.size
		assert_equal attachments[0].filename, mail.attachments[0].filename
		assert_equal attachments[0].content, mail.attachments[0].content
	end

	if REAL_MAIL_TEST then
		# リアルメール送受信テストです。
		# 本テストは、設定（conf/setting.rb）に依存します。
		# また、本テストは、SMTP サーバへの認証に SMTP Authentication を想定しています。
		def test_real
			# メールを送信する
			#from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS)
			#to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS), MailAddress.new(POP3_SERVER2_MAILADDRESS)]
			from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS, 'テスト送信者')
			to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS, 'テスト受信者１'), MailAddress.new(POP3_SERVER2_MAILADDRESS, 'テスト受信者２')]
			attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/information.png")]
			Net::SMTP.start(SMTP_SERVER_ADDRESS, SMTP_SERVER_PORT, SMTP_SERVER_ADDRESS, SMTP_SERVER_USER, SMTP_SERVER_PASSWORD, :login) do |smtp|
				smtp.send_message(
					MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6)),
					from_address.address, MailProcessor.make_address_list(to_addresses)
				)
			end

			# メールの受信を待つ
			mail = nil
			first_time = Time.now.to_i
			while mail == nil
				pop = Net::POP3.new(POP3_SERVER1_ADDRESS, POP3_SERVER1_PORT)
				pop.open_timeout = 10
				pop.start(POP3_SERVER1_USER, POP3_SERVER1_PASSWORD)
				pop.each_mail do |m|
					mail = MailProcessor.parse(m.pop, 'UTF-8')
					m.delete
				end

				# 10 秒経過しても受信できなかった場合、テストは失敗とみなす
				if Time.now.to_i - first_time >= 10 then
					assert(false)
				end
			end

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content

			# メールの受信を待つ
			mail = nil
			first_time = Time.now.to_i
			while mail == nil
				pop = Net::POP3.new(POP3_SERVER2_ADDRESS, POP3_SERVER2_PORT)
				pop.open_timeout = 10
				pop.start(POP3_SERVER2_USER, POP3_SERVER2_PASSWORD)
				pop.each_mail do |m|
					mail = MailProcessor.parse(m.pop, 'UTF-8')
					m.delete
				end

				# 10 秒経過しても受信できなかった場合、テストは失敗とみなす
				if Time.now.to_i - first_time >= 10 then
					assert(false)
				end
			end

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content
		end
	end

end
