# coding: UTF-8

require 'm2w_common'

require 'style/formatter'

require 'xmlrpc/client'

#=メールを本文を解析してブログへ投稿するブログ投稿クラス =
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: weblog_post.rb 41 2011-06-02 11:38:37Z yutaka_at_home $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
#=想定するメールの例
#
#==タイトル例（メールヘッダーの Subject）
#
# m2w:post:1
#
# ----
#
# m2w は、mail2weblog が処理対象とするメールであることを認識するためのヘッダーです。
# m2w_config.rb で定義している M2W_SUBJECT_HEADER に従っています。
#
# post は、ブログへの投稿のコマンドであることを表しています。
# m2w_config.rb で定義している M2W_POST_COMMAND に従っています。
#
# 最後の 1 は、投稿対象となるブログのブログＩＤです。
#
#==本文例（メール本文）
#
# tit:今日の天気
# dat:20110309T07:30:00
# 
# 今日の天気は晴れです。
#
# ----
#
# メール本文にタイトルや記載日時などの情報とブログ記事の本文の両方を含めます。
# タイトル（tit）や記載日時（dat）の情報をヘッダー、空行で区切られたそれ以降の情報を本文として判断します。
#
#===ヘッダー
#
# ヘッダーは、以下のフォーマットに従ったキーと値の対を複数指定できます。
#
# キー名:値
#
# キー名には、MetaWeblog.newPost の content 構造体の名称を指定できる他 m2w_config.rb の M2W_HEADER_MAP で定義している略称も使用可能です。
# 初期設定時、tit は title を、dat は dateCreated を表しています。
#
#===ブログ記事の本文
#
# ブログ記事の本文は Hiki 構文（http://hikiwiki.org/ja/TextFormattingRules.html）に従って記載してください。
# 変換された XHTML がブログ記事として投稿されます。
# 内部で HikiDoc ライブラリを使用しています。
# HikiDoc の動作オプションは、m2w_config.rb の M2W_HIKI_OPTIONS に従います。
# 詳細は、HikiDoc ライブラリを参照してください。
class WeblogPost

	attr_accessor :host
	attr_accessor :address
	attr_accessor :username
	attr_accessor :password
	attr_accessor :blogid
	attr_accessor :attributes

	def initialize(host, address, username, password, blogid, buffer)
		@host = host
		@address = address
		@username = username
		@password = password
		@blogid = blogid
		@attributes = parse(buffer)
	end

	def parse(mail)
		buffers = mail.body(M2W_XMLRPC_ENCODING).split(/\r\n|\r|\n/)

		attributes = {}

		# ヘッダーを読み込む
		i = 0
		while i < buffers.size && buffers[i] != ''
			line = buffers[i]

			if line =~ /^([a-zA-Z_]+):(.+)$/ then
				key = $1
				value = $2
				if M2W_POST_COMMAND_HEADER_ALIAS.has_key?(key) then
					attributes[M2W_POST_COMMAND_HEADER_ALIAS[key]] = value
				else
					attributes[key] = value
				end
			end

			i = i + 1
		end

		# 本文を読み込む
		i = i + 1
		body = ''
		while i < buffers.size
			body = body + buffers[i] + "\n"

			i = i + 1
		end

		@formatter = Formatter.new
		attributes['description'] = @formatter.preformat(body)

		@attachments = []
		mail.attachments.each do |mailpart|
			@attachments.push([mailpart.filename, mailpart.decode])
		end

		return attributes
	end

	def get_attachment_filename(postid, filename)
		return "#{M2W_WEBLOG_ATTACHMENT_PATH}#{postid}/#{filename}"
	end

	def get_attachments_vars(postid, attachments)
		result = []
		attachments.each do |attachment|
			result.push(get_attachment_filename(postid, attachment[0]))
		end
		return result
	end

	def run(extension)
		server = XMLRPC::Client.new(@host, @address)

		if extension != nil then
			extension.postBefore(self, server)
		end

		# ブログ本文を投稿する
		M2W_LOGGER.info("Start  metaWeblog.newPost ... #{@attributes} to http://#{host}/#{address}, blogid = #{@blogid}")
		postid = server.call("metaWeblog.newPost", @blogid, @username, @password, @attributes)
		M2W_LOGGER.info("Finish metaWeblog.newPost ... postid = #{postid}")

		# 添付ファイルを投稿する
		@attachments.each do |attachment|
			M2W_LOGGER.info("Start  metaWeblog.newMediaObject ... #{attachment[0]} to http://#{host}/#{address}, blogid = #{@blogid}, postid = #{@postid}")
			server.call("metaWeblog.newMediaObject", @blogid, @username, @password, {"bits" => XMLRPC::Base64.new(attachment[1]), "name" => get_attachment_filename(postid, attachment[0])})
			M2W_LOGGER.info("Finish metaWeblog.newMediaObject ... #{attachment[0]} to http://#{host}/#{address}, blogid = #{@blogid}, postid = #{@postid}")
		end

		# ブログ本文のプラグインを有効化して再編集する
		# ※ 再編集する理由は、一度投稿しないと postid をプラグインに伝えられないからである
		if @formatter.has_plugin_contents then
			vars = {:blogid => @blogid, :postid => postid}
			vars = vars.merge(@attributes)
			vars[:attachments] = get_attachments_vars(postid, @attachments)
			M2W_LOGGER.info("Start  metaWeblog.editPost ... #{vars}")
			@attributes['description'] = @formatter.postformat(@attributes['description'], vars)
			edit_successed = server.call("metaWeblog.editPost", postid, @username, @password, @attributes)
			M2W_LOGGER.info("Finish metaWeblog.editPost ... result = #{edit_successed}")
		end

		if extension != nil then
			begin
				extension.postAfter(self, server, postid)
			rescue => e
				M2W_LOGGER.error(e)
			end
		end

		return true
	end

end
