# coding: UTF-8

require 'hikidoc/hikidoc'
require 'style/formatter4html'
require 'style/formatter4htmlplain'

require 'cgi'

require 'plugin/format/lib/format_plugin_error'
require 'plugin/format/lib/format_plugin_warning'

M2W_FORMATTER_MARK_VERSION = "m2w_version"
M2W_FORMATTER_MARK_WIKICONTENT = "wikicontent"
M2W_FORMATTER_MARK_HTMLCONTENT = "htmlcontent"
M2W_FORMATTER_MARK_HTMLPLAINCONTENT = "htmlplain"
M2W_FORMATTER_MARK_ATTACHMENT_URLS = "attachment_urls"
M2W_FORMATTER_MARK_RELATED_ATTACHMENT_URLS = "related_urls"
M2W_FORMATTER_COMMENTOUT_REGEX = /^(<!--#{M2W_FORMATTER_MARK_VERSION}:[0-9]{2}\.[0-9]{2}\.[0-9]{2}-->)?(<!--\[#{M2W_FORMATTER_MARK_WIKICONTENT}\] .+? -->)?(<!--\[#{M2W_FORMATTER_MARK_HTMLCONTENT}\] .+? -->)?(<!--\[#{M2W_FORMATTER_MARK_HTMLPLAINCONTENT}\] .+? -->)?(<!--\[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}\] .+? -->)?(<!--\[#{M2W_FORMATTER_MARK_RELATED_ATTACHMENT_URLS}\] .+? -->)?(<!-- .+? -->)?/m
M2W_PLUGIN_NAME_REGEX = "[a-zA-Z_][0-9a-zA-Z_]*"

#=本文のフォーマット変換を行います。
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: formatter.rb 857 2012-10-21 12:18:48Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class Formatter

	attr_reader :has_plugin_content
	attr_writer :logger

	def initialize(plugin_manager = nil, logger = nil)
		@plugin_manager = plugin_manager != nil ? plugin_manager : Mail2WeblogPluginManager.new({}, {}, {}, logger)
		@logger = logger
	end

	def preset(request, vars, errors, warnings)
		@logger.debug("Start  #{self.class}#preset ... vars = #{vars}") if @logger && @logger.debug?

		formatted_content = __format(request.format, request.content, vars)
		__calls_plugin_preset(formatted_content, vars, errors, warnings)

		@logger.debug("Finish #{self.class}#preset ... vars = #{vars}") if @logger && @logger.debug?
	end

	def format(request, vars, errors, warnings)
		@logger.debug("Start  #{self.class}#format ... vars = #{vars}") if @logger && @logger.debug?

		# 全てのコンテンツに書式プラグインを適用する
		result = {'description' => ''}
		request.contents.each do |name, each_content|
			formatted_content = __format(request.format, each_content, vars)
			result[name] = __calls_plugin_content(formatted_content, vars, errors, warnings)
		end

		description = __get_commentout_content(request)
		description << __get_commentout_attachment_urls_content(vars != nil ? vars[:attachment_urls] : {})
		description << __get_commentout_related_attachment_urls_content(vars != nil ? vars[:related_attachment_urls] : {})
		description << result['description']
		result['description'] = description

		@logger.debug("Finish #{self.class}#format ... result = #{result.keys}") if @logger && @logger.debug?

		return result
	end

	def escape(content)
		return CGI.escapeHTML(content)
	end

	def unescape(content)
		return CGI.unescapeHTML(content)
	end

	def get_uncommentout_content(content)
		@logger.debug("Start  #{self.class.name}#get_uncommentout_content") if @logger && @logger.debug?

		result_version = nil
		result_content = nil
		result_type = nil
		if M2W_FORMATTER_COMMENTOUT_REGEX =~ content then
			version = $1
			wikicontent = $2
			htmlcontent = $3
			htmlplaincontent = $4
			old_wikicontent = $7
			@logger.debug("match ... version = #{version}") if @logger && @logger.debug?

			if version != nil && (wikicontent != nil || htmlcontent != nil || htmlplaincontent != nil) then
				if /^<!--#{M2W_FORMATTER_MARK_VERSION}:([0-9]{2}\.[0-9]{2}\.[0-9]{2})-->$/ =~ version then
					result_version = $1
				end

				if wikicontent != nil && /^<!--\[#{M2W_FORMATTER_MARK_WIKICONTENT}\] (.+?) -->$/m =~ wikicontent then
					result_content = unescape($1)
					result = true
					result_type = 'wiki'
				elsif htmlcontent != nil && /^<!--\[#{M2W_FORMATTER_MARK_HTMLCONTENT}\] (.+?) -->$/m =~ htmlcontent then
					result_content = unescape($1)
					result = true
					result_type = 'html'
				elsif htmlplaincontent != nil && /^<!--\[#{M2W_FORMATTER_MARK_HTMLPLAINCONTENT}\] (.+?) -->$/m =~ htmlplaincontent then
					result_content = unescape($1)
					result = true
					result_type = 'htmlplain'
				else
					@logger.debug("no match content") if @logger && @logger.debug?
				end
			elsif old_wikicontent != nil then
				@logger.debug('old_wikicontent != ""') if @logger && @logger.debug?
				if /^<!-- *(.+?) *-->$/m =~ old_wikicontent then
					result_content = unescape($1)
					@logger.debug("match old_wikicontent ... #{$1}") if @logger && @logger.debug?
					result = true
					result_type = 'wiki'
				else
					@logger.debug('no match old_wikicontent') if @logger && @logger.debug?
				end
			else
				@logger.debug("no match all_content") if @logger && @logger.debug?
			end
		else
			@logger.debug("no match") if @logger && @logger.debug?
		end

		@logger.debug("Finish #{self.class.name}#get_uncommentout_content ... version = #{result_version}") if @logger && @logger.debug?

		return [result_version, result_content, result_type]
	end

	def get_uncommentout_attachment_urls_content(content)
		@logger.debug("Start  #{self.class.name}#get_uncommentout_attachment_urls_content") if @logger && @logger.debug?

		result = {}
		if M2W_FORMATTER_COMMENTOUT_REGEX =~ content then
			attachment_urls = $5
			@logger.debug("match ... attachment_urls = #{attachment_urls}") if @logger && @logger.debug?

			if /^<!--\[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}\] (.+?) -->$/ =~ attachment_urls then
				urls_map = $1.split(',')
				urls_map.each do |url_map|
					if url_map =~ /^([0-9]+)=(.*)$/ then
						index = $1
						url = $2
						if url.length > 0 then
							result[index] = url
						else
							result[index] = nil
						end
					end
				end
			end
		else
			@logger.debug("no match") if @logger && @logger.debug?
		end

		@logger.debug("Finish #{self.class.name}#get_uncommentout_attachment_urls_content ... result = #{result.to_s}") if @logger && @logger.debug?
		return result
	end

	def get_uncommentout_related_attachment_urls_content(content)
		@logger.debug("Start  #{self.class.name}#get_uncommentout_related_attachment_urls_content") if @logger && @logger.debug?

		result = {}
		if M2W_FORMATTER_COMMENTOUT_REGEX =~ content then
			related_attachment_urls = $6
			@logger.debug("match ... related_attachment_urls = #{related_attachment_urls}") if @logger && @logger.debug?

			if /^<!--\[#{M2W_FORMATTER_MARK_RELATED_ATTACHMENT_URLS}\] (.+?) -->$/ =~ related_attachment_urls then
				urls_map = $1.split(',')
				urls_map.each do |url_map|
					if url_map =~ /^(.+?)=(.*)$/ then
						content_id = $1
						url = $2
						if url.length > 0 then
							result[content_id] = url
						else
							result[content_id] = nil
						end
					end
				end
			end
		else
			@logger.debug("no match") if @logger && @logger.debug?
		end

		@logger.debug("Finish #{self.class.name}#get_uncommentout_attachment_urls_content ... result = #{result.to_s}") if @logger && @logger.debug?
		return result
	end

	def __format(format, content, vars)
		case format
		when 'wiki'
			return HikiDoc.to_xhtml(content, M2W_HIKI_OPTIONS).encode(M2W_SYSTEM_INNER_ENCODING, M2W_SYSTEM_HIKIDOC_ENCODING)
		when 'html'
			return Formatter4HTML.to_xhtml(content, @plugin_manager, vars)
		when 'htmlplain'
			return Formatter4HTMLPlain.to_xhtml(content, vars)
		else
			raise FormatPluginError.new(
				"不明な書式モードエラー(フォーマットモード = #{format})",
				"不明な書式モードを指定しました（フォーマットモード = #{format}）。")
		end
	end

	def __calls_plugin_preset(content, vars, errors, warnings)
		@logger.debug("Start  #{self.class}#__calls_plugin_preset ... vars = #{vars}") if @logger && @logger.debug?

		# プラグインを事前処理する
		while %r!(.*?)<(div|span) class="plugin">\{\{(.+?)\}\}</\2>(.*)$!m =~ content
			plugin_type = $2
			plugin_content = $3
			content = $4

			begin
				case plugin_type
				when "div"
					vars.deep_merge!(__call_plugin_preset(plugin_content, vars))
				when "span"
					vars.deep_merge!(__call_plugin_preset_inline(plugin_content, vars))
				end
			rescue FormatPluginWarning => warning
				warnings.push(warning)
			rescue FormatPluginError => error
				errors.push(error)
			end
  	end

		@logger.debug("Finish #{self.class}#__calls_plugin_preset ... vars = #{vars}") if @logger && @logger.debug?
	end

	def __calls_plugin_content(content, vars, errors, warnings)
		@logger.debug("Start  #{self.class}#__calls_plugin_content ... vars = #{vars}") if @logger && @logger.debug?

		# プラグインを処理する
		result = ""
		while %r!(.*?)<(div|span) class="plugin">\{\{(.+?)\}\}</\2>(.*)$!m =~ content
			left = $1
			plugin_type = $2
			plugin_content = $3
			content = $4

			result << left

			begin
				case plugin_type
				when "div"
					result << __call_plugin_contents(plugin_content, vars)
				when "span"
					result << __call_plugin_contents_inline(plugin_content, vars)
				end
			rescue FormatPluginWarning => warning
				result << "{{警告: #{warning.content}}}"
				warnings.push(warning)
			rescue FormatPluginError => error
				result << "{{エラー: #{error.content}}}"
				errors.push(error)
			end
  	end

		result << content

		@logger.debug("Finish #{self.class}#__calls_plugin_content") if @logger && @logger.debug?

		return result
	end

	def __call_plugin_preset(content, vars)
		@logger.debug("Start  #{self.class}#__call_plugin_preset ... vars = #{vars}") if @logger && @logger.debug?

		(plugin_name, param) = __parse_plugin_content(content)

		result = nil
		begin
			result = @plugin_manager.call_format_plugin_preset(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			@logger.error(e) if @logger
			raise e
		rescue FormatPluginError => e
			@logger.error(e) if @logger
			raise e
		rescue => e
			@logger.error("プラグイン事前準備中に例外が発生しました。") if @logger
			@logger.error(e) if @logger
			raise FormatPluginError.new(
				"プラグイン事前準備中例外(コンテンツ = #{content})",
				"プラグイン事前準備中に例外が発生しました（コンテンツ = #{content}）。")
		ensure
			@logger.debug("Finish #{self.class}#__call_plugin_preset ... result = #{result}") if @logger && @logger.debug?
		end

		return result
	end

	def __call_plugin_preset_inline(content, vars)
		@logger.debug("Start  #{self.class}#__call_plugin_preset_inline ... vars = #{vars}") if @logger && @logger.debug?

		(plugin_name, param) = __parse_plugin_content(content)

		result = nil
		begin
			result = @plugin_manager.call_format_plugin_preset_inline(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			@logger.error(e) if @logger
			raise e
		rescue FormatPluginError => e
			@logger.error(e) if @logger
			raise e
		rescue => e
			@logger.error("プラグイン事前準備中に例外が発生しました。") if @logger
			@logger.error(e) if @logger
			raise FormatPluginError.new(
				"プラグイン事前準備中例外(コンテンツ = #{content})",
				"プラグイン事前準備中に例外が発生しました（コンテンツ = #{content}）。")
		ensure
			@logger.debug("Finish #{self.class}#__call_plugin_preset_inline ... result = #{result}") if @logger && @logger.debug?
		end

		return result
	end

	def __call_plugin_contents(content, vars)
		@logger.debug("Start  #{self.class}#__call_plugin_contents ... vars = #{vars}") if @logger && @logger.debug?

		(plugin_name, param) = __parse_plugin_content(content)

		result = ""
		begin
			result = @plugin_manager.call_format_plugin_contents(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			@logger.error(e) if @logger
			raise e
		rescue FormatPluginError => e
			@logger.error(e) if @logger
			raise e
		rescue => e
			@logger.error("プラグイン動作時に例外が発生しました。") if @logger
			@logger.error(e) if @logger
			raise FormatPluginError.new(
				"プラグイン動作時例外(コンテンツ = #{content})",
				"プラグイン動作時例外が発生しました（コンテンツ = #{content}）。")
		ensure
			@logger.debug("Finish #{self.class}#__call_plugin_contents ... result = #{result}") if @logger && @logger.debug?
		end

		return result
	end

	def __call_plugin_contents_inline(content, vars)
		@logger.debug("Start  #{self.class}#call_plugin_contents_inline ... vars = #{vars}") if @logger && @logger.debug?

		(plugin_name, param) = __parse_plugin_content(content)

		result = ""
		begin
			result = @plugin_manager.call_format_plugin_contents_inline(plugin_name, param, vars)
		rescue FormatPluginWarning => e
			@logger.error(e) if @logger
			raise e
		rescue FormatPluginError => e
			@logger.error(e) if @logger
			raise e
		rescue => e
			@logger.error("プラグイン動作時に例外が発生しました。") if @logger
			@logger.error(e) if @logger
			raise FormatPluginError.new(
				"プラグイン動作時例外(コンテンツ = #{content})",
				"プラグイン動作時例外が発生しました（コンテンツ = #{content}）。")
		ensure
			@logger.debug("Finish #{self.class}#__call_plugin_contents_inline ... result = #{result}") if @logger && @logger.debug?
		end

		return result
	end

	def __get_commentout_content(request)
		case request.format
		when 'wiki'
			return "<!--#{M2W_FORMATTER_MARK_VERSION}:#{M2W_VERSION}--><!--[#{M2W_FORMATTER_MARK_WIKICONTENT}] #{escape(request.original_content)} -->"
		when 'html'
			return "<!--#{M2W_FORMATTER_MARK_VERSION}:#{M2W_VERSION}--><!--[#{M2W_FORMATTER_MARK_HTMLCONTENT}] #{escape(request.original_content)} -->"
		when 'htmlplain'
			return "<!--#{M2W_FORMATTER_MARK_VERSION}:#{M2W_VERSION}--><!--[#{M2W_FORMATTER_MARK_HTMLPLAINCONTENT}] #{escape(request.original_content)} -->"
		else
			raise FormatPluginError.new(
				"不明な書式モードエラー(フォーマットモード = #{request.format})",
				"不明な書式モードを指定しました（フォーマットモード = #{request.format}）。")
		end
	end

	def __get_commentout_attachment_urls_content(attachment_urls)
		if attachment_urls != nil && attachment_urls.size > 0 then
			result = "<!--[#{M2W_FORMATTER_MARK_ATTACHMENT_URLS}] "
			attachment_urls.keys.sort.each do |i|
				if i > 1 then
					result << ','
				end
				attachment_url = attachment_urls[i]
				result << "#{i}=#{attachment_url}"
			end
			result << " -->"

			return result
		else
			return ''
		end
	end

	def __get_commentout_related_attachment_urls_content(related_attachment_urls)
		if related_attachment_urls != nil && related_attachment_urls.size > 0 then
			result = "<!--[#{M2W_FORMATTER_MARK_RELATED_ATTACHMENT_URLS}] "
			related_attachment_urls.keys.sort.each_with_index do |content_id, i|
				if i > 0 then
					result << ','
				end
				attachment_url = related_attachment_urls[content_id]
				result << "#{content_id}=#{attachment_url}"
			end
			result << " -->"

			return result
		else
			return ''
		end
	end

	def __parse_plugin_content(content)
		@logger.debug("Start  #{self.class}#__parse_plugin_content") if @logger && @logger.debug?

		plugin_name = nil
		param = nil
		if /^ *(#{M2W_PLUGIN_NAME_REGEX}) *(.*)$/m =~ content then
			plugin_name = $1
			param_content = $2
			if /^\((.+)\)$/m =~ param_content then
				param = $1
			else
				param = param_content
			end
		else
			raise FormatPluginError.new(
				"プラグイン書式不正(コンテンツ = #{content})",
				"プラグインの書式が不正です（コンテンツ = #{content}）。")
		end

		@logger.debug("Finish #{self.class}#__parse_plugin_content ... plugin_name = #{plugin_name}, param = #{param}") if @logger && @logger.debug?
		return [plugin_name, param]
	end

end
