# coding: UTF-8

if $0 == __FILE__ then
	ROOT_PATH = File.dirname(File.expand_path(__FILE__))

	$:.unshift(ROOT_PATH + "/../")
	$:.unshift(ROOT_PATH + "/../../main/")
	$:.unshift(ROOT_PATH + "/../../main/lib/")

	# mail2weblog のルートパスを設定する
	M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

	require 'm2w_configurator'
	require 'minitest/autorun'
end

require 'm2w_plugin_manager'
require 'style/formatter'
require 'plugin/format/lib/format_plugin_utils'

class Example_FormatPlugin
	include FormatPluginUtils

	def plugin_contents(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning", "warning")
			when "error"
				raise FormatPluginError.new("error", "error")
			end
		else
			if content == nil then
				return "example"
			else
				return "example[#{content}]"
			end
		end
	end

	def plugin_contents_inline(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning_inline", "warning_inline")
			when "error"
				raise FormatPluginError.new("error_inline", "error_inline")
			end
		else
			if content == nil then
				return "example_inline"
			else
				return "example_inline[#{content}]"
			end
		end
	end

end

#=Formatter のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_formatter_wiki.rb 675 2012-05-12 15:12:21Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestFormatter4Wiki < MiniTest::Unit::TestCase

	def setup
		@formatter = Formatter.new(Mail2WeblogPluginManager.new({}, {}, {'COMMON' => {'default_format_mode' => 'wiki'}}))
	end

	def test_format
original_content = <<"END_CONTENT"
tit.タイトル <&>

１行目

２行目
END_CONTENT

content = <<"END_CONTENT"
１行目

２行目
END_CONTENT

		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] tit.タイトル &lt;&amp;&gt;

１行目

２行目
 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin
		original_content = "{{example}}"
		content = "{{example}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] {{example}} -->example
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_multiline_with_plugin
original_content = <<"END_CONTENT"
{{example
line1
line2}}
END_CONTENT

content = <<"END_CONTENT"
{{example
line1
line2}}
END_CONTENT

		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] {{example
line1
line2}}
 -->example[line1
line2]
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_warning
		original_content = "{{example(type@warning)}}"
		content = "{{example(type@warning)}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] {{example(type@warning)}} -->{{警告: warning}}
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 1, warnings.size
	end

	def test_format_with_plugin_error
		original_content = "{{example(type@error)}}"
		content = "{{example(type@error)}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] {{example(type@error)}} -->{{エラー: error}}
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 1, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_inline
		original_content = "テスト{{example}}"
		content = "テスト{{example}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example}} --><p>テストexample_inline</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_multiline_with_plugin_inline
original_content = <<"END_CONTENT"
テスト{{example
line1
line2}}
END_CONTENT

content = <<"END_CONTENT"
テスト{{example
line1
line2}}
END_CONTENT

		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example
line1
line2}}
 --><p>テストexample_inline[line1
line2]</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_inline_warning
		original_content = "テスト{{example(type@warning)}}"
		content = "テスト{{example(type@warning)}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example(type@warning)}} --><p>テスト{{警告: warning_inline}}</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 1, warnings.size
	end

	def test_format_with_plugin_inline_error
		original_content = "テスト{{example(type@error)}}"
		content = "テスト{{example(type@error)}}"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト{{example(type@error)}} --><p>テスト{{エラー: error_inline}}</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 1, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_attachment
		original_content = "テスト"
		content = "テスト"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト --><!--[attachment_urls] 1=http://abc.def.ghi/jkl.txt --><p>テスト</p>
END_CONTENT

		vars = {
			:attachment_urls => {1 => 'http://abc.def.ghi/jkl.txt'}
		}
		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), vars, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_attachments
		original_content = "テスト"
		content = "テスト"
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] テスト --><!--[attachment_urls] 1=http://abc.def.ghi/jkl.txt,2=http://mno.pqr.stu/vwx.txt --><p>テスト</p>
END_CONTENT

		vars = {
			:attachment_urls => {
				1 => 'http://abc.def.ghi/jkl.txt',
				2 => 'http://mno.pqr.stu/vwx.txt',
			}
		}
		errors = []
		warnings = []

		result = @formatter.format(Mail2WeblogRequest.new(Mail.new(nil, nil, nil, original_content), {}, nil, contents), vars, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_get_uncommentout_content
content = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[wikicontent] tit.タイトル &lt;&amp;&gt;

１行目

２行目 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

expected_content = <<"END_CONTENT"
tit.タイトル <&>

１行目

２行目
END_CONTENT

		(version, content) = @formatter.get_uncommentout_content(content)
		assert_equal M2W_VERSION, version
		assert_equal expected_content.chomp, content
	end

	def test_get_uncommentout_content_old
content = <<"END_CONTENT"
<!-- １行目

２行目 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

expected_content = <<"END_CONTENT"
１行目

２行目
END_CONTENT

		(version, content) = @formatter.get_uncommentout_content(content)
		assert version == nil
		assert_equal expected_content.chomp, content
	end

	def test_get_uncommentout_attachment_urls_content
		content = "<!--[attachment_urls] 1=abc.txt,2=,3=ghi.txt -->"
		urls = @formatter.get_uncommentout_attachment_urls_content(content)

		assert_equal 3, urls.length
		assert_equal 'abc.txt', urls['1']
		assert_equal nil, urls['2']
		assert_equal 'ghi.txt', urls['3']
	end

end
