/*
 * Copyright (C) 2004-2013 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package quests.Q00628_HuntGoldenRam;

import java.util.HashMap;
import java.util.Map;

import jp.sf.l2j.arrayMaps.SortedIntObjectArrayMap;

import com.l2jserver.gameserver.datatables.SkillTable;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.holders.ItemChanceHolder;
import com.l2jserver.gameserver.model.quest.Quest;
import com.l2jserver.gameserver.model.quest.QuestState;
import com.l2jserver.gameserver.model.quest.State;
import com.l2jserver.gameserver.model.skills.L2Skill;
import com.l2jserver.gameserver.util.Util;

/**
 * Hunt Golden Ram (628)
 * @author netvirus
 */
public final class Q00628_HuntGoldenRam extends Quest
{
	// NPCs
	private static final int PIERCE = 31553;	//+[JOJO]
	private static final int KAHMAN = 31554;
	private static final int ABERCROMBIE = 31555;
	private static final int SELINA = 31556;
	// Items
	private static final int GOLDEN_RAM_COIN = 7251;
	private static final int BADGE_RECRUIT = 7246;
	private static final int BADGE_SOLDIER = 7247;
	private static final int SPLINTER_STAKATO_CHITIN = 7248;
	private static final int NEEDLE_STAKATO_CHITIN = 7249;
	// Misc
	private static final int REQUIRED_ITEM_COUNT = 100;
	private static final int MIN_LVL = 66;
	// Mobs
	private static final SortedIntObjectArrayMap<ItemChanceHolder> MOBS_DROP_CHANCES = new SortedIntObjectArrayMap<ItemChanceHolder>()
		.append(21508, new ItemChanceHolder(SPLINTER_STAKATO_CHITIN, 0.5, 1)) // splinter_stakato
		.append(21509, new ItemChanceHolder(SPLINTER_STAKATO_CHITIN, 0.43, 1)) // splinter_stakato_worker
		.append(21510, new ItemChanceHolder(SPLINTER_STAKATO_CHITIN, 0.521, 1)) // splinter_stakato_soldier
		.append(21511, new ItemChanceHolder(SPLINTER_STAKATO_CHITIN, 0.575, 1)) // splinter_stakato_drone
		.append(21512, new ItemChanceHolder(SPLINTER_STAKATO_CHITIN, 0.746, 1)) // splinter_stakato_drone_a
		.append(21513, new ItemChanceHolder(NEEDLE_STAKATO_CHITIN, 0.5, 2)) // needle_stakato
		.append(21514, new ItemChanceHolder(NEEDLE_STAKATO_CHITIN, 0.43, 2)) // needle_stakato_worker
		.append(21515, new ItemChanceHolder(NEEDLE_STAKATO_CHITIN, 0.52, 2)) // needle_stakato_soldier
		.append(21516, new ItemChanceHolder(NEEDLE_STAKATO_CHITIN, 0.531, 2)) // needle_stakato_drone
		.append(21517, new ItemChanceHolder(NEEDLE_STAKATO_CHITIN, 0.744, 2)) // needle_stakato_drone_a
	;
	// Buffs
	private static final Map<String, Buffs> BUFFS = new HashMap<>();
	static
	{
		BUFFS.put("Focus", new Buffs(4404, 2, 2)); // (buff_id, buff_level, buff_cost)
		BUFFS.put("Death", new Buffs(4405, 2, 2));
		BUFFS.put("Might", new Buffs(4393, 3, 3));
		BUFFS.put("Acumen", new Buffs(4400, 2, 3));
		BUFFS.put("Berserker", new Buffs(4397, 1, 3));
		BUFFS.put("Vampiric", new Buffs(4399, 2, 3));
		BUFFS.put("Empower", new Buffs(4401, 1, 6));
		BUFFS.put("Haste", new Buffs(4402, 2, 6));
	}
	//[JOJO]-------------------------------------------------
	private static class Buffs {
		private final int _skillHashCode;
		private final int _cost;
		Buffs(int id, int level, int cost) {
			_skillHashCode = SkillTable.getSkillHashCode(id, level);
			_cost = cost;
		}
		L2Skill getSkill() { return SkillTable.getSkill(_skillHashCode); }
		int getCost() { return _cost; }
	}
	//-------------------------------------------------------
	
	private Q00628_HuntGoldenRam(int questId, String name, String descr)
	{
		super(questId, name, descr);
		addStartNpc(KAHMAN);
		addTalkId(KAHMAN, SELINA);
		addFirstTalkId(PIERCE, KAHMAN, ABERCROMBIE, SELINA);	//[JOJO] +PIERCE,KAHMAN
		addKillId(MOBS_DROP_CHANCES.keySet());
		registerQuestItems(SPLINTER_STAKATO_CHITIN, NEEDLE_STAKATO_CHITIN, BADGE_RECRUIT, BADGE_SOLDIER);
	}
	
	@Override
	public String onAdvEvent(String event, L2Npc npc, L2PcInstance player)
	{
		final QuestState st = player.getQuestState(getName());
		String htmltext = null;
		if (st == null)
		{
			return htmltext;
		}
		
		switch (event)
		{
			case "accept":
			{
				if (st.isCreated())
				{
					final boolean itemRecruit = st.hasQuestItems(BADGE_RECRUIT);
					final boolean itemSoldier = st.hasQuestItems(BADGE_SOLDIER);
					st.startQuest();
					if (itemSoldier)
					{
						htmltext = "31554-05.htm";
						st.setCond(3);
					}
					else if (itemRecruit)
					{
						htmltext = "31554-04.htm";
						st.setCond(2);
					}
					else
					{
						htmltext = "31554-03.htm";
					}
				}
				break;
			}
			case "31554-08.html":
			{
				if (st.isStarted())
				{
					st.giveItems(BADGE_RECRUIT, 1);
					st.takeItems(SPLINTER_STAKATO_CHITIN, -1);
					st.setCond(2, true);
					htmltext = event;
				}
				break;
			}
			case "31554-12.html":
			case "31554-13.html":
			{
				if (st.isStarted())
				{
					htmltext = event;
				}
				break;
			}
			case "31554-14.html":
			{
				if (st.isStarted())
				{
					st.exitQuest(true, true);
					htmltext = event;
				}
				break;
			}
			case "Focus":
			case "Death":
			case "Might":
			case "Acumen":
			case "Berserker":
			case "Vampiric":
			case "Empower":
			case "Haste":
			{
				if (st.isCond(3))
				{
					final Buffs buff = BUFFS.get(event);
					if (st.getQuestItemsCount(GOLDEN_RAM_COIN) >= buff.getCost())
					{
						st.takeItems(GOLDEN_RAM_COIN, buff.getCost());
						npc.setTarget(player);
						npc.doCast(buff.getSkill());
						htmltext = "31556-03.htm";
					}
					else
					{
						htmltext = "31556-04.htm";
					}
				}
				break;
			}
		}
		return htmltext;
	}
	
	@Override
	public String onFirstTalk(L2Npc npc, L2PcInstance player)
	{
		String htmltext;
		final QuestState st = player.getQuestState(getName());
		switch (npc.getId())
		{
			case PIERCE:
				htmltext = "data/html/default/31553.htm";
				if (st != null && st.isStarted())
				{
					if (st.hasQuestItems(BADGE_SOLDIER) || st.hasQuestItems(BADGE_RECRUIT))
						htmltext = "31553-90.htm";
				}
				break;
			case KAHMAN:
				htmltext = "data/html/default/31554.htm";
				if (st != null && st.isStarted())
				{
					if (st.hasQuestItems(BADGE_SOLDIER) || st.hasQuestItems(BADGE_RECRUIT))
						htmltext = "31554-90.htm";
				}
				break;
			case ABERCROMBIE:
				htmltext = "31555-00.htm";
				if (st != null && st.isStarted())
				{
					if (st.hasQuestItems(BADGE_SOLDIER))
						htmltext = "31555-02.htm";
					else if (st.hasQuestItems(BADGE_RECRUIT))
						htmltext = "31555-01.htm";
				}
				break;
			case SELINA:
				htmltext = "31556-00.htm";
				if (st != null && st.isStarted())
				{
					if (st.hasQuestItems(BADGE_SOLDIER))
						htmltext = "31556-02.htm";
					else if (st.hasQuestItems(BADGE_RECRUIT))
						htmltext = "31556-01.htm";
				}
				break;
			default:
				throw new RuntimeException("onFirstTalk(" + npc.getNpcId() + npc.getName() + "," + player.getName() + ")");
		}
		return htmltext;
	}
	
	@Override
	public String onKill(L2Npc npc, L2PcInstance killer, boolean isSummon)
	{
		QuestState qs = getRandomPartyMemberState(killer, -1, 1, npc);
		if ((qs != null) && qs.isStarted() && Util.checkIfInRange(1500, npc, killer, false))
		{
			final ItemChanceHolder item = MOBS_DROP_CHANCES.get(npc.getId());
			if ((item.getCount() <= qs.getCond()) && !qs.isCond(3))
			{
				giveItemRandomly(killer, npc, item.getId(), 1, REQUIRED_ITEM_COUNT, item.getChance(), true);
			}
		}
		return super.onKill(npc, killer, isSummon);
	}
	
	@Override
	public String onTalk(L2Npc npc, L2PcInstance player)
	{
		final QuestState st = player.getQuestState(getName());
		String htmltext = getNoQuestMsg(player);
		if (st == null)
		{
			return htmltext;
		}
		
		switch (npc.getId())
		{
			case KAHMAN:
			{
				switch (st.getState())
				{
					case State.CREATED:
					{
						htmltext = (player.getLevel() >= MIN_LVL) ? "31554-01.htm" : "31554-02.htm";
						break;
					}
					case State.STARTED:
					{
						final long itemCountSplinter = st.getQuestItemsCount(SPLINTER_STAKATO_CHITIN);
						final long itemCountNeedle = st.getQuestItemsCount(NEEDLE_STAKATO_CHITIN);
						switch (st.getCond())
						{
							case 1:
							{
								htmltext = (itemCountSplinter >= REQUIRED_ITEM_COUNT) ? "31554-07.html" : "31554-06.html";
								break;
							}
							case 2:
							{
								if (st.hasQuestItems(BADGE_RECRUIT))
								{
									if ((itemCountSplinter >= REQUIRED_ITEM_COUNT) && (itemCountNeedle >= REQUIRED_ITEM_COUNT))
									{
										st.takeItems(BADGE_RECRUIT, -1);
										st.takeItems(SPLINTER_STAKATO_CHITIN, -1);
										st.takeItems(NEEDLE_STAKATO_CHITIN, -1);
										st.giveItems(BADGE_SOLDIER, 1);
										st.setCond(3, true);
										htmltext = "31554-10.html";
									}
									else
									{
										htmltext = "31554-09.html";
									}
								}
								else
								{
									st.setCond(1);
									htmltext = (itemCountSplinter >= REQUIRED_ITEM_COUNT) ? "31554-07.html" : "31554-06.html";
								}
								break;
							}
							case 3:
							{
								if (st.hasQuestItems(BADGE_SOLDIER))
								{
									htmltext = "31554-11.html";
								}
								else
								{
									st.setCond(1);
									htmltext = (itemCountSplinter >= REQUIRED_ITEM_COUNT) ? "31554-07.html" : "31554-06.html";
								}
								break;
							}
						}
						break;
					}
				}
				break;
			}
			case SELINA:
			{
				if (st.isCond(3))
				{
					htmltext = "31556-03.htm";
				}
				break;
			}
		}
		return htmltext;
	}
	
	public static void main(String[] args)
	{
		new Q00628_HuntGoldenRam(628, Q00628_HuntGoldenRam.class.getSimpleName(), "Hunt Golden Ram");
	}
}
