/*
 * Copyright (C) 2004-2013 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.admincommandhandlers;

import java.util.Collection;
import java.util.List;
import java.util.StringTokenizer;

import com.l2jserver.Config;
import com.l2jserver.gameserver.datatables.SkillTreesData;
import com.l2jserver.gameserver.handler.IAdminCommandHandler;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.L2World;
import com.l2jserver.gameserver.model.actor.L2Character;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.effects.L2Effect;
import com.l2jserver.gameserver.model.skills.L2Skill;
import com.l2jserver.gameserver.network.SystemMessageId;
import com.l2jserver.gameserver.network.serverpackets.NpcHtmlMessage;
import com.l2jserver.gameserver.network.serverpackets.SkillCoolTime;
import com.l2jserver.gameserver.util.GMAudit;
import com.l2jserver.util.StringUtil;

public class AdminBuffs implements IAdminCommandHandler
{
	private final static int PAGE_LIMIT = 10;
	
	private static final String[] ADMIN_COMMANDS =
	{
		"admin_getbuffs",
		"admin_stopbuff",
		"admin_stopallbuffs",
		"admin_areacancel",
		"admin_removereuse",
		"admin_switch_gm_buffs"
	};
	
	@Override
	public boolean useAdminCommand(String command, L2PcInstance activeChar)
	{
		
		if (command.startsWith("admin_getbuffs"))
		{
			StringTokenizer st = new StringTokenizer(command, " ");
			command = st.nextToken();
			
			L2Character targetCharacter;
			int page = 1;
			
			L2Object object;	// temp
			if (st.hasMoreTokens())
			{
				String playername = st.nextToken();
				if (playername.matches("\f\\d+"))
				{
					int objectId = Integer.parseInt(playername.substring(1));
					if ((object = L2World.getInstance().findObject(objectId)) instanceof L2Character)
					{
						targetCharacter = (L2Character) object;
					}
					else
					{
						activeChar.sendMessage("The object " + playername + " is not character.");
						return false;
					}
				}
				else
				{
					targetCharacter = L2World.getInstance().getPlayer(playername);
					if (targetCharacter == null)
					{
						activeChar.sendMessage("The player " + playername + " is not online.");
						return false;
					}
				}
			}
			else if ((object = activeChar.getTarget()) instanceof L2Character)
			{
				targetCharacter = (L2Character) object;
			}
			else
			{
				activeChar.sendPacket(SystemMessageId.TARGET_IS_INCORRECT);
				return false;
			}
			if (st.hasMoreTokens())
			{
				page = Integer.parseInt(st.nextToken());
			}
			
			showBuffs(activeChar, targetCharacter, page);
			return true;
		}
		else if (command.startsWith("admin_stopbuff"))
		{
			try
			{
				StringTokenizer st = new StringTokenizer(command, " ");
				
				st.nextToken();
				int objectId = Integer.parseInt(st.nextToken());
				int skillId = Integer.parseInt(st.nextToken());
				
				removeBuff(activeChar, objectId, skillId);
				return true;
			}
			catch (Exception e)
			{
				activeChar.sendMessage("Failed removing effect: " + e.getMessage());
				activeChar.sendMessage("Usage: //stopbuff <objectId> <skillId>");
				return false;
			}
		}
		else if (command.startsWith("admin_stopallbuffs"))
		{
			try
			{
				StringTokenizer st = new StringTokenizer(command, " ");
				st.nextToken();
				int objectId = Integer.parseInt(st.nextToken());
				removeAllBuffs(activeChar, objectId);
				return true;
			}
			catch (Exception e)
			{
				activeChar.sendMessage("Failed removing all effects: " + e.getMessage());
				activeChar.sendMessage("Usage: //stopallbuffs <objectId>");
				return false;
			}
		}
		else if (command.startsWith("admin_areacancel"))
		{
			StringTokenizer st = new StringTokenizer(command, " ");
			st.nextToken();
			String val = st.nextToken();
			try
			{
				int radius = Integer.parseInt(val);
				
				for (L2Character knownChar : activeChar.getKnownList().getKnownCharactersInRadius(radius))
				{
					if (knownChar.isPlayer() && !knownChar.equals(activeChar))
					{
						knownChar.stopAllEffects();
					}
				}
				
				activeChar.sendMessage("All effects canceled within radius " + radius);
				return true;
			}
			catch (NumberFormatException e)
			{
				activeChar.sendMessage("Usage: //areacancel <radius>");
				return false;
			}
		}
		else if (command.startsWith("admin_removereuse"))
		{
			StringTokenizer st = new StringTokenizer(command, " ");
			command = st.nextToken();
			
			L2PcInstance player = null;
			if (st.hasMoreTokens())
			{
				String playername = st.nextToken();
				
				try
				{
					player = L2World.getInstance().getPlayer(playername);
				}
				catch (Exception e)
				{
				}
				
				if (player == null)
				{
					activeChar.sendMessage("The player " + playername + " is not online.");
					return false;
				}
			}
			else if (activeChar.getTarget().isPlayer())
			{
				player = activeChar.getTarget().getActingPlayer();
			}
			else
			{
				activeChar.sendPacket(SystemMessageId.TARGET_IS_INCORRECT);
				return false;
			}
			
			try
			{
				player.getSkillReuseTimeStamps().clear();
				player.getDisabledSkills().clear();
				player.sendPacket(new SkillCoolTime(player));
				activeChar.sendMessage("Skill reuse was removed from " + player.getName() + ".");
				return true;
			}
			catch (NullPointerException e)
			{
				return false;
			}
		}
		else if (command.startsWith("admin_switch_gm_buffs"))
		{
			if (Config.GM_GIVE_SPECIAL_SKILLS != Config.GM_GIVE_SPECIAL_AURA_SKILLS)
			{
				final boolean toAuraSkills = activeChar.getKnownSkill(7041) != null;
				switchSkills(activeChar, toAuraSkills);
				activeChar.sendSkillList();
				activeChar.sendMessage("You have succefully changed to target " + (toAuraSkills ? "aura" : "one") + " special skills.");
				return true;
			}
			activeChar.sendMessage("There is nothing to switch.");
			return false;
		}
		else
		{
			return true;
		}
	}
	
	/**
	 * @param gmchar the player to switch the Game Master skills.
	 * @param toAuraSkills if {@code true} it will remove "GM Aura" skills and add "GM regular" skills, vice versa if {@code false}.
	 */
	public static void switchSkills(L2PcInstance gmchar, boolean toAuraSkills)
	{
		final Collection<L2Skill> skills = toAuraSkills ? SkillTreesData.getInstance().getGMSkillTree().values() : SkillTreesData.getInstance().getGMAuraSkillTree().values();
		for (L2Skill skill : skills)
		{
			gmchar.removeSkill(skill, false); // Don't Save GM skills to database
		}
		SkillTreesData.getInstance().addSkills(gmchar, toAuraSkills);
	}
	
	@Override
	public String[] getAdminCommandList()
	{
		return ADMIN_COMMANDS;
	}
	
	public static void showBuffs(L2PcInstance activeChar, L2Character target, int page)
	{
		final List<L2Effect> effects = target.getAllEffects();
		
		int max = (effects.size() + PAGE_LIMIT - 1) / PAGE_LIMIT;
		if (max > 0)
			if (page < 1 || page > max)
				return;
		
		final StringBuilder html = StringUtil.startAppend(500 + (effects.size() * 200), "<html>"
			+ "<table width=278><tr>"
			+ "<td width=45><button value=\"Main\" action=\"bypass -h admin_admin\" width=45 height=21 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></td>"
			+ "<td fixwidth=180><center><font color=\"LEVEL\">Effects of ", target.getName(), "</font></td>"
			+ "<td width=45><button value=\"Back\" action=\"bypass -h admin_current_player\" width=45 height=21 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></td>"
			+ "</tr></table><br1>"
			//
			+ "<table width=278 cellspacing=0 cellpadding=0 bgcolor=444444>"
			+ "<tr>"
			+ "<td width=180><font color=CCFF99>Skill</font></td>"
			+ "<td><font color=CCFF99>In use</font></td>"
			+ "</tr>"
			+ "<tr>"
			+ "<td width=180>(Effect)</td>"
			+ "<td>Rem.Time</td>"
			+ "</tr>"
			+ "</table>"
			//
			+ "<table width=278 cellspacing=0 cellpadding=0><tr>"
			);
		int start = (page - 1) * PAGE_LIMIT;
		int end = Math.min(start + PAGE_LIMIT, effects.size());
		for (int i = start; i < end; i++)
		{
			L2Effect e = effects.get(i);
			if (e != null)
			{
				final L2Skill skill = e.getSkill();
				StringUtil.append(html, "<tr>"
					+ "<td VALIGN=TOP width=180><font color=CCFF99>", String.valueOf(skill.getId()), " ", skill.getName(), "</font></td>"	// Skill
					+ "<td VALIGN=TOP><font color=CCFF99>", String.valueOf(e.isInUse()), "</font></td>"	// In use
					+ "<td><button action=\"bypass -h admin_stopbuff ", Integer.toString(target.getObjectId()), " ", String.valueOf(skill.getId()), "\" width=12 height=12 back=L2UI.bbs_delete_Down fore=L2UI.bbs_delete></td>"	// [x]
					+ "</tr>"
					+ "<tr>"
					+ "<td VALIGN=TOP width=180>", "(", e.getClass().getSimpleName(), ")", "</td>"	// Effect
					+ "<td VALIGN=TOP>", skill.isToggle() ? "T (" + e.getTickCount() + ")" : skill.isPassive() ? "P" : e.getTimeLeft() + "s", "</td>"	// Rem.Time
					+ "<td></td>"
					+ "</tr>"
					);
			}
		}
		html.append("</table><br>");
		
		if (max >= 2)
		{
			html.append("<table width=278 cellspacing=0 cellpadding=0 bgcolor=444444><tr>");
			for (int pagenr = 1; pagenr <= max; ++pagenr)
			{
				html.append("<td align=center><a action=\"bypass -h admin_getbuffs \f");
				html.append(target.getObjectId());
				html.append(" ");
				html.append(pagenr);
				html.append("\"> ");
				if (pagenr == page) html.append("<font color=00CCFF>");
				html.append("Page ");
				html.append(pagenr);
				if (pagenr == page) html.append("</font>");
				html.append(" </a></td>");
			}
			html.append("</tr></table>");
		}
		else
		{
			html.append("<br><center><button value=\"Refresh\" action=\"bypass -h admin_getbuffs \f")
			    .append(target.getObjectId())
			    .append(" ")
			    .append(page)
			    .append("\" width=80 height=21 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\">");
		}
		
		StringUtil.append(html, "<br><center><button value=\"Remove All\" action=\"bypass -h admin_stopallbuffs ", Integer.toString(target.getObjectId()), "\" width=80 height=21 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></html>");
		
		activeChar.sendPacket(new NpcHtmlMessage(html.toString()));
		
		if (Config.GMAUDIT)
		{
			GMAudit.auditGMAction(activeChar.getName() + " [" + activeChar.getObjectId() + "]", "getbuffs", target.getName() + " (" + Integer.toString(target.getObjectId()) + ")", "");
		}
	}
	
	private static void removeBuff(L2PcInstance activeChar, int objId, int skillId)
	{
		L2Character target = null;
		try
		{
			target = (L2Character) L2World.getInstance().findObject(objId);
		}
		catch (Exception e)
		{
		}
		
		if ((target != null) && (skillId > 0))
		{
			for (L2Effect e : target.getAllEffects())
			{
				if ((e != null) && (e.getSkill().getId() == skillId))
				{
					e.exit();
					activeChar.sendMessage("Removed " + e.getSkill().getName() + " level " + e.getSkill().getLevel() + " from " + target.getName() + " (" + objId + ")");
				}
			}
			showBuffs(activeChar, target, 1);
			if (Config.GMAUDIT)
			{
				GMAudit.auditGMAction(activeChar.getName() + " [" + activeChar.getObjectId() + "]", "stopbuff", target.getName() + " (" + objId + ")", Integer.toString(skillId));
			}
		}
	}
	
	private static void removeAllBuffs(L2PcInstance activeChar, int objId)
	{
		L2Character target = null;
		try
		{
			target = (L2Character) L2World.getInstance().findObject(objId);
		}
		catch (Exception e)
		{
		}
		
		if (target != null)
		{
			target.stopAllEffects();
			activeChar.sendMessage("Removed all effects from " + target.getName() + " (" + objId + ")");
			showBuffs(activeChar, target, 1);
			if (Config.GMAUDIT)
			{
				GMAudit.auditGMAction(activeChar.getName() + " [" + activeChar.getObjectId() + "]", "stopallbuffs", target.getName() + " (" + objId + ")", "");
			}
		}
	}
}