/*
 * Copyright (C) 2004-2013 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package ai.npc.WyvernManager;

import jp.sf.l2j.arrayMaps.SortedIntObjectArrayMap;
import ai.npc.AbstractNpcAI;

import com.l2jserver.Config;
import com.l2jserver.gameserver.SevenSigns;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.entity.clanhall.SiegableHall;
import com.l2jserver.gameserver.util.Util;

/**
 * Wyvern Manager
 * @author xban1x
 */
public final class WyvernManager extends AbstractNpcAI
{
	private enum ManagerType
	{
		CASTLE,
		CLAN_HALL,
		FORT,
	}
	
	// Misc
	private static final int CRYSTAL_B_GRADE = 1460;
	private static final int WYVERN = 12621;
	private static final int WYVERN_FEE = 25;
	private static final int STRIDER_LVL = 55;
	private static final int[] STRIDERS =
	{
		12526,
		12527,
		12528,
		16038,
		16039,
		16040,
		16068,
		13197 
	};
	// NPCS
	private static final SortedIntObjectArrayMap<ManagerType> MANAGERS = new SortedIntObjectArrayMap<ManagerType>()
		.append(35101, ManagerType.CASTLE)
		.append(35143, ManagerType.CASTLE)
		.append(35185, ManagerType.CASTLE)
		.append(35227, ManagerType.CASTLE)
		.append(35275, ManagerType.CASTLE)
		.append(35317, ManagerType.CASTLE)
		.append(35364, ManagerType.CASTLE)
		.append(35510, ManagerType.CASTLE)
		.append(35536, ManagerType.CASTLE)
		.append(35419, ManagerType.CLAN_HALL)
		.append(35638, ManagerType.CLAN_HALL)
		.append(36457, ManagerType.FORT)
		.append(36458, ManagerType.FORT)
		.append(36459, ManagerType.FORT)
		.append(36460, ManagerType.FORT)
		.append(36461, ManagerType.FORT)
		.append(36462, ManagerType.FORT)
		.append(36463, ManagerType.FORT)
		.append(36464, ManagerType.FORT)
		.append(36465, ManagerType.FORT)
		.append(36466, ManagerType.FORT)
		.append(36467, ManagerType.FORT)
		.append(36468, ManagerType.FORT)
		.append(36469, ManagerType.FORT)
		.append(36470, ManagerType.FORT)
		.append(36471, ManagerType.FORT)
		.append(36472, ManagerType.FORT)
		.append(36473, ManagerType.FORT)
		.append(36474, ManagerType.FORT)
		.append(36475, ManagerType.FORT)
		.append(36476, ManagerType.FORT)
		.append(36477, ManagerType.FORT)
	;
	
	private WyvernManager(String name, String descr)
	{
		super(name, descr);
		addStartNpc(MANAGERS.keySet());
		addTalkId(MANAGERS.keySet());
		addFirstTalkId(MANAGERS.keySet());
	}
	
	private String mountWyvern(L2Npc npc, L2PcInstance player)
	{
		if (player.isMounted() && (player.getMountLevel() >= STRIDER_LVL) && Util.contains(STRIDERS, player.getMountNpcId()))
		{
			if (isOwnerClan(npc, player) && (getQuestItemsCount(player, CRYSTAL_B_GRADE) >= WYVERN_FEE))
			{
				takeItems(player, CRYSTAL_B_GRADE, WYVERN_FEE);
				player.dismount();
				player.mount(WYVERN, 0, true);
				return "wyvernmanager-04.html";
			}
			return replacePart(player.getHtmlPrefix(), "wyvernmanager-06.html");
		}
		return replacePart(player.getHtmlPrefix(), "wyvernmanager-05.html");
	}
	
	private boolean isOwnerClan(L2Npc npc, L2PcInstance player)
	{
		switch (MANAGERS.get(npc.getNpcId()))
		{
			case CASTLE:
			{
				if ((player.getClan() != null) && (npc.getCastle() != null))
				{
					return (player.isClanLeader() && (player.getClanId() == npc.getCastle().getOwnerId()));
				}
				return false;
			}
			case CLAN_HALL:
			{
				if ((player.getClan() != null) && (npc.getConquerableHall() != null))
				{
					return (player.isClanLeader() && (player.getClanId() == npc.getConquerableHall().getOwnerId()));
				}
				return false;
			}
			case FORT:
			{
				if ((player.getClan() != null) && (npc.getFort() != null))
				{
					return (player.isClanLeader() && (player.getClanId() == npc.getFort().getOwnerClan().getClanId()));
				}
				return false;
			}
			default:
			{
				return false;
			}
		}
	}
	
	private boolean isInSiege(L2Npc npc)
	{
		switch (MANAGERS.get(npc.getNpcId()))
		{
			case CASTLE:
			{
				return npc.getCastle().getZone().isActive();
			}
			case CLAN_HALL:
			{
				SiegableHall hall = npc.getConquerableHall();
				return (hall != null) ? hall.isInSiege() : npc.getCastle().getSiege().getIsInProgress();
			}
			case FORT:
			{
				return npc.getFort().getZone().isActive();
			}
			default:
			{
				return false;
			}
		}
	}
	
	private String getResidenceName(L2Npc npc)
	{
		switch (MANAGERS.get(npc.getNpcId()))
		{
			case CASTLE:
			{
				return npc.getCastle().getLocNameHtm();	//[JOJO] getName() --> getLocNameHtm()
			}
			case CLAN_HALL:
			{
				return npc.getConquerableHall().getNameHtm();	//[JOJO] getName() --> getNameHtm()
			}
			case FORT:
			{
				return npc.getFort().getFortNameHtm();	//[JOJO] getName() --> getFortNameHtm()
			}
			default:
			{
				return null;
			}
		}
	}
	
	private String replaceAll(L2Npc npc, String htmlPrefix)
	{
		return replacePart(htmlPrefix, "wyvernmanager-01.html").replace("%residence_name%", getResidenceName(npc));
	}
	
	private String replacePart(String htmlPrefix, String htmlFile)
	{
		return getHtm(htmlPrefix, htmlFile).replace("%wyvern_fee%", String.valueOf(WYVERN_FEE)).replace("%strider_level%", String.valueOf(STRIDER_LVL));
	}
	
	@Override
	public String onAdvEvent(String event, L2Npc npc, L2PcInstance player)
	{
		String htmltext = null;
		switch (event)
		{
			case "Return":
			{
				if (!isOwnerClan(npc, player))
				{
					htmltext = "wyvernmanager-02.html";
				}
				else if (Config.ALLOW_WYVERN_ALWAYS)
				{
					htmltext = replaceAll(npc, player.getHtmlPrefix());
				}
				else if ((MANAGERS.get(npc.getNpcId()) == ManagerType.CASTLE) && SevenSigns.getInstance().isSealValidationPeriod() && (SevenSigns.getInstance().getSealOwner(SevenSigns.SEAL_STRIFE) == SevenSigns.CABAL_DUSK))
				{
					htmltext = "wyvernmanager-dusk.html";
				}
				else
				{
					htmltext = replaceAll(npc, player.getHtmlPrefix());
				}
				break;
			}
			case "Help":
			{
				htmltext = replacePart(player.getHtmlPrefix(), MANAGERS.get(npc.getNpcId()) == ManagerType.CASTLE ? "wyvernmanager-03.html" : "wyvernmanager-03b.html");
				break;
			}
			case "RideWyvern":
			{
				if (!Config.ALLOW_WYVERN_ALWAYS)
				{
					if (!Config.ALLOW_WYVERN_DURING_SIEGE && (isInSiege(npc) || player.isInSiege()))
					{
						player.sendMessage("You cannot summon wyvern while in siege.");
						return null;
					}
					if ((MANAGERS.get(npc.getNpcId()) == ManagerType.CASTLE) && SevenSigns.getInstance().isSealValidationPeriod() && ((SevenSigns.getInstance()).getSealOwner(SevenSigns.SEAL_STRIFE) == SevenSigns.CABAL_DUSK))
					{
						htmltext = "wyvernmanager-dusk.html";
					}
					else
					{
						htmltext = mountWyvern(npc, player);
					}
				}
				else
				{
					htmltext = mountWyvern(npc, player);
				}
				break;
			}
		}
		return htmltext;
	}
	
	@Override
	public String onFirstTalk(L2Npc npc, L2PcInstance player)
	{
		String htmltext = null;
		if (!isOwnerClan(npc, player))
		{
			htmltext = "wyvernmanager-02.html";
		}
		else
		{
			if (Config.ALLOW_WYVERN_ALWAYS)
			{
				htmltext = replaceAll(npc, player.getHtmlPrefix());
			}
			else
			{
				if ((MANAGERS.get(npc.getNpcId()) == ManagerType.CASTLE) && SevenSigns.getInstance().isSealValidationPeriod() && (SevenSigns.getInstance().getSealOwner(SevenSigns.SEAL_STRIFE) == SevenSigns.CABAL_DUSK))
				{
					htmltext = "wyvernmanager-dusk.html";
				}
				else
				{
					htmltext = replaceAll(npc, player.getHtmlPrefix());
				}
			}
		}
		return htmltext;
	}
	
	public static void main(String[] args)
	{
		new WyvernManager(WyvernManager.class.getSimpleName(), "ai/npc");
	}
}
