/**
 *  Impulser2 GUI
 *
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __FreeverbEditor_H
#define __FreeverbEditor_H

#include <aeffguieditor.h>
#include <vstgui.h>
#include "vstguiev.hpp"
#include <vstcontrols.h>
#include <cfileselector.h>
#include "Freeverb.hpp"

#ifndef WIN32
#define _MAX_PATH PATH_MAX
#include <libgen.h>
#endif

// 128x64
class CAHDSRGraph : public CView
{
public:
  CAHDSRGraph(const CRect &size, const CColor &line) ALIGN_ARG_POINTER :
  CView(size)
  {
    setTransparency(false);
    setMouseEnabled(false);
    offset(size.left, size.top);
    linecolor = line;
  }

  virtual ~CAHDSRGraph(){};

  void drawLineXY(float x1, float y1, float x2, float y2)
  {
    y1 = LINCV(0,64,64,0,y1);
    y2 = LINCV(0,64,64,0,y2);
    x1 += offset.x, y1 += offset.y;
    x2 += offset.x, y2 += offset.y;    
    pContext->moveTo(CPoint(x1,y1));
    pContext->lineTo(CPoint(x2,y2));
  }
  
  void updateValue(float va, float vh, float vd, float vs, float vr) ALIGN_ARG_POINTER
  {
    _a = va; _h = vh; _d = vd, _s = vs; _r = vr;
    tr = 128.0f*(1-_r);
    ts = 64.0f*_s;
    ta = tr*_a;
    th = ta+(tr-ta)*_h*(1-_d);
    td = th+(tr-ta)*_d;
    setDirty();
  }

  virtual void draw(CDrawContext *_pContext) ALIGN_ARG_POINTER
  {
    pContext = _pContext;
    pContext->setLineWidth(3);
    pContext->setFrameColor(linecolor);
    // 0,0   -> ta,64
    drawLineXY(0, 0, ta,64);
    // ta,64 -> th,64
    drawLineXY(ta,64,th,64);
    // th,64 -> td,ts
    drawLineXY(th,64,td,ts);
    // td,ts -> tr,ts
    drawLineXY(td,ts,tr,ts);
    // tr,ts -> 128,0
    drawLineXY(tr,ts,128,0);
   
    setDirty(false);
  }

  virtual void doIdleStuff() ALIGN_ARG_POINTER
  {
    if(pParentFrame) pParentFrame->doIdleStuff();
  }

  virtual bool isDirty() const
  {
    if(dirty||CView::isDirty()) return true;
    else return false;
  }

  virtual void setDirty(const bool val = true)
  {
    dirty = val;
  }

private:
  CAHDSRGraph(const CAHDSRGraph&);
  bool dirty;
  float _a, _h, _d, _s, _r;
  float ta, th, td, ts, tr;
  CPoint offset;
  CColor linecolor;
  CDrawContext *pContext;
};

#define kBack          "back.bmp"
#define kOnOff         "load.bmp"
#define kFaderBodyId   "vol.bmp"
#define kFaderHandleId "tag.bmp"
#define kVuOnBitmap    "vuon.bmp"
#define kVuOffBitmap   "vuoff.bmp"
#define kM2SBitmap     "m2s.bmp"
#define kSWAPBitmap    "swap.bmp"
#define kPlainBitmap   "plain.bmp"
#define kUnloadBitmap  "unload.bmp"
#define kPlain2Bitmap  "plain2.bmp"
#define kSelectBitmap  "select.bmp"
#define kSplashBitmap  "splash.bmp"
#define kResetBitmap   "reset.bmp"
#define kMuteWetBitmap "mute.bmp"
#define kKnob1Bitmap   "knob1.bmp"
#define kOnOffBitmap   "sonoff.bmp"

// positions
enum {
  kFaderX = 13,
  kFaderY = 30,
  kFaderInc = 18,
  kDisplayXWidth = 40,
  kDisplayHeight = 15,
  kcGraphWidth = 128,
  kcGraphHeight = 64,
  kpGraphHX = 410,
  kpGraphHY = 330,
  //
  kSlotLabelW = 200,
  kSlotIncY = 8,
};

class FreeverbEditor :
  public AEffGUIEditor, CControlListener
{
public:
  FreeverbEditor(AudioEffect *effect) ALIGN_ARG_POINTER;
  virtual ~FreeverbEditor() ALIGN_ARG_POINTER;
  virtual bool open(void *ptr) ALIGN_ARG_POINTER;
  virtual void close() ALIGN_ARG_POINTER;
  virtual void idle() ALIGN_ARG_POINTER;
  virtual void setParameter(VstInt32 index, float value) ALIGN_ARG_POINTER;
  virtual void valueChanged(CControl * pControl) ALIGN_ARG_POINTER;

  void setFileNameLabelA(const char * filename);
  void setSlotLabelA(int slot, const char * filename);
  void setViewSlot(int slot);
  void setVuMeter(float L, float R);
  void setInfo(const char * str);
  void setGUIZoomFactor(float factor);

  static void stringConvert(float value, char * string);
  static void stringConvert2(float value, char * string);
  static void stringConvertDB(float value, char * string);

#if (__LP64__ & MAC)
  virtual CMessageResult notify (CBaseObject * sender, const char * message);
#endif

private:
  float limitFader(float value);
  void updateGraph();
  Freeverb *freeverb;
  CKickButton *cKickButton, *cUnloadButton, *cResetButton;

  // CFileSelector/CNewFileSelector
#if (__LP64__ & MAC)
  CNewFileSelector *fileSelector;
#else
  CFileSelector *fileSelector;
#endif
  char *returnPath1, *initialPath1;

  // overall
  CVerticalSlider *thrFader, *ceiFader, *relFader, *dryFader;
  CTextEdit *thrTextEdit, *ceiTextEdit, *relTextEdit, *dryTextEdit;
  // slot realtime
  CVerticalSlider *wetFader, *lpfFader, *hpfFader, *idlFader, *widFader;
  CAnimKnob *lrbAKnob;
  CTextEdit *wetTextEdit, *lpfTextEdit, *hpfTextEdit, *idlTextEdit, *widTextEdit, *lrbTextEdit;
  // slot non-realtime
  CVerticalSliderE *strFader;
  CAnimKnobE *irattAKnob, *irholAKnob, *irdecAKnob, *irsusAKnob, *irrelAKnob;
  CTextEdit *strTextEdit, *irattTextEdit, *irholTextEdit, *irdecTextEdit, *irsusTextEdit, *irrelTextEdit;

  COnOffButton *skipLimiterB, *m2sButton, *lrSwapButton, *muteWetButton, *cSlotSelector[MODEL_SLOT_SIZE];
  CTextLabel *cLabel, *cSlotLabel[MODEL_SLOT_SIZE], *cFsInfoLabel, *cInfoLabel;
  float vuL, vuR;
  CVuMeter *cVuMeterL, *cVuMeterR;
  CSplashScreen *cSplashScreen;
  CAHDSRGraph *ccGraph;
  float zoomFactor;
  std::string infoText;
};

#endif
