/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.service;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.entity.SearchQuery.SortField;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.QueryResponseList;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.seasar.framework.util.StringUtil;

public class SearchService implements Serializable {

    private static final long serialVersionUID = 1L;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected QueryHelper queryHelper;

    public List<Map<String, Object>> selectList(String query, int start,
            int rows) {
        return selectList(query, start, rows, true);
    }

    public List<Map<String, Object>> selectList(String query, int start,
            int rows, boolean forUser) {
        long startTime = System.currentTimeMillis();

        SolrServerGroup solrServerGroup = forUser ? solrServerManager
                .getSelectSolrServerGroup() : solrServerManager
                .getUpdateSolrServerGroup();

        QueryResponse queryResponse = null;
        SolrQuery solrQuery = new SolrQuery();
        SearchQuery searchQuery = queryHelper.build(query, forUser);
        String q = searchQuery.getQuery();
        if (StringUtil.isNotBlank(q)) {
            // fields
            solrQuery.setFields(queryHelper.getResponseFields());
            // query
            solrQuery.setQuery(q);
            solrQuery.setStart(start);
            solrQuery.setRows(rows);
            // sort
            for (SortField sortField : searchQuery.getSortFields()) {
                solrQuery.setSortField(sortField.getField(), Constants.DESC
                        .equals(sortField.getOrder()) ? SolrQuery.ORDER.desc
                        : SolrQuery.ORDER.asc);
            }
            // highlighting
            if (queryHelper.getHighlightingFields() != null
                    && queryHelper.getHighlightingFields().length != 0) {
                for (String hf : queryHelper.getHighlightingFields()) {
                    solrQuery.addHighlightField(hf);
                }
                solrQuery.setHighlightSnippets(queryHelper
                        .getHighlightSnippetSize());
            }
            // shards
            if (queryHelper.getShards() != null) {
                solrQuery.setParam("shards", queryHelper.getShards());
            }

            queryResponse = solrServerGroup.query(solrQuery,
                    SolrRequest.METHOD.POST);
        }
        long execTime = System.currentTimeMillis() - startTime;

        QueryResponseList queryResponseList = new QueryResponseList(
                queryResponse, rows);
        queryResponseList.setSearchQuery(q);
        queryResponseList.setSolrQuery(solrQuery.toString());
        queryResponseList.setExecTime(execTime);
        return queryResponseList;
    }

}
