/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.font;

import jp.sourceforge.dvibrowser.dvicore.DviFontSpec;
import jp.sourceforge.dvibrowser.dvicore.DviResolution;
import jp.sourceforge.dvibrowser.dvicore.DviUnit;
import jp.sourceforge.dvibrowser.dvicore.util.Canonicalizer;
import jp.sourceforge.dvibrowser.dvicore.util.SimpleCanonicalizer;

// immutable

public final class LogicalFont
implements java.io.Serializable
{
  private static final long serialVersionUID = -8543173850765210687L;
  private final DviFontSpec fs;
  private final DviUnit dviUnit;
  private final DviResolution res;

  private LogicalFont(DviFontSpec fs, DviUnit dviUnit, DviResolution res) {
    this.fs = fs;
    this.dviUnit = dviUnit;
    this.res = res;
    if (fs == null) throw new IllegalArgumentException("font spec cannot be null");
    if (dviUnit == null) throw new IllegalArgumentException("DVI unit cannot be null");
    if (res == null) throw new IllegalArgumentException("resolution cannot be null");
  }

  private static final Canonicalizer<LogicalFont> canonicalizer
    = new SimpleCanonicalizer<LogicalFont>();

  public static LogicalFont getInstance(
    DviFontSpec fs, DviUnit dviUnit, DviResolution res)
  {
    return canonicalizer.canonicalize(
      new LogicalFont(fs, dviUnit, res)
    );
  }

  public DviFontSpec fontSpec()     { return fs; }
  public DviUnit dviUnit()       { return dviUnit; }
  public DviResolution resolution() { return res; }

  private String string = null;
  public String toString() {
    if (string == null) {
      string = getClass().getName()
             + "[" + fs + "," + dviUnit + "," + res + "]"
             ;
    }
    return string;
  }
  
  public LogicalFont renameTo(String newName)
  {
    return getInstance(fs.rename(newName), dviUnit, res);
  }

  public boolean equals(Object obj) {
    if (obj == this) return true;
    if (obj instanceof LogicalFont) {
      LogicalFont lf = (LogicalFont) obj;
      return fs.equals(lf.fs) &&
             dviUnit.equals(lf.dviUnit) &&
             res.equals(lf.res);
    }
    return false;
  }

  public int hashCode() {
    return fs.hashCode() + 33*(dviUnit.hashCode() + 33*res.hashCode());
  }
}
